
#include "Therm.h"

//----------------------------------------------------------------------
double Therm::hpa2m (double hpa)
{
	const double L = -6.5e-3;
	const double Ra = 287.05;
	const double Pb = 101325;
	const double Tb = 288.15;
	const double gb = 9.807;
	double Ps = hpa*100;
	return (Tb * pow(Pb/Ps, L*Ra/gb) - Tb) / L;
}
//------------------------------------------------------
double Therm::m2hpa (double z)
{
	const double L = -6.5e-3;
	const double Ra = 287.05;
	const double Pb = 101325;
	const double Tb = 288.15;
	const double gb = 9.807;
	double Ps = Pb / pow ( (z*L+Tb)/Tb, gb/L/Ra);
	return Ps/100.0;
}
//------------------------------------------------------
double  Therm::vaporPressure (double tempC)
{
	// August-Roche-Magnus formula
	// http://en.wikipedia.org/wiki/Clausius%E2%80%93Clapeyron_relation
	const double A = 17.625;
	const double B = 243.04;
	const double C = 6.1094;
	return C*exp(A*tempC/(tempC+B));	// hPa
}
//------------------------------------------------------
double  Therm::tempFromVaporPressure (double hpa)
{
	// inverse August-Roche-Magnus formula
	// http://en.wikipedia.org/wiki/Clausius%E2%80%93Clapeyron_relation
	const double A = 17.625;
	const double B = 243.04;
	const double C = 6.1094;
	double esc = log (hpa/C);
	return B*esc/(A-esc);
}
//------------------------------------------------------
double Therm::mixingRatio (double tempC, double hpa)
{
	const double eps = 0.622;
	double psat = vaporPressure (tempC);
	return eps*psat/(hpa-psat);
}
//------------------------------------------------------
double Therm::tempFromMixingRatio (double mixr, double hpa)
{
	const double eps = 0.622;
	double psat = hpa*mixr/(eps+mixr);
	return tempFromVaporPressure (psat);
}
//------------------------------------------------------
double  Therm::latentHeatWater (double tempC)    // J/kg
{
	// http://en.wikipedia.org/wiki/Latent_heat
    const double A = -6.14342e-5;
    const double B = 1.58927e-3;
    const double C = -2.36418;
    const double D = 2500.79;
    return 1000.0 * (A*tempC*tempC*tempC + B*tempC*tempC + C*tempC + D);
}
//------------------------------------------------------
double Therm::gammaSaturatedAdiabatic (double tempC, double hpa)
{
	// return saturated adiabatic lapse rate in Kelvin/m
	const double g = 9.8076;   // m/s2
	const double eps = 0.622;
	const double Cpd = 1005.7;	// J kg−1 K−1;
	const double Cpv = 1952;	// J kg−1 K−1;
	const double Rsd = 287.053;	   // J kg−1 K−1;		
	double Hv = latentHeatWater (tempC);
	double T = tempC + 273.15;
	double r = mixingRatio (tempC, hpa);
 	return g * (1.0+r)*(1.0+Hv*r/(Rsd*T))
 		  / (Cpd + r*Cpv + Hv*Hv*r*(eps+r)/(Rsd*T*T));
}
//------------------------------------------------------
double Therm::saturated_dT_dP (double tempC, double hpa)
{
	const double Rd = 287.053;	   // J kg−1 K−1;
	const double Rv = 461.5; 	   // J kg−1 K−1;
	const double g = 9.8076;   // m/s2
	double gamma = gammaSaturatedAdiabatic (tempC,  hpa);
	double es = vaporPressure (tempC);
	double tempK = tempC + 273.15;
	double rho = (hpa-es)/(Rd*tempK) + es/(Rv*tempK);
	return gamma / (g*rho);
}

//------------------------------------------------------
double Therm::saturatedAdiabaticTemperature (double tempC0, double hpa0, double hpa)
{
	double deltap = 0.1;
	double T = tempC0;
	if (hpa < hpa0) {
		for (double P=hpa0; P >= hpa; P -= deltap) {
			double dtdp = saturated_dT_dP (T, P); 
			T -= dtdp*deltap;
		}
	}
	else  {
		for (double P=hpa0; P <= hpa; P += deltap) {
			double dtdp = saturated_dT_dP (T, P); 
			T += dtdp*deltap;
		}
	}
	return T;
}
//------------------------------------------------------
void Therm::curveSaturatedAdiabatic (TPCurve *curve, double tempC0, double hpa0, double hpaLimit, double step)
{
	double T, P;
	curve->clear ();
	T = tempC0;
	curve->addPoint (tempC0, hpa0);
	if (step < 0) {
		for (P=hpa0+step; P >= hpaLimit+step; P += step) {
			T = Therm::saturatedAdiabaticTemperature (tempC0, hpa0, P);
			curve->addPoint (T, P);
		}
	}
	else if (step > 0) {
		for (P=hpa0+step; P <= hpaLimit+step; P += step) {
			T = Therm::saturatedAdiabaticTemperature (tempC0, hpa0, P);
			curve->addPoint (T, P);
		}
	}
}
//------------------------------------------------------
void Therm::curveSaturatedAdiabatic (TPCurve *curve, TPoint &start, double hpaLimit, double step)
{
	Therm::curveSaturatedAdiabatic (curve, start.tempC, start.hpa, hpaLimit, step);
}
//------------------------------------------------------
double Therm::dryAdiabaticTemperature (double hpa0, double t0, double hpa)
{
	const double R = 8.314472;
	const double Ma = 0.029;
	const double Cpa = 1005;
	return (t0+273.15) * pow (hpa/hpa0 , R/(Ma*Cpa)) - 273.15;
}
//------------------------------------------------------
double Therm::dryAdiabaticPressure (double hpa0, double t0, double tempC)
{
	const double R = 8.314472;
	const double Ma = 0.029;
	const double Cpa = 1005;
	return  hpa0 * pow ((tempC+273.15)/(t0+273.15), (Ma*Cpa)/R);
}


//===================================================================
// Sounding
//===================================================================
Sounding::Sounding ()
{
	clvl_ok = false;
}
//------------------------------------------------------
void Sounding::addSoundingPointC (double hpa, double tempC, double dewpC)
{
	allSounds << SoundingPoint (hpa, tempC, dewpC);
	qSort (allSounds);
}
//------------------------------------------------------
void Sounding::addSoundingPointK (double hpa, double tempK, double dewpK)
{
	addSoundingPointC (hpa, tempK - 273.15, dewpK - 273.15);
}
//------------------------------------------------------
void Sounding::addSoundingPointWind (double hpa, double vx, double vy)
{
	allSoundsWind << SoundingPointWind (hpa, vx, vy);
}
//------------------------------------------------------
double Sounding::hpaMax ()
{
	if (allSounds.size() > 0)
		return allSounds[allSounds.size()-1].hpa;
	else
		return GRIB_NOTDEF;
}
//------------------------------------------------------
double Sounding::hpaMin ()
{
	if (allSounds.size() > 0)
		return allSounds[0].hpa;
	else
		return GRIB_NOTDEF;
}
//------------------------------------------------------
double Sounding::getTempCByAlt (double hpa)
{
	double res = GRIB_NOTDEF;
	bool found = false;
	int i;
	if (allSounds.size() >= 2) {
		for (i=0; i<allSounds.size()-1; i++) {
			if (hpa>=allSounds[i].hpa && hpa<=allSounds[i+1].hpa) {
				found = true;
				break;
			}
		}
		if (found) {
			double v0 = allSounds[i].tempC;
			double v1 = allSounds[i+1].tempC;
			double k = Therm::hpa2m(allSounds[i+1].hpa) - Therm::hpa2m(allSounds[i].hpa);
			if (k != 0)
				k = (Therm::hpa2m(hpa) - Therm::hpa2m(allSounds[i].hpa))/k;
			else
				k = 0;
			res = v0 + k * (v1-v0);
		}
	}
	return res;
}
//------------------------------------------------------
double Sounding::getDewpCByAlt (double hpa)
{
	double res = GRIB_NOTDEF;
	bool found = false;
	int i;
	if (allSounds.size() >= 2) {
		for (i=0; i<allSounds.size()-1; i++) {
			if (hpa>=allSounds[i].hpa && hpa<=allSounds[i+1].hpa) {
				found = true;
				break;
			}
		}
		if (found) {
			double v0 = allSounds[i].dewpC;
			double v1 = allSounds[i+1].dewpC;
			double k = Therm::hpa2m(allSounds[i+1].hpa) - Therm::hpa2m(allSounds[i].hpa);
			if (k != 0)
				k = (Therm::hpa2m(hpa) - Therm::hpa2m(allSounds[i].hpa))/k;
			else
				k = 0;
			res = v0 + k * (v1-v0);
		}
	}
	return res;
}
//------------------------------------------------------
double Sounding::getAvgTempCByAlt (double hpa1, double hpa2) 
{
	Util::orderMinMax (hpa1, hpa2);
	int n = 0;
	double res = 0;
	for (double h=hpa1; h<=hpa2; h+=0.5) {
		double t = getTempCByAlt (h);
		if (t != GRIB_NOTDEF) {
			n ++;
			res += t;
		}
	}
	return (n > 0) ? res/n : GRIB_NOTDEF;
}
//------------------------------------------------------
double Sounding::getAvgDewpCByAlt (double hpa1, double hpa2)
{
	Util::orderMinMax (hpa1, hpa2);
	int n = 0;
	double res = 0;
	for (double h=hpa1; h<=hpa2; h+=0.5) {
		double t = getDewpCByAlt (h);
		if (t != GRIB_NOTDEF) {
			n ++;
			res += t;
		}
	}
	return (n > 0) ? res/n : GRIB_NOTDEF;
}
//------------------------------------------------------
double Sounding::getAltByTempC (double tempC)
{
	double res = GRIB_NOTDEF;
	bool found = false;
	int i;
	if (allSounds.size() >= 2) {
		for (i=0; i<allSounds.size()-1; i++) {
			if (   (tempC>=allSounds[i].tempC && tempC<=allSounds[i+1].tempC)
				|| (tempC<=allSounds[i].tempC && tempC>=allSounds[i+1].tempC)
			) {
				found = true;
				break;
			}
		}
		if (found) {
			double v0 = Therm::hpa2m(allSounds[i].hpa);
			double v1 = Therm::hpa2m(allSounds[i+1].hpa);
			double k = allSounds[i+1].tempC - allSounds[i].tempC;
			if (k != 0)
				k = (tempC - allSounds[i].tempC)/k;
			else
				k = 0;
			res = v0 + k * (v1-v0);
		}
	}
	return res;
}
//------------------------------------------------------
double Sounding::getAltByDewpC (double dewpC)
{
	double res = GRIB_NOTDEF;
	bool found = false;
	int i;
	if (allSounds.size() >= 2) {
		for (i=0; i<allSounds.size()-1; i++) {
			if (   (dewpC>=allSounds[i].dewpC && dewpC<=allSounds[i+1].dewpC)
				|| (dewpC<=allSounds[i].dewpC && dewpC>=allSounds[i+1].dewpC)
			) {
				found = true;
				break;
			}
		}
		if (found) {
			double v0 = Therm::hpa2m(allSounds[i].hpa);
			double v1 = Therm::hpa2m(allSounds[i+1].hpa);
			double k = allSounds[i+1].dewpC - allSounds[i].dewpC;
			if (k != 0)
				k = (dewpC - allSounds[i].dewpC)/k;
			else
				k = 0;
			res = v0 + k * (v1-v0);
		}
	}
	return res;
}
//------------------------------------------------------
TPoint Sounding::get_LCL (double hpa0max, double hpa0min)
{
	if (clvl_ok && clvl_hpa0max==hpa0max && clvl_hpa0min==hpa0min)
		return LCL;
	compute_convective_levels (hpa0max, hpa0min);
	return LCL;
}
//------------------------------------------------------
TPoint Sounding::get_CCL (double hpa0max, double hpa0min)
{
	if (clvl_ok && clvl_hpa0max==hpa0max && clvl_hpa0min==hpa0min)
		return CCL;
	compute_convective_levels (hpa0max, hpa0min);
	return CCL;
}
//------------------------------------------------------
TPoint Sounding::get_LFC (double hpa0max, double hpa0min)
{
	if (clvl_ok && clvl_hpa0max==hpa0max && clvl_hpa0min==hpa0min)
		return LFC;
	compute_convective_levels (hpa0max, hpa0min);
	return LFC;
}
//------------------------------------------------------
TPoint Sounding::get_EL (double hpa0max, double hpa0min)
{
	if (clvl_ok && clvl_hpa0max==hpa0max && clvl_hpa0min==hpa0min)
		return EL;
	compute_convective_levels (hpa0max, hpa0min);
	return EL;
}
//------------------------------------------------------
void Sounding::compute_convective_levels (double hpa0max, double hpa0min)
{
	clvl_ok = true;
	clvl_hpa0max = hpa0max;
	clvl_hpa0min = hpa0min;
	
	double hpa0mean = (hpa0min+hpa0max)/2.0;
	
	LCL = CCL = LFC = EL = TPoint (GRIB_NOTDEF, GRIB_NOTDEF);
	if (allSounds.size() < 2)
		return;
	double deltap = 1.5;
	//-----------------------------------------
	// Compute LCL (follows mixing ratio)
	double p = hpa0mean;
	double temp0 = getAvgTempCByAlt (hpa0min, hpa0max);
	double temp = temp0;
	double dewp = getAvgDewpCByAlt (hpa0min, hpa0max);
	double mixr = Therm::mixingRatio (dewp, hpa0mean);
	double tmixr = dewp;
	//DBG ("temp0 %g   dewp %g   mixr %g ",temp0,dewp,mixr);
	while (tmixr < temp && p>hpaMin()-1)
	{
		p = p - deltap;
		temp = Therm::dryAdiabaticTemperature (hpa0mean, temp0, p);
		tmixr = Therm::tempFromMixingRatio (mixr, p);
	}
	if (tmixr >= temp)
		LCL = TPoint (temp, p);
	if (LCL.hpa==GRIB_NOTDEF)
		return;
	//-----------------------------------------
	// Compute CCL  (continues to follow mixing ratio)
	temp = getTempCByAlt (p);
	while (tmixr < temp && p>hpaMin()-1)
	{
		p = p - deltap;
		temp = getTempCByAlt (p);
		tmixr = Therm::tempFromMixingRatio (mixr, p);
	}
	if (tmixr >= temp)
		CCL = TPoint (temp, p);
	//-----------------------------------------
	// Compute LFC (follows saturated curve from LCL)
	TPCurve curve;
	Therm::curveSaturatedAdiabatic (&curve, LCL, hpaMin()-1, -deltap);
	//DBGN (curve.points.size());
	//-----------------------------------------
	int i;
	for (i=0; i<curve.points.size() && LFC.hpa==GRIB_NOTDEF; i++) 
	{
		TPoint tp = curve.points [i];
		double temp = getTempCByAlt (tp.hpa);
		if (tp.tempC!=GRIB_NOTDEF && temp!=GRIB_NOTDEF && tp.tempC > temp) {
			LFC = tp;
			//DBG ("found LFC : %g", tp.hpa);
		}
	}
	//-----------------------------------------
	// Compute EL (continues to follow saturated curve from LFC)
	for ( ; i<curve.points.size(); i++) 
	{
		TPoint tp = curve.points [i];
		//DBG("hpa=%.2f temp=%.2f T=%.2f", tp.hpa, tp.tempC, getTempCByAlt (tp.hpa));
		double temp = getTempCByAlt (tp.hpa);
		if (tp.tempC!=GRIB_NOTDEF && temp!=GRIB_NOTDEF) {
			if (tp.tempC < temp) {
				if (! EL.ok()) 	
					EL = tp;
			}
			else {
				EL =  TPoint (GRIB_NOTDEF, GRIB_NOTDEF);
			}
			//DBG ("found EL : %g", tp.hpa);
		}
	}
}








