#
# Test PloneFormGen initialisation and set-up
#

import os, sys
if __name__ == '__main__':
    execfile(os.path.join(sys.path[0], 'framework.py'))

from Products.PloneFormGen.tests import pfgtc

from Products.CMFCore.utils import getToolByName

class TestInstallation(pfgtc.PloneFormGenTestCase):
    """Ensure product is properly installed"""

    def afterSetUp(self):
        pfgtc.PloneFormGenTestCase.afterSetUp(self)
    
        self.css        = self.portal.portal_css
        self.kupu       = self.portal.kupu_library_tool
        self.skins      = self.portal.portal_skins
        self.types      = self.portal.portal_types
        self.factory    = self.portal.portal_factory
        self.workflow   = self.portal.portal_workflow
        self.properties = self.portal.portal_properties

        self.fieldTypes = (
            'FormSelectionField',
            'FormMultiSelectionField',
            'FormLabelField',
            'FormDateField',
            'FormLinesField',
            'FormIntegerField',
            'FormBooleanField',
            'FormPasswordField',
            'FormFixedPointField',
            'FormStringField',
            'FormTextField',
            'FormRichTextField',
            'FormFileField',
        )
        self.adapterTypes = (
            'FormSaveDataAdapter',
            'FormMailerAdapter',
        )
        self.thanksTypes = (
            'FormThanksPage',
        )
        self.metaTypes = ('FormFolder',) + self.fieldTypes + self.adapterTypes + self.thanksTypes

    def testSkinLayersInstalled(self):
        self.failUnless('PloneFormGen' in self.skins.objectIds())

    def testTypesInstalled(self):
        for t in self.metaTypes:
            self.failUnless(t in self.types.objectIds())

    def testPortalFactorySetup(self):
        for f in self.metaTypes:
            self.failUnless(f in self.factory.getFactoryTypes())

    def testTypesNotSearched(self):
        types_not_searched = self.properties.site_properties.getProperty('types_not_searched')
        for f in self.fieldTypes + self.adapterTypes + self.thanksTypes:
            self.failUnless(f in types_not_searched)

    def testTypesNotListed(self):
        metaTypesNotToList  = self.properties.navtree_properties.getProperty('metaTypesNotToList')
        for f in self.fieldTypes + self.adapterTypes + self.thanksTypes:
            self.failUnless(f in metaTypesNotToList)

    def testFieldsHaveNoWorkflow(self):
        for f in self.fieldTypes:
            self.assertEqual(self.workflow.getChainForPortalType(f), ())

    def testAdaptersHaveNoWorkflow(self):
        for f in self.adapterTypes:
            self.assertEqual(self.workflow.getChainForPortalType(f), ())

    def testThankspagessHaveNoWorkflow(self):
        for f in self.thanksTypes:
            self.assertEqual(self.workflow.getChainForPortalType(f), ())

    def testKupuResources(self):
        linkable = self.kupu.getPortalTypesForResourceType('linkable')
        self.failUnless('FormFolder' in linkable)

    def test_FormGenTool(self):
        self.failUnless( getToolByName(self.portal, 'formgen_tool')) 
    

class TestContentCreation(pfgtc.PloneFormGenTestCase):
    """Ensure content types can be created and edited"""

    fieldTypes = (
        'FormSelectionField',
        'FormMultiSelectionField',
        'FormLabelField',
        'FormDateField',
        'FormLinesField',
        'FormIntegerField',
        'FormBooleanField',
        'FormPasswordField',
        'FormFixedPointField',
        'FormStringField',
        'FormTextField',
        'FormRichTextField',
        'FormFileField',
    )

    adapterTypes = (
        'FormSaveDataAdapter',
        'FormMailerAdapter',
    )

    thanksTypes = (
        'FormThanksPage',
    )

    sampleContentIds = ('mailer', 'replyto', 'topic', 'comments', 'thank-you')


    def afterSetUp(self):
        pfgtc.PloneFormGenTestCase.afterSetUp(self)
        self.folder.invokeFactory('FormFolder', 'ff1')
        self.ff1 = getattr(self.folder, 'ff1')

    def testCreateFormFolder(self):
        self.failUnless('ff1' in self.folder.objectIds())

    def testSampleContent(self):
        # check embedded content
        oi = self.ff1.objectIds()
        for id in self.sampleContentIds:
            self.failUnless(id in oi)

    def testSampleMailerSetup(self):
        self.assertEqual(self.ff1.actionAdapter, ('mailer',))
        self.assertEqual(self.ff1.mailer.replyto_field, 'replyto')
        self.assertEqual(self.ff1.mailer.subject_field, 'topic')
        self.assertEqual(self.ff1.mailer.thanksPage, 'thank-you')

    def testFormFolderCanSetDefaultPage(self):
        self.assertEqual(self.ff1.canSetDefaultPage(), False)

    def testEditFormFolder(self):
        self.ff1.setTitle('A title')
        self.ff1.setDescription('A description')

        self.assertEqual(self.ff1.Title(), 'A title')
        self.assertEqual(self.ff1.Description(), 'A description')

    def testCreateFields(self):
        for f in self.fieldTypes:
            fname = "%s1" % f
            self.ff1.invokeFactory(f, fname)
            self.failUnless(fname in self.ff1.objectIds())

    def testCreateAdapters(self):
        for f in self.adapterTypes:
            fname = "%s1" % f
            self.ff1.invokeFactory(f, fname)
            self.failUnless(fname in self.ff1.objectIds())
            self.failUnless(hasattr(self.ff1[fname], 'onSuccess'))

    def testCreateThanksPages(self):
        for f in self.thanksTypes:
            fname = "%s1" % f
            self.ff1.invokeFactory(f, fname)
            self.failUnless(fname in self.ff1.objectIds())
            self.failUnless(hasattr(self.ff1[fname], 'displayFields'))

    def testEditField(self):
        for f in self.fieldTypes:
            fname = "%s1" % f
            self.ff1.invokeFactory(f, fname)
            f1 = getattr(self.ff1, fname)
            f1.setTitle('Field title')
            f1.setDescription('Field description')

            self.assertEqual(f1.Title(), 'Field title')
            self.assertEqual(f1.fgField.widget.label, 'Field title')
            self.assertEqual(f1.Description(), 'Field description')
            self.assertEqual(f1.fgField.widget.description, 'Field description')

    def testTALESFieldValidation(self):
        for f in self.fieldTypes:
            if f != 'FormLabelField':
                fname = "%s1" % f
                self.ff1.invokeFactory(f, fname)
                f1 = getattr(self.ff1, fname)
                self.assertEqual(f1.getFgTValidator(), False)
                f1.setFgTValidator('python:True')
                self.assertEqual(f1.getFgTValidator(), True)

    def testEditAdapter(self):
        for f in self.adapterTypes:
            fname = "%s1" % f
            self.ff1.invokeFactory(f, fname)
            f1 = getattr(self.ff1, fname)
            f1.setTitle('title')
            f1.setDescription('description')

            self.assertEqual(f1.Title(), 'title')
            self.assertEqual(f1.Description(), 'description')

    def testMailerZPTBody(self):
        fname = 'FormMailerAdapter1'
        self.ff1.invokeFactory('FormMailerAdapter', fname)
        f1 = getattr(self.ff1, fname)
        self.failUnless( f1.getBody_pt(fields=[]) )        

    def testEditThanksPages(self):
        for f in self.thanksTypes:
            fname = "%s1" % f
            self.ff1.invokeFactory(f, fname)
            f1 = getattr(self.ff1, fname)
            f1.setTitle('title')
            f1.setDescription('description')

            self.assertEqual(f1.Title(), 'title')
            self.assertEqual(f1.Description(), 'description')

    def testCreateFieldsAdaptersOutsideFormFolder(self):
        for f in self.fieldTypes + self.adapterTypes + self.thanksTypes:
            self.assertRaises(ValueError, self.folder.invokeFactory, f, 'f1')

    def testBadIdField(self):
        # test for tracker #32 - Field with id 'language' causes problems with PTS
        
        from Products.CMFCore.exceptions import BadRequest

        fname = 'test_field'
        self.ff1.invokeFactory('FormStringField', fname)
        f1 = getattr(self.ff1, fname)

        self.assertRaises(BadRequest, f1.setId, 'language')

        # also not such a good idea ...
        self.assertRaises(BadRequest, f1.setId, 'form')


class TestTools(pfgtc.PloneFormGenTestCase):
    """ test our tool """

    def test_FormGenTool(self):
        fgt = getToolByName(self.portal, 'formgen_tool')
        self.failUnless(fgt.getdefaultMailTemplateBody())
        fgt.setdefaultMailTemplateBody('something')
        self.failUnlessEqual(fgt.getdefaultMailTemplateBody(), 'something')    


class TestCustomValidators(pfgtc.PloneFormGenTestCase):
    """ test our validators """

    def test_inExNumericRange(self):
        from Products.validation import validation

        v = validation.validatorFor('inExNumericRange')
        self.failUnlessEqual(v(10, minval=1, maxval=20), 1)
        self.failUnlessEqual(v('10', minval=1, maxval=20), 1)
        self.failUnlessEqual(v('1', minval=1, maxval=20), 1)
        self.failUnlessEqual(v('20', minval=1, maxval=20), 1)
        self.failIfEqual(v(0, minval=1, maxval=5), 1)
        self.failIfEqual(v(6, minval=1, maxval=5), 1)
        self.failIfEqual(v(4, minval=5, maxval=3), 1)

    def test_isNotTooLong(self):
        from Products.validation import validation

        v = validation.validatorFor('isNotTooLong')
        self.failUnlessEqual(v('', maxlength=20), 1)
        self.failUnlessEqual(v('1234567890', maxlength=20), 1)
        self.failUnlessEqual(v('1234567890', maxlength=10), 1)
        self.failUnlessEqual(v('1234567890', maxlength=0), 1)
        self.failIfEqual(v('1234567890', maxlength=9), 1)
        self.failIfEqual(v('1234567890', maxlength=1), 1)

    def test_isChecked(self):
        from Products.validation import validation

        v = validation.validatorFor('isChecked')
        self.failUnlessEqual(v('1'), 1)
        self.failIfEqual(v('0'), 1)

    def test_isUnchecked(self):
        from Products.validation import validation

        v = validation.validatorFor('isUnchecked')
        self.failUnlessEqual(v('0'), 1)
        self.failIfEqual(v('1'), 1)

    def test_isNotTooLong(self):
        from Products.validation import validation

        v = validation.validatorFor('isNotTooLong')
        v.maxlength = 10
        self.failUnlessEqual(v('abc'), 1)
        self.failIfEqual(v('abcdefghijklmnopqrstuvwxyz'), 1)

        # there was a bug where widget.maxlength could possibly be defined as
        # '' which means calling int(widget.maxlength) would fail

        class Mock(object):
            pass
        field = Mock()
        field.widget = Mock()
        field.widget.maxlength = ''
        
        self.failUnlessEqual(v('abc', field=field), 1)

class TestGPG(pfgtc.PloneFormGenTestCase):
    """ test ya_gpg.py """
    
    def test_gpg(self):
        from Products.PloneFormGen.content.ya_gpg import gpg, GPGError
        
        if gpg is None:
            print "Skipping GPG tests; gpg binary not found"
        else:
            self.assertRaises(GPGError, gpg.encrypt, 'spam', 'eggs')
           


if  __name__ == '__main__':
    framework()

def test_suite():
    from unittest import TestSuite, makeSuite
    suite = TestSuite()
    suite.addTest(makeSuite(TestInstallation))
    suite.addTest(makeSuite(TestContentCreation))
    suite.addTest(makeSuite(TestCustomValidators))
    suite.addTest(makeSuite(TestTools))
    suite.addTest(makeSuite(TestGPG))
    return suite
