##########################################################################
#                                                                        #
#           copyright (c) 2003 ITB, Humboldt-University Berlin           #
#           written by: Raphael Ritz, r.ritz@biologie.hu-berlin.de       #
#                                                                        #
##########################################################################

"""Article reference main class"""

from types import StringTypes

from AccessControl import ClassSecurityInfo
from Products.CMFCore.permissions \
     import View, ModifyPortalContent
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.utils import getToolByName

from Products.Archetypes.public import registerType
from Products.Archetypes.public import Schema
from Products.Archetypes.public import StringField, BooleanField
from Products.Archetypes.public import StringWidget, BooleanWidget

from Products.CMFBibliographyAT.content.base import BaseEntry
from Products.CMFBibliographyAT.content.schemata \
    import HeaderSchema, AuthorSchema, CoreSchema, TrailingSchema
from Products.CMFBibliographyAT.content.fields \
    import publisherField, addressField, editionField, volumeField, \
           numberField, editor_flagField


AuthorTrailingSchema = Schema((
    editor_flagField,
    ))

SourceSchema = Schema((
    publisherField,
    addressField,
    editionField,
    volumeField,
    numberField,
    StringField('isbn',
        searchable=1,
        widget=StringWidget(label="ISBN",
            label_msgid="label_isbn_number",
            description="ISBN Number.",
            description_msgid="help_isbn_number",
            i18n_domain="plone",),
            ),
     ))

BookSchema = HeaderSchema + AuthorSchema + AuthorTrailingSchema + \
             CoreSchema + SourceSchema + TrailingSchema


class BookReference(BaseEntry):
    """ content type to make reference to a (scientific) article.
    """
    archetype_name = "Book Reference"
    source_fields = ('publisher', 'address', 'edition', 'volume', 'number', 'isbn')

    security = ClassSecurityInfo()

    schema = BookSchema

    actions = BaseEntry.actions + ({
        'id': 'setDetails',
        'name': 'Set Details from Amazon',
        'action': 'string:${object_url}/setDetailsFromISBN',
        'permissions': (CMFCorePermissions.ModifyPortalContent,),
        'condition': 'python:getattr(portal, "amazon_tool", None)'
        },)

    security.declareProtected(View, 'Source')
    def Source(self):
        """ the default book source format
        """
        try:
            return self.BookSource()
        except AttributeError:
            context = self
            publisher = context.getPublisher()
            address   = context.getAddress()
            edition   = context.getEdition()
            isbn      = context.getIsbn()
            
            source = publisher
            if address: source += ', ' + address
            if edition: source += " (%s edition)" % edition
            if isbn: source += "(ISBN: %s)" % isbn
            
            return source + '.'

    security.declareProtected(View, 'ISBN')
    def ISBN(self):
        """ the ISBN number
        """
        return self.getIsbn()

    security.declareProtected(ModifyPortalContent, 'setDetailsFromISBN')
    def setDetailsFromISBN(self, isbn=None, REQUEST=None):
        """ Get details from Amazon using the AmazonTool.
            If asin == None, try to use the asin (isbn) already on the object.
        """
        atool = getToolByName(self, 'amazon_tool')
        if isbn is None:
            isbn = self.getIsbn()
        if not isbn:
            raise Exception('Bad ISBN')
        result = atool.searchByASIN(asin=isbn)[0]
        self.setTitle(result.ProductName)
        # result.Authors can look like:
        #   u'Joseph S. Nye' or
        #   [u'Peter A. Hall', u'David W. Soskice']
        authors = []
        if isinstance(result.Authors, StringTypes):
            result.Authors = [result.Authors]
        for each in result.Authors:
            author = {}
            l = each.split(' ')
            author['firstname'] = l[0]
            if len(l) == 3:
                author['middlename'] = l[1]
                author['lastname'] = l[2]
            else:
                author['middlename'] = ''
                author['lastname'] = l[1]
            authors.append(author)
        self.setAuthors(authors)
        # result.ReleaseDate looks like:
        #   u'01 December, 2001' or
        #   u'May, 2004'
        l = result.ReleaseDate.split(', ')
        year = l[1]
        if len(l) > 2:
            month = l[0].split(' ')[1]
        else:
            month = l[0]
        self.setPublication_year(year)
        self.setPublication_month(month)
        self.setPublisher(result.Manufacturer)

        if REQUEST is not None:
            self.REQUEST.RESPONSE.redirect(self.absolute_url())


registerType(BookReference)
