/*
 * YICS: Connect a FICS interface to the Yahoo! Chess server.
 * Copyright (C) 2004  Chris Howie
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include "types.h"
#include "lists.h"
#include "util.h"
#include "sockets.h"
#include "globals.h"
#include "console.h"
#include "ropcodes.h"
#include "network.h"

static void censorAdd(const char *);
static void censorSub(const char *);

List lists[] = {
	{"censor",	0,	NULL,	censorAdd,	censorSub},
	{"notify",	0,	NULL,	NULL,		NULL},
	{NULL,		0,	NULL}
};

static void genericAddSub(const char *item, char rop) {
	String *packet = StringNew(item, -1);

	lowercase(packet->string);
	packutfString(packet, packet);

	nprintropString(rop, packet);

	StringFree(packet);
}

static void censorAdd(const char *item) {
	genericAddSub(item, (char) ROP_IGNORE);
}

static void censorSub(const char *item) {
	genericAddSub(item, (char) ROP_UNIGNORE);
}

bool listAdd(List *list, const char *item) {
	char **nc;
	char *sd = strdup(item);

	if (inList(list, item))
		return false;

	if (sd == NULL)
		return false;

	nc = realloc(list->contents, sizeof(char *) * (list->size + 2));

	if (nc == NULL) {
		free(sd);
		return false;
	}

	list->contents = nc;
	list->contents[list->size++] = sd;
	list->contents[list->size] = NULL;

	qsort(nc, list->size, sizeof(char *), s_istrcmp);

	return true;
}

bool listSub(List *list, const char *item) {
	int i;
	char **nc;

	if (list->contents == NULL)
		return false;

	for (i = 0; i < list->size; i++)
		if (!istrcmp(list->contents[i], item))
			break;

	if (i == list->size)
		return false;

	free(list->contents[i]);

	if (--list->size == 0) {
		free(list->contents);
		list->contents = NULL;
		return true;
	}

	for (; i < list->size; i++)
		list->contents[i] = list->contents[i + 1];

	nc = realloc(list->contents, sizeof(char *) * list->size);
	if (nc != NULL)
		list->contents = nc;

	return true;
}

bool inList(List *list, const char *item) {
	int i;

	if (list->contents == NULL)
		return false;

	for (i = 0; i < list->size; i++) {
		if (!istrcmp(list->contents[i], item))
			return true;
	}

	return false;
}

void buildList(List *list, const String *packet) {
	unsigned short count;
	String *cp;
	String *item = StringNull();

	memcpy(&count, packet->string, 2);
	count = ntohs(count);

	cp = StringNew(&packet->string[2], packet->length - 2);

	while (count--) {
		unpackutfStringP(item, cp);
		listAdd(list, item->string);
	}

	StringFree(cp);
	StringFree(item);
}

List *findList(const char *name) {
	List *match = NULL;
	List *lp;
	int len = strlen(name);
	bool youlose = false;

	for (lp = lists; lp->name != NULL; lp++) {
		if (!strcmp(name, lp->name))
			return lp;

		if ((len < strlen(lp->name)) && !memcmp(name, lp->name, len)) {
			if (youlose) {
				iprintf(", %s", lp->name);
			} else if (match != NULL) {
				iprintf("Ambiguous list - matches: %s, %s",
						match->name, lp->name);
				youlose = true;
				match = NULL;
			} else {
				match = lp;
			}
		}
	}

	if (match != NULL)
		return match;

	if (youlose)
		iprint(".\n");
	else
		iprintf("\"%s\" does not match any list name.\n", name);

	return NULL;
}
