/*******************************************************************
 * EAPTLS (RFC 2716) Function implementations
 * 
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: eaptls.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eaptls.c,v 1.36 2005/10/17 03:56:54 chessing Exp $
 * $Date: 2005/10/17 03:56:54 $
 * $Log: eaptls.c,v $
 * Revision 1.36  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.35  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.34  2005/08/09 01:39:18  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <string.h>
#include <stdlib.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "eap_types/tls/eaptls.h"
#include "eap_types/tls/tls_funcs.h"
#include "eap.h"

int eaptls_setup(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;
  int retVal;
  struct config_eap_tls *userdata;

  if ((!thisint) || (!thisint->eap_conf_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to eaptls_setup()!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_tls *)thisint->eap_conf_data;

  retVal = XENONE;

  // First, set up the structure to hold all of our instance specific
  // variables.
  thisint->eap_data = (char *)malloc(sizeof(struct tls_vars));
  if (!thisint->eap_data) return XEMALLOC;

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  bzero(mytls_vars, sizeof(struct tls_vars));

  // Set our variables to NULL.
  mytls_vars->ctx = NULL;
  mytls_vars->ssl = NULL;
  mytls_vars->engine = NULL;
  mytls_vars->ssl_in = NULL;
  mytls_vars->ssl_out = NULL;
  mytls_vars->tlsoutdata = NULL;
  mytls_vars->tlsoutsize = 0;
  mytls_vars->tlsoutptr = 0;
  mytls_vars->cncheck = NULL;    // NO
  mytls_vars->cnexact = 0;
  mytls_vars->phase = 0;         // This has no meaning for TLS.
  mytls_vars->resume = userdata->session_resume;
  mytls_vars->resuming = 0;
  mytls_vars->quickResponse = FALSE;
  mytls_vars->cert_loaded = FALSE;
  mytls_vars->verify_mode = SSL_VERIFY_PEER;  // We don't want the option of
                                              // not checking certs here!  It
                                              // would be a *SERIOUSLY* bad
                                              // idea!

  mytls_vars->sessionkeyconst = (char *)malloc(TLS_SESSION_KEY_CONST_SIZE);
  if (mytls_vars->sessionkeyconst == NULL) return XEMALLOC;

  strncpy(mytls_vars->sessionkeyconst, TLS_SESSION_KEY_CONST,
	  TLS_SESSION_KEY_CONST_SIZE);

  mytls_vars->sessionkeylen = TLS_SESSION_KEY_CONST_SIZE;

  debug_printf(DEBUG_EVERYTHING, "(EAP-TLS) Initialized.\n");
  
  if ((retVal = tls_funcs_init(thisint))!=XENONE)
    {
      debug_printf(DEBUG_NORMAL, "Error initializing TLS functions!\n");
      return retVal;
    }
 
  if ((retVal = tls_funcs_load_root_certs(thisint, userdata->root_cert,
					  userdata->root_dir, userdata->crl_dir))!=XENONE)
    {
      debug_printf(DEBUG_NORMAL, "Error loading root certificate!\n");
      return retVal;
    }

  if (userdata->user_key_pass != NULL)
    {
      if ((retVal = tls_funcs_load_user_cert(thisint, userdata->user_cert, 
					     userdata->user_key,
					     userdata->user_key_pass,
					     userdata->random_file))!=XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Error loading user certificate!\n");
	  return retVal;
	} else {

	  // Otherwise, the certificate is loaded.
	  mytls_vars->cert_loaded = TRUE;

	  // We really don't need to free tempPwd here, since TLS won't have
	  // a second phase.  But, we do it anyway, just to keep things
	  // consistant.
	  if (thisint->tempPwd != NULL)
	    {
	      free(thisint->tempPwd);
	      thisint->tempPwd = NULL;
	    }
	}
    }

  if (tls_funcs_load_random(thisint, userdata->random_file) != XENONE)
    {
      debug_printf(DEBUG_NORMAL, "Failed to load random data\n");
      return -1;
    }

  return XENONE;
}

int eaptls_process(struct generic_eap_data *thisint, u_char *dataoffs, 
		   int insize, u_char *outframe, int *outsize)
{
  struct config_eap_tls *userdata;
  struct tls_vars *mytls_vars;
  int retVal;

  if ((!thisint) || (!thisint->eap_conf_data) || (!outframe) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to eaptls_process()!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_tls *)thisint->eap_conf_data;
  mytls_vars = (struct tls_vars *)thisint->eap_data;

  // The state machine wants to know if we have anything else to say.
  // We may be waiting for the server to send us more information, or
  // we may need to send a request to the GUI for a password, and wait
  // for an answer.
  
  if (mytls_vars->cert_loaded == FALSE)
    {
      if ((thisint->tempPwd == NULL) && (userdata->user_key_pass == NULL))
	{
	  thisint->need_password = 1;
	  thisint->eaptype = strdup("EAP-TLS User Certificate");
	  thisint->eapchallenge = NULL;

	  *outsize = 0;

	  return XENONE;
	}

      if ((userdata->user_key_pass == NULL) && (thisint->tempPwd != NULL))
	{
	  userdata->user_key_pass = thisint->tempPwd;
	  thisint->tempPwd = NULL;
	}

      if ((mytls_vars->cert_loaded == FALSE) && ((thisint->tempPwd != NULL) ||
						 (userdata->user_key_pass != NULL)))
      {
	// Load the user certificate.
	if ((retVal = tls_funcs_load_user_cert(thisint, userdata->user_cert, 
					       userdata->user_key,
					       userdata->user_key_pass,
					       userdata->random_file))!=XENONE)
	  {
	    debug_printf(DEBUG_NORMAL, "Error loading user certificate!\n");
	    return retVal;
	  } else {

	    // Otherwise, the certificate is loaded.
	    mytls_vars->cert_loaded = TRUE;
	  }
      }  
    }

  // Make sure we have something to process...
  if (dataoffs == NULL) return XENONE;
  
  retVal=tls_funcs_decode_packet(thisint, (char *) dataoffs, insize, (char *) outframe, outsize, 
				 NULL, userdata->chunk_size);

  return retVal;
}

int eaptls_get_keys(struct interface_data *thisint)
{
  struct config_network *network_data;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed to eaptls_get_keys()!\n");
      return -1;
    }

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration struct! (%s:"
		   "%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (thisint->keyingMaterial != NULL)
    {
      free(thisint->keyingMaterial);
    }

  thisint->keyingMaterial = (u_char *) tls_funcs_gen_keyblock(network_data->activemethod);
  thisint->keyingLength = 32;
  
  if (thisint->keyingMaterial == NULL) return -1;
  return 0;
}

int eaptls_cleanup(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to eaptls_cleanup()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  tls_funcs_cleanup(thisint);

  if (mytls_vars->sessionkeyconst) free(mytls_vars->sessionkeyconst);

  if (mytls_vars) free(mytls_vars);

  debug_printf(DEBUG_EVERYTHING, "(EAP-TLS) Cleaned up.\n");
  return XENONE;
}

int eaptls_failed(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to eaptls_failed()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  tls_funcs_failed(thisint);

  debug_printf(DEBUG_EVERYTHING, "(EAP-TLS) Failed. Resetting.\n");
  return XENONE;
}
