/*******************************************************************
 * EAPOL Function implementations for supplicant
 * 
 * File: eapaka.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapaka.c,v 1.10 2006/03/08 00:16:04 chessing Exp $
 * $Date: 2006/03/08 00:16:04 $
 * $Log: eapaka.c,v $
 * Revision 1.10  2006/03/08 00:16:04  chessing
 * Fixed EAP hints code to work correctly when the request ID packet is padded out with null bytes.  (Observed in Aruba APs.)  Some changes/fixes for the EAP-AKA module.
 *
 * Revision 1.9  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.8  2005/08/14 02:11:09  chessing
 * Fixes for EAP-AKA.  It should now really behave the way it is supposed to. ;)
 *
 * Revision 1.7  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

/*******************************************************************
 *
 * The development of the EAP/AKA support was funded by Internet
 * Foundation Austria (http://www.nic.at/ipa)
 *
 *******************************************************************/


#ifdef EAP_SIM_ENABLE     // Only build this if it has been enabled.

#include <inttypes.h>
#include <stdio.h>
#include <openssl/hmac.h>
#include <openssl/sha.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "winscard.h"
#include "profile.h"
#include "xsupconfig.h"
#include "eap.h"
#include "eapaka.h"
#include "../sim/eapsim.h"
#include "../sim/sm_handler.h"
#include "../sim/fips.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "aka.h"

int eapaka_get_username(struct interface_data *thisint)
{
  char *imsi;  
  char realm[25], card_mode=0;
  char *readers, *username;
  struct config_eap_aka *userdata;
  struct generic_eap_data mydata;
  struct config_network *network_data;
  SCARDCONTEXT ctx;
  SCARDHANDLE hdl;

  if (!thisint) return XEMALLOC;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  userdata = (struct config_eap_aka *)network_data->methods->method_data;

  mydata.eap_conf_data = userdata;

  // Initalize our smartcard context, and get ready to authenticate.
  if (sm_handler_init_ctx(&ctx) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't initialize smart card context!\n");
      return XESIMGENERR;
    }

  readers = sm_handler_get_readers(&ctx);
  if (readers == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't list available readers!\n");
      return XESIMGENERR;
    }

  // Connect to the smart card.
  if (sm_handler_card_connect(&ctx, &hdl, readers) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Error connecting to smart card reader!\n");
      return XESIMGENERR;
    }

  // Wait for up to 10 seconds for the smartcard to become ready.
  if (sm_handler_wait_card_ready(&hdl, 10) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Smart Card wasn't ready after 10 seconds!\n");
      return XESIMGENERR;
    }

  imsi = sm_handler_3g_imsi(&hdl, card_mode, userdata->password);
  if (imsi == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error starting smart card, and getting IMSI!\n");
      return XESIMGENERR;
    }

  debug_printf(DEBUG_AUTHTYPES, "SIM IMSI (AKA) : %s\n",imsi);

  if (network_data->identity != NULL)
    {
      free(network_data->identity);
    }

  network_data->identity = (char *)malloc(50);  // 50 should be plenty!
  if (network_data->identity == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for identity!\n");
      return XEMALLOC;
    }

  username = network_data->identity;
  userdata->username = username;
  bzero(username, 50);

  username[0] = '1';  // An IMSI should always start with a 1.
  strncpy(&username[1], imsi, 18);

  if (userdata->auto_realm == TRUE)
    {
      bzero(&realm, 25);
      sprintf((char *)&realm, "@mnc%c%c%c.mcc%c%c%c.owlan.org",
	      username[4], username[5], username[6], username[1], username[2],
	      username[3]);

      debug_printf(DEBUG_AUTHTYPES, "Realm Portion : %s\n",realm);
      strcat(username, realm);
    }

  // Close the smartcard, so that we know what state we are in.
  sm_handler_close_sc(&hdl, &ctx);

  free(imsi);
  imsi = NULL;

  free(readers);
  readers = NULL;

  debug_printf(DEBUG_AUTHTYPES, "Username is now : %s\n", username);

  return XENONE;
}


int eapaka_setup(struct generic_eap_data *thisint)
{
  struct aka_eaptypedata *mydata;
  struct config_eap_aka *userdata;
  char *imsi;

  debug_printf(DEBUG_AUTHTYPES, "(EAP-AKA) Initalized\n");

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed to eapaka_setup()!\n");
      return XEMALLOC;
    }

  thisint->eap_data = (char *)malloc(sizeof(struct aka_eaptypedata));
  if (thisint->eap_data == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for EAP-AKA specific data structure!\n");
      return XEMALLOC;
    }

  mydata = (struct aka_eaptypedata *)thisint->eap_data;
  userdata = (struct config_eap_aka *)thisint->eap_conf_data;

  mydata->numrands = 0;
  mydata->nonce_mt = NULL;
  mydata->keyingMaterial = NULL;

  thisint->eap_data = (void *)mydata;

#ifndef RADIATOR_TEST
  // Initalize our smartcard context, and get ready to authenticate.
  if (sm_handler_init_ctx(&mydata->scntx) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't initialize smart card context!\n");
      return XESIMGENERR;
    }

  mydata->readers = sm_handler_get_readers(&mydata->scntx);
  if (mydata->readers == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get any available readers!\n");
      return XESIMGENERR;
    }

  // Connect to the smart card.
  if (sm_handler_card_connect(&mydata->scntx, &mydata->shdl, mydata->readers) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Error connecting to smart card reader!\n");
      return XESIMGENERR;
    }

  // Wait for up to 20 seconds for the smartcard to become ready.
  if (sm_handler_wait_card_ready(&mydata->shdl, 20) != 0)
    {
      debug_printf(DEBUG_NORMAL, "Smart Card wasn't ready after 20 seconds!\n");
      return XESIMGENERR;
    }

  imsi = sm_handler_3g_imsi(&mydata->shdl, mydata->card_mode, userdata->password);
  if (imsi == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error starting smart card, and getting IMSI!\n");
      return XESIMGENERR;
    }
#endif

  if (userdata->username == NULL)
    {
      userdata->username = imsi;
    } else {
#ifndef RADIATOR_TEST
      free(imsi);
      imsi = NULL;
#endif
    }

  return XENONE;
}


int eapaka_process(struct generic_eap_data *thisint, u_char *dataoffs,
		   int insize, u_char *out, int *outsize)
{
  int packet_offset, outptr, i, value16, retsize, retval;
  struct typelength *typelen;
  struct typelengthres *typelenres;
  struct aka_eaptypedata *mydata;
  char *username, *framecpy, mac_val[16], mac_calc[20], reslen = 0, reallen;
  unsigned char sres[16];
  struct config_eap_aka *userdata;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed in to eapaka_process()!\n");
      return XEMALLOC;
    }

  mydata = (struct aka_eaptypedata *)thisint->eap_data;
  userdata = (struct config_eap_aka *)thisint->eap_conf_data;

  if (!userdata)
    {
      debug_printf(DEBUG_NORMAL, "Invalid user data struct in eapaka_process()!\n");
      return XEMALLOC;
    }

  if ((thisint->tempPwd == NULL) && (userdata->password == NULL))
    {
      thisint->need_password = 1;
      thisint->eaptype = strdup("EAP-AKA");
      thisint->eapchallenge = NULL;
      
      *outsize = 0;
      return XENONE;
    }

  // Make sure we have something to process...
  if (dataoffs == NULL) return XENONE;

  if (userdata->username == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No username, setting from identity!\n");
      username = thisint->identity;
    } else {
      username = userdata->username;
    }

  if ((userdata->password == NULL) && (thisint->tempPwd != NULL))
    {
      userdata->password = thisint->tempPwd;
      thisint->tempPwd = NULL;
    }

  *outsize = 0;
  outptr = 0;
  bzero(&mac_calc[0], 16);
  bzero(&mac_val[0], 16);

  switch (dataoffs[0])
    {
    case AKA_IDENTITY:
      debug_printf(DEBUG_AUTHTYPES, "Got AKA_IDENTITY!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_AUTHENTICATION_REJECT:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_AUTHENTICATION_REJECT!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_SYNC_FAILURE:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_SYNC_FAILURE!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!  (And, we should *NEVER* get this!\n");
      break;

    case AKA_NOTIFICATION:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_NOTIFICATION!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_REAUTHENTICATION:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_REAUTHENTICATION!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_CLIENT_ERROR:
      debug_printf(DEBUG_AUTHTYPES, "Got an AKA_CLIENT_ERROR!\n");
      debug_printf(DEBUG_AUTHTYPES, "Not implemented!\n");
      break;

    case AKA_CHALLENGE:
      debug_printf(DEBUG_AUTHTYPES, "Got AKA_CHALLENGE!\n");
      packet_offset = 3;

      typelen = (struct typelength *)&out[0];
      bzero(out, 10);
      typelen->type = AKA_CHALLENGE;
      outptr = 3;

      while (packet_offset < insize)
	{
	  switch (dataoffs[packet_offset])
	    {
	    case AT_RAND:
	      retval = aka_do_at_rand(mydata, dataoffs, &packet_offset);
	      if (retval != XENONE) return retval;
	      break;

	    case AT_AUTN:
	      retval = aka_do_at_autn(mydata, dataoffs, &packet_offset);
	      if (retval != XENONE) return retval;
	      break;

	    case AT_IV:
	      debug_printf(DEBUG_AUTHTYPES, "Got an IV (Not supported)\n");
	      aka_skip_not_implemented(dataoffs, &packet_offset);
	      break;

	    case AT_MAC:
	      retval = aka_do_at_mac(thisint, mydata, dataoffs, insize, 
				     &packet_offset, username);
	      if (retval == XEAKASYNCFAIL)
		{
		  printf("Sync failure 2..  Doing sync failure.\n");
		  return aka_do_sync_fail(mydata, out, outsize);
		} else if (retval != XENONE) return retval;
	      break;
	    }
	}

      reslen = mydata->reslen;
      if ((reslen % 4) != 0)
	{
	  reallen = reslen + (reslen % 4);
	} else {
	  reallen = reslen;
	}

      // Build the challenge response.
      typelenres = (struct typelengthres *)&out[outptr];
      typelenres->type = AT_RES;
      typelenres->length = (reallen/4)+1;
      typelenres->reserved = htons(reslen);
      outptr += 4;

      memcpy((char *)&out[outptr], (char *)&mydata->res[0], reslen);
      outptr += reslen;

      if (reallen > reslen)
	{
	  for (i=0;i<(reallen-reslen);i++)
	    {
	      out[outptr+i] = 0x00;
	    }
	  outptr += (reallen-reslen);
	}

      typelenres = (struct typelengthres *)&out[outptr];
      typelenres->type = AT_MAC;
      typelenres->length = 5;
      typelenres->reserved = 0x0000;
      outptr += 4;

      retsize = outptr+16+5;
      
      framecpy = (char *)malloc(retsize);
      if (framecpy == NULL) return XEMALLOC;

      framecpy[0] = 2;   // This is a response.
      framecpy[1] = thisint->eapid;
      value16 = retsize;
      value16 = htons(value16);

      memcpy((char *)&framecpy[2], &value16, 2);
      framecpy[4] = EAP_TYPE_AKA;
      
      memcpy((char *)&framecpy[5], out, retsize);
      debug_printf(DEBUG_AUTHTYPES, "Preframe :\n");
      debug_hex_dump(DEBUG_AUTHTYPES, framecpy, retsize);

      // Zero out the mac.
      bzero((char *)&framecpy[outptr+5], 16);
      debug_printf(DEBUG_AUTHTYPES, "Frame to hash : \n");
      debug_hex_dump(DEBUG_AUTHTYPES, framecpy, retsize);

      HMAC(EVP_sha1(), (char *)&mydata->K_aut[0], 16, framecpy, retsize, (char *)&mac_calc[0], &i);

      free(framecpy);
      framecpy = NULL;

      debug_printf(DEBUG_AUTHTYPES, "MAC = ");
      debug_hex_printf(DEBUG_AUTHTYPES, (char *)&mac_calc[0], 16);

      memcpy((char *)&out[outptr], (char *)&mac_calc[0], 16);

      // Then, calculate the MAC, and return it.
      outptr +=16;
      break;
	  
    default:
      debug_printf(DEBUG_NORMAL, "Unknown SubType value! (%d)\n", 
		   dataoffs[0]);
      break;
    }
  out[2] = 0;
  *outsize = outptr;

  return XENONE;
}

int eapaka_get_keys(struct interface_data *thisint)
{
  struct aka_eaptypedata *mydata;
  struct config_network *network_data;

  if (!thisint) return XEMALLOC;

  network_data = config_get_network_config();

  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  mydata = (struct aka_eaptypedata *)network_data->activemethod->eap_data;
  if (thisint->keyingMaterial != NULL)
    {
      free(thisint->keyingMaterial);
    }

  thisint->keyingMaterial = (char *)malloc(64);
  if (thisint->keyingMaterial == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for keying material! (%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (mydata->keyingMaterial == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Not keying material was stored in EAP-AKA!\n");
      return XEGENERROR;
    }

  memcpy(thisint->keyingMaterial, mydata->keyingMaterial, 64);
  thisint->keyingLength = 32;

  return XENONE;
}

int eapaka_failed(struct generic_eap_data *thisint)
{
  struct config_eap_aka *userdata;

  if ((!thisint) || (!thisint->eap_conf_data))
    {
      debug_printf(DEBUG_AUTHTYPES, "No valid configuration information in EAP-AKA!  Nothing to do!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_aka *)thisint->eap_conf_data;

#ifndef NO_PWD_RESET
  if (userdata->password != NULL)
    {
      free(userdata->password);
      userdata->password = NULL;
    }
#endif

  return XENONE;
}

int eapaka_cleanup(struct generic_eap_data *thisint)
{
  struct aka_eaptypedata *mydata;

  debug_printf(DEBUG_AUTHTYPES, "(EAP-AKA) Cleaning up!\n");
  mydata = (struct aka_eaptypedata *)thisint->eap_data;

#ifndef RADIATOR_TEST
  sm_handler_close_sc(&mydata->shdl, &mydata->scntx);
#endif

  free(mydata);
  mydata = NULL;

  return XENONE;
}

#endif
