/*******************************************************************
 * A generic PCAP/DNET handler for support of various OSes.  (Doesn't
 * provide any kind of wireless support!!!!!!)
 *
 * File: cardif_generic.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_generic.c,v 1.5 2006/01/19 05:37:04 chessing Exp $
 * $Date: 2006/01/19 05:37:04 $
 * $Log: cardif_generic.c,v $
 * Revision 1.5  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.4  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef GENERIC_FRAMER
#include <dnet.h>
#include <pcap.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "cardif/cardif.h"
#include "config.h"
#include "profile.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "cardif/generic/cardif_generic.h"

#ifndef ETH_P_EAPOL
#define ETH_P_EAPOL 0x888e
#endif

const struct pcap_pkthdr *packet_header;
const u_char *packet_ptr;
int more_frames_avail = 0;

pcap_handler packet_catch(u_char *args, const struct pcap_pkthdr *header, 
			  const u_char *packet)
{
  packet_header = header;
  packet_ptr = packet;
}

/***********************************************
 *
 * Return a handle to a pcap descriptor.  If NULL is returned,
 * we have an error.
 *
 ***********************************************/
pcap_t *setup_pcap(char *dev_to_use, char *src_mac, int buf_size, int timeout, 
		   char pcapErr[PCAP_ERRBUF_SIZE])
{
  char pcap_err[PCAP_ERRBUF_SIZE];   // pcap error buffer.
  pcap_t *pcap_descr = NULL;
  bpf_u_int32 pcap_maskp;
  bpf_u_int32 pcap_netp;
  char pcap_filter[100];
  struct bpf_program pcap_fp;
  char *errbuf=NULL;

  pcap_lookupnet(dev_to_use, &pcap_netp, &pcap_maskp, pcap_err);

  pcap_descr = pcap_open_live(dev_to_use, buf_size, 1, timeout, pcap_err);
  if (pcap_descr == NULL)
    {
      debug_printf(DEBUG_NORMAL, "pcap_open_live(): %s\n", pcap_err);
      return NULL;
    }

  sprintf(pcap_filter, "ether dst %s or ether dst 01:80:c2:00:00:03 and ether proto 0x888e", eth_ntoa(src_mac));

  debug_printf(DEBUG_INT, "PCAP Filter : %s\n", pcap_filter);
  
  if (pcap_compile(pcap_descr, &pcap_fp, pcap_filter, 0, pcap_netp) == -1)
    {
      debug_printf(DEBUG_NORMAL, "Error running pcap compile!\n");
      return NULL;
    }

  if (pcap_setfilter(pcap_descr, &pcap_fp) == -1)
    {
      debug_printf(DEBUG_NORMAL, "Error setting filter!\n");
      return NULL;
    }

  if (pcap_setnonblock(pcap_descr, 1, errbuf) == -1)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't put interface in to promiscuous mode!\n");
      debug_printf(DEBUG_NORMAL, "Timers won't work, and problems may result!\n");
    }

  return pcap_descr;
}

/***********************************************
 *
 * Do whatever is needed to get the interface in to a state that we can send
 * and recieve frames on the network.  Any information that we need to later
 * use should be stored in the interface_data structure.
 *
 ***********************************************/
int cardif_init(struct interface_data *thisint, char driver, int block_wpa)
{
  char pcap_err[PCAP_ERRBUF_SIZE], source_mac[6];
  struct gen_sock_data *sockData;

  debug_printf(DEBUG_INT, "Initializing interface %s..\n", thisint->intName);

  if (thisint->intName == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface!\n");
      return -1;
    }

  // For this code, we only handle 1 interface, so the index doesn't matter.
  thisint->intIndex = 0;

  // Allocate memory for the things we need.
  thisint->sockData = (void *)malloc(sizeof(struct gen_sock_data));
  if (thisint->sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error allocating memory!\n");
      return XEMALLOC;
    }

  sockData= thisint->sockData;

  if ((sockData->eth = eth_open(thisint->intName)) == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't open interface %s at line %d in cardif_generic.c!\n",
		   thisint->intName, __LINE__);
      return -1;
    }

  // Get our MAC address.
  if (eth_get(sockData->eth, &source_mac) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get MAC address!\n");
      return -1;
    }

  if ((sockData->pcap_descr = setup_pcap(thisint->intName, 
					 (char *)&source_mac, 
					 1700, 0, 
					 pcap_err)) == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't open interface %s at line %d in cardif_generic.c!\n",
		   thisint->intName, __LINE__);
      return -1;
    }



  // Store a copy of our source MAC for later use.
  memcpy((char *)&thisint->source_mac[0], (char *)&source_mac[0], 6);

  return XENONE;
}

/**************************************************************
 *
 * We don't know how to handle keys, so don't do anything.
 *
 **************************************************************/
void cardif_reset_keys(struct interface_data *thisint)
{
  return;
}

/**************************************************************
 *
 * If we determine that this interface is a wireless interface, then
 * we should call this, to have the destination address changed to the
 * AP that we are talking to.  Otherwise, we will always send frames to
 * the multicast address, instead of the AP.  (And, most APs won't answer
 * to the multicast address.)
 *
 **************************************************************/
int cardif_check_dest(struct interface_data *thisint)
{
  // We probably don't need this either.
  return XENOWIRELESS;
}

/******************************************
 *
 * Return the socket number for functions that need it.
 *
 ******************************************/
int cardif_get_socket(struct interface_data *thisint)
{
  // This has no meaning in the context of this driver!
  return -1;
}

/******************************************
 *
 * Clean up anything that was created during the initialization and operation
 * of the interface.  This will be called before the program terminates.
 *
 ******************************************/
int cardif_deinit(struct interface_data *thisint)
{
  struct gen_sock_data *sockData;

  sockData = thisint->sockData;

  debug_printf(DEBUG_EVERYTHING, "Cleaning up interface %s...\n",thisint->intName);

  // Shutdown libdnet
  if (sockData->eth != NULL)
    {
      sockData->eth = eth_close(sockData->eth);
    }

  if (sockData->pcap_descr != NULL)
    {
      pcap_close(sockData->pcap_descr);
    }

  // Now clean up the memory.
  if (thisint->sockData != NULL)
    {
      free(thisint->sockData);
      thisint->sockData = NULL;
    }

  return XENONE;
}

/******************************************
 *
 * Set a wireless key.  Also, based on the index, we may change the transmit
 * key.
 *
 ******************************************/
int cardif_set_wireless_key(struct interface_data *thisint, u_char *key, 
			    int keylen, int index)
{
  // We won't ever set a key, so return an error.
  return XENOWIRELESS;
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_GetSSID(struct interface_data *thisint, char *ssid_name)
{
  // We don't have any wireless interfaces.
  return XENOWIRELESS;
}

/******************************************
 *
 * Normally set the SSID on the card.  But, cardif_generic doesn't understand
 * keying, so return XENOWIRELESS.
 *
 ******************************************/
int cardif_SetSSID(struct interface_data *thisint, char *ssid_name)
{
  return XENOWIRELESS;
}

/******************************************
 *
 * Check the SSID against what we currently have, and determine if we need
 * to reset our configuration.
 *
 ******************************************/
int cardif_check_ssid(struct interface_data *thisint)
{
  // We aren't wireless!
  return XENOWIRELESS;
}

/******************************************
 *
 * Get the Broadcast SSID (MAC address) of the Access Point we are connected 
 * to.  If this is not a wireless card, or the information is not available,
 * we should return an error.
 *
 ******************************************/
int cardif_GetBSSID(struct interface_data *thisint, char *bssid_dest)
{
  // Not wireless
  return XENOWIRELESS;
}

/******************************************
 *
 * Set the flag in the state machine that indicates if this interface is up
 * or down.  If there isn't an interface, we should return an error.
 *
 ******************************************/
int cardif_get_if_state(struct interface_data *thisint)
{
  // Not sure if there is a good way to do this.
  return TRUE;
}

/******************************************
 *
 * Send a frame out of the network card interface.  If there isn't an 
 * interface, we should return an error.  We should return a different error
 * if we have a problem sending the frame.
 *
 ******************************************/
int cardif_sendframe(struct interface_data *thisint, char *sendframe, 
		     int sendsize)
{
  char nomac[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  struct gen_sock_data *sockData;
  struct config_network *network_data;

  sockData = thisint->sockData;

  if (thisint == NULL) return XEMALLOC;

  if (sendframe == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Cannot send NULL frame!\n");
      return XENOFRAMES;
    }

  if (thisint->send_size == 0) return XENONE;

  network_data = config_get_network_config();

  // The frame we are handed in shouldn't have a src/dest, so put it in.
  memcpy(&sendframe[0], &thisint->dest_mac[0], 6);
  memcpy(&sendframe[6], &thisint->source_mac[0], 6);

  if (thisint->userdata != NULL)
    {
      if (memcmp(nomac, (char *)&network_data->dest_mac[0], 6) != 0)
	{
	  debug_printf(DEBUG_INT, "Static MAC address defined!  Using it!\n");
	  memcpy(&sendframe[0], &thisint->userdata->dest_mac[0], 6);
	}
    }

  debug_printf(DEBUG_EVERYTHING, "Frame to be sent : \n");
  debug_hex_dump(DEBUG_EVERYTHING, sendframe, sendsize);

  if (eth_send(sockData->eth, sendframe, sendsize) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error sending frame!\n");
      return -1;
    }

  return XENONE;  // We didn't get an error.
}

/******************************************
 * 
 * Get a frame from the network.  Since we are in promisc. mode, we will get
 * frames that aren't intended for us.  So, check the frame, determine if it
 * is something we care about, and act accordingly.
 *
 ******************************************/
int cardif_getframe(struct interface_data *thisint)
{
  int pcap_ret_val = 0;
  char dot1x_default_dest[6] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};
  struct gen_sock_data *sockData;
  u_char resultframe[1520];

  sockData = thisint->sockData;

  pcap_ret_val = pcap_dispatch(sockData->pcap_descr, 1, 
			       (pcap_handler)packet_catch, NULL);

  if (pcap_ret_val == 1)
    {
      // We have more frames available.
      memcpy(resultframe, packet_ptr, packet_header->len);
      more_frames_avail = 1;
    } else {
      debug_printf(DEBUG_EVERYTHING, "Didn't get a frame.\n");
      more_frames_avail = 0;
    }
  
  debug_printf(DEBUG_EVERYTHING, "Got Frame : \n");
  debug_hex_dump(DEBUG_EVERYTHING, resultframe, packet_header->len);

  snmp_dot1xSuppEapolFramesRx();

  // Make sure that the frame we got is for us..
  if ((memcmp(&thisint->source_mac[0], &resultframe[0], 6) == 0) ||
      ((memcmp(&resultframe[0], &dot1x_default_dest[0], 6) == 0) &&
       (memcmp(&resultframe[6], &thisint->source_mac[0], 6) != 0)))
    {
      thisint->recv_size = packet_header->len;
      memcpy(thisint->recvframe, resultframe, packet_header->len);
      return packet_header->len;
    }

  // Otherwise it isn't for us. 
  debug_printf(DEBUG_INT, "Got a frame, not for us.\n");
  return XENOFRAMES;
}

/******************************************
 * 
 * Return true if there is a frame in the queue to be processed.
 *
 ******************************************/
int cardif_frameavail(struct interface_data *thisint)
{
  return more_frames_avail;
}

/******************************************
 *
 * Validate an interface, based on if it has a MAC address.
 *
 ******************************************/
int cardif_validate(char *interface)
{
  // Assume that the interface is valid, or the user wouldn't have
  // told us to use it. ;)
  return TRUE;
}

/******************************************
 *
 * Get the name of an interface, based on an index value.
 *
 ******************************************/
int cardif_get_int(int index, char *retInterface)
{
  // Enumerating interfaces isn't available with this driver!
  return XNOMOREINTS;
}


/*******************************************************
 *
 * Check to see if an interface is wireless.  On linux, we look in
 * /proc/net/wireless to see if the interface is registered with the
 * wireless extensions.
 *
 *******************************************************/
int cardif_int_is_wireless(char *interface)
{
  // Not ever going to be wireless!
  return FALSE;
}

/******************************************************
 *
 * Stub for wireless scan.
 *
 *****************************************************/
int cardif_start_wireless_scan(struct interface_data *thisint)
{
  return XENONE;
}

/*****************************************************
 *
 * Stub for encryption capabilities.
 *
 *****************************************************/
void cardif_get_abilities(struct interface_data *thisint)
{
  thisint->enc_capa = 0;
}

#endif
