/*
 * Copyright 2005-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <unistd.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_output.h"
#include "via_mode.h"

#include "via_ch7xxx.h"

/*
 *
 */
enum CH7xxxDevices {
    CH7011
};


/*
 * Output->Private
 */
struct CH7xxxOutputPrivate {
    int Device;

#define CH7XXX_REGSSIZE 0x4A
    CARD8 *Regs; /* I2C Registers */
    int  RegsSize; /* Number of registers */

    /* Options */
    int   Output;
    int   Standard;
};

/*
 *
 */
static void
CH7xxxPrivateDestroy(struct ViaOutput *Output)
{
    struct CH7xxxOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    xfree(Private->Regs);
    xfree(Private);

    Output->PrivateDestroy = NULL;
}

/*
 *
 */
static struct CH7xxxOutputPrivate *
CH7xxxPrivateCreate(struct ViaOutput *Output)
{
    struct CH7xxxOutputPrivate *Private;

    VIAFUNC(Output->scrnIndex);

    Output->PrivSize = sizeof(struct CH7xxxOutputPrivate);
    Output->Private =  xnfcalloc(1, Output->PrivSize);
    Private = Output->Private;

    Private->RegsSize = CH7XXX_REGSSIZE;
    Private->Regs = xnfcalloc(CH7XXX_REGSSIZE, sizeof(CARD32));

    Output->PrivateDestroy = CH7xxxPrivateDestroy;

    return Output->Private;
}


/*
 *
 * Option handling.
 *
 */

/* For Private->Output */
#define TVOUTPUT_NONE       0x00
#define TVOUTPUT_COMPOSITE  0x01
#define TVOUTPUT_SVIDEO     0x02
#define TVOUTPUT_RGB        0x04
#define TVOUTPUT_YCBCR      0x08
#define TVOUTPUT_SC         0x16

enum CH7xxxOpts {
    OPTION_TVOUTPUT,
    OPTION_TVSTANDARD
};

static OptionInfoRec CH7xxxOptions[] =
{
    {OPTION_TVOUTPUT,    "TVOutput",    OPTV_ANYSTR,  {0}, FALSE},
    {OPTION_TVSTANDARD,  "TVStandard",  OPTV_ANYSTR,  {0}, FALSE},
    {-1,                 NULL,          OPTV_NONE,    {0}, FALSE}
};

/*
 *
 */
static OptionInfoPtr
CH7xxxGetOptions(ScrnInfoPtr pScrn, struct CH7xxxOutputPrivate *Private)
{
    OptionInfoPtr  Options;
    char  *s = NULL;

    Options = xnfalloc(sizeof(CH7xxxOptions));
    memcpy(Options, CH7xxxOptions, sizeof(CH7xxxOptions));

    xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, Options);

    /* TV output combination */
    Private->Output = TVOUTPUT_NONE;
    if ((s = xf86GetOptValString(Options, OPTION_TVOUTPUT))) {
        if (!xf86NameCmp(s, "S-Video")) {
            Private->Output = TVOUTPUT_SVIDEO;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is S-Video\n");
        } else if(!xf86NameCmp(s, "Composite")) {
            Private->Output = TVOUTPUT_COMPOSITE;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is Composite\n");
        } else if(!xf86NameCmp(s, "SC")) {
            Private->Output = TVOUTPUT_SC;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is SC\n");
        } else if(!xf86NameCmp(s, "RGB")) {
            Private->Output = TVOUTPUT_RGB;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is RGB\n");
        } else if(!xf86NameCmp(s, "YCbCr")) {
            Private->Output = TVOUTPUT_YCBCR;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Output Signal is YCbCr\n");
        }
    }

    /* TV standard */
    Private->Standard = VIAPTR(pScrn)->Scratch->TVStandard;
    if ((s = xf86GetOptValString(Options, OPTION_TVSTANDARD))) {
        if (!xf86NameCmp(s, "NTSC")) {
            Private->Standard = TVSTANDARD_NTSC;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Standard is NTSC\n");
        } else if(!xf86NameCmp(s, "PAL")) {
            Private->Standard = TVSTANDARD_PAL;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "TV Standard is PAL\n");
        }
    }

    return Options;
}

/*
 *
 */
static void
CH7xxxPrintRegs(struct ViaOutput *Output, const char *function)
{
    CARD8 i, buf;

    ViaDebug(Output->scrnIndex, "%s: Printing registers for %s\n",
             function, Output->I2CDev->DevName);

    for (i = 0; i < 0x4C; i++) {
	xf86I2CReadByte(Output->I2CDev, i, &buf);
	ViaDebug(Output->scrnIndex, "%s %02X: 0x%02X\n",
                 Output->I2CDev->DevName, i, buf);
    }

    ViaDebug(Output->scrnIndex, "End of %s registers.\n",
             Output->I2CDev->DevName);
}

/*
 *
 */
static void
CH7011Save(struct ViaOutput *Output)
{
    struct CH7xxxOutputPrivate *Private = Output->Private;
    int i;

    VIAFUNC(Output->scrnIndex);

    for (i = 0; i < 0x11; i++)
        xf86I2CReadByte(Output->I2CDev, i, &(Private->Regs[i]));

    for (i = 0x1C; i < 0x22; i++)
        xf86I2CReadByte(Output->I2CDev, i, &(Private->Regs[i]));

    i = 0x48;
    xf86I2CReadByte(Output->I2CDev, i, &(Private->Regs[i]));
    i = 0x49;
    xf86I2CReadByte(Output->I2CDev, i, &(Private->Regs[i]));

    /* Fix common restoration issue: When !CIVEN then force CIVC1 = 0 */
    if ((Private->Regs[0x10] & 0x11) == 0x10) {
        xf86DrvMsg(Output->scrnIndex, X_WARNING, "%s: Caught bad restoration "
                   "(CIVC).\n", __func__);
        Private->Regs[0x10] &= 0xEF;
    }
}

/*
 *
 */
static void
CH7011Restore(struct ViaOutput *Output)
{
    struct CH7xxxOutputPrivate *Private = Output->Private;
    int i;

    VIAFUNC(Output->scrnIndex);

    for (i = 0; i < 0x11; i++)
        xf86I2CWriteByte(Output->I2CDev, i, Private->Regs[i]);

    for (i = 0x1C; i < 0x22; i++)
        xf86I2CWriteByte(Output->I2CDev, i, Private->Regs[i]);

    xf86I2CWriteByte(Output->I2CDev, 0x48, Private->Regs[0x48]);
    xf86I2CWriteByte(Output->I2CDev, 0x49, Private->Regs[0x49]);

    usleep(1);
}

/*
 *
 */
static CARD8
CH7xxxTVDACSense(struct ViaOutput *Output)
{
    CARD8 dacsave, save, sense;

    VIAFUNC(Output->scrnIndex);

    /* Enable all DACs */
    xf86I2CReadByte(Output->I2CDev, 0x49, &dacsave);
    xf86I2CWriteByte(Output->I2CDev, 0x49, 0x20);

    /* Disable bypass mode: DACBP = 0 */
    xf86I2CReadByte(Output->I2CDev, 0x21, &save);
    xf86I2CWriteByte(Output->I2CDev, 0x21, save & 0xFE);

    /* SENSE = 1 */
    xf86I2CReadByte(Output->I2CDev, 0x20, &save);
    xf86I2CWriteByte(Output->I2CDev, 0x20, save | 0x01);

    Output->I2CDev->pI2CBus->I2CUDelay(Output->I2CDev->pI2CBus, 10);

    /* Restore SENSE */
    xf86I2CReadByte(Output->I2CDev, 0x20, &save);
    xf86I2CWriteByte(Output->I2CDev, 0x20, save & 0xFE);

    /* Read DAC0-3 */
    xf86I2CReadByte(Output->I2CDev, 0x20, &sense);

    sense >>= 1;
    sense &= 0x0F;

    ViaDebug(Output->scrnIndex, "%s: Sense: 0x%01X\n", __func__, sense);

    xf86I2CWriteByte(Output->I2CDev, 0x49, dacsave);

    return sense;
}

/*
 *
 */
static Bool
CH7011DACSense(struct ViaOutput *Output)
{
    struct CH7xxxOutputPrivate *Private = Output->Private;
    CARD8 sense;

    VIAFUNC(Output->scrnIndex);

    /* Config already set this */
    if (Private->Output != TVOUTPUT_NONE)
        return TRUE;

    sense = CH7xxxTVDACSense(Output);

    switch (sense) {
    case 0x08:
	Private->Output = TVOUTPUT_COMPOSITE;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "%s: Composite connected.\n",
                   Output->I2CDev->DevName);
	return TRUE;
    case 0x06:
	Private->Output = TVOUTPUT_SVIDEO;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "%s: S-Video connected.\n",
                   Output->I2CDev->DevName);
	return TRUE;
    case 0x0E:
	Private->Output = TVOUTPUT_SC;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "%s: Composite+S-Video connected.\n",
                   Output->I2CDev->DevName);
	return TRUE;
    case 0x00:
	Private->Output = TVOUTPUT_NONE;
	xf86DrvMsg(Output->scrnIndex, X_PROBED, "%s: Nothing connected.\n",
                   Output->I2CDev->DevName);
	return FALSE;
    default:
	Private->Output = TVOUTPUT_NONE;
	xf86DrvMsg(Output->scrnIndex, X_WARNING, "%s: Unknown cable combination: 0x0%2X.\n",
		   Output->I2CDev->DevName, sense);
        return FALSE;
    }
}

/*
 * Depends on Output Name, Modes and Private being initialised already.
 */
static void
CH7011TVStandardSet(struct ViaOutput *Output, int Standard)
{
    struct CH7xxxOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    if (Output->MonitorName)
        xfree(Output->MonitorName);

    Private->Standard = Standard;

    ViaModesDestroy(Output->Modes);
    Output->Modes = NULL;

    if (Standard == TVSTANDARD_NTSC) {
        Output->MonitorName = xnfstrdup("TV (NTSC)");

        Output->numHSync = 1;
        Output->HSync[0].lo = 31.40;
        Output->HSync[0].hi = 56.70;

        Output->numVRefresh = 1;
        Output->VRefresh[0].lo = 59.94;
        Output->VRefresh[0].hi = 59.94;

        ViaOutputAddModetable(Output, CH7011TVModesNTSC);
    } else {
        Output->MonitorName = xnfstrdup("TV (PAL)");

        Output->numHSync = 1;
        Output->HSync[0].lo = 25.00;
        Output->HSync[0].hi = 68.60;

        Output->numVRefresh = 1;
        Output->VRefresh[0].lo = 50.00;
        Output->VRefresh[0].hi = 50.00;

        ViaOutputAddModetable(Output, CH7011TVModesPAL);
    }

    Output->ModesExclusive = TRUE;
    Output->ReducedAllowed = FALSE;
}

/*
 *
 */
static ModeStatus
CH7011TVModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct CH7xxxOutputPrivate *Private = Output->Private;
    struct CH7011TableRec *Table = NULL;
    char ID[12] = CH7011ID;
    int i;

    VIAFUNC(Output->scrnIndex);

    if (mode->PrivSize == sizeof(struct CH7011TableRec))
        Table = (struct CH7011TableRec *) mode->Private;

    if (!Table || strncmp(ID, Table->ID, 12)) {
        xf86DrvMsg(Output->scrnIndex, X_INFO, "Not a mode defined by the TV Encoder.\n");
	return MODE_BAD;
    }

    if (Private->Standard != Table->Standard) {
        if (Private->Standard == TVSTANDARD_NTSC)
            xf86DrvMsg(Output->scrnIndex, X_INFO, "TV standard is NTSC. This is a PAL mode.\n");
        else
            xf86DrvMsg(Output->scrnIndex, X_INFO, "TV standard is PAL. This is a NTSC mode.\n");
	return MODE_BAD;
    }

    /* Look for matching clock. */
    for (i = 0; CH7011Clocks[i].Clock; i++)
        if (CH7011Clocks[i].Clock == mode->Clock)
            return MODE_OK;

    xf86DrvMsg(Output->scrnIndex, X_INFO, "%s: Unable to find matching dotclock.\n", __func__);
    return MODE_BAD;
}

static void
CH7011TVClock(I2CDevPtr I2CDev, int clock)
{
    int i;

    /* Look for matching clock. */
    for (i = 0; CH7011Clocks[i].Clock; i++)
        if (CH7011Clocks[i].Clock == clock)
            break;

    if (CH7011Clocks[i].Clock) {
        xf86I2CWriteByte(I2CDev, 0x09, 0x80 | ((CH7011Clocks[i].PLLN >> 5) & 0x18) |
                         ((CH7011Clocks[i].PLLM >> 6) & 0x04) |
                         (CH7011Clocks[i].PLLCap ? 0x01 : 0x00));

        xf86I2CWriteByte(I2CDev, 0x0A, CH7011Clocks[i].PLLM & 0xFF);
        xf86I2CWriteByte(I2CDev, 0x0B, CH7011Clocks[i].PLLN & 0xFF);
    }
}

/*
 * ModeValid should ensure that only modes with proper privates are passed here.
 */
static void
CH7011TVMode(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct CH7011TableRec *Table = NULL;
    CARD8 tmp;
    int  i;

    VIAFUNC(Output->scrnIndex);

    Table = (struct CH7011TableRec *) mode->Private;

    xf86I2CWriteByte(Output->I2CDev, 0x00, Table->Mode);
    xf86I2CWriteByte(Output->I2CDev, 0x01, 0x3F);

    /* low pass filter */
    if (Table->Standard == TVSTANDARD_NTSC)
        xf86I2CWriteByte(Output->I2CDev, 0x02, 0x7E); /* default */
    else
        xf86I2CWriteByte(Output->I2CDev, 0x02, 0xE0); /* default */

    /* Text Enhancement */
    xf86I2CReadByte(Output->I2CDev, 0x03, &tmp);
    xf86I2CWriteByte(Output->I2CDev, 0x03, tmp | 0x07); /* max */

    /* Start Active Video */
    i = mode->CrtcHTotal - mode->CrtcHSyncEnd + 1;
    xf86I2CReadByte(Output->I2CDev, 0x03, &tmp);
    xf86I2CWriteByte(Output->I2CDev, 0x03, (tmp & 0xDF) | ((i >> 3) & 0x20));
    xf86I2CWriteByte(Output->I2CDev, 0x04, i & 0xFF);

    /* Horizontal Position */
    xf86I2CReadByte(Output->I2CDev, 0x03, &tmp);
    xf86I2CWriteByte(Output->I2CDev, 0x03, (tmp & 0xEF) | ((Table->HPosition >> 4) & 0x10));
    xf86I2CWriteByte(Output->I2CDev, 0x05, Table->HPosition & 0xFF);

    /* Vertical Position */
    xf86I2CReadByte(Output->I2CDev, 0x03, &tmp);
    xf86I2CWriteByte(Output->I2CDev, 0x03, (tmp & 0xF7) | ((Table->VPosition >> 5) & 0x08));
    xf86I2CWriteByte(Output->I2CDev, 0x06, Table->VPosition & 0xFF);

    /* Black level */
    if (Table->Standard == TVSTANDARD_NTSC)
        xf86I2CWriteByte(Output->I2CDev, 0x07, 0x83); /* default */
    else
        xf86I2CWriteByte(Output->I2CDev, 0x07, 0x6E); /* default */

    /* contrast */
    xf86I2CWriteByte(Output->I2CDev, 0x08, 0x03); /* default to 3 */


    CH7011TVClock(Output->I2CDev, mode->Clock);

    xf86I2CWriteByte(Output->I2CDev, 0x0C, (Table->FSCI >> 24) & 0xFF);
    xf86I2CWriteByte(Output->I2CDev, 0x0D, (Table->FSCI >> 16) & 0xFF);
    xf86I2CWriteByte(Output->I2CDev, 0x0E, (Table->FSCI >> 8) & 0xFF);
    xf86I2CWriteByte(Output->I2CDev, 0x0F, Table->FSCI & 0xFF);

    xf86I2CWriteByte(Output->I2CDev, 0x10, 0x00);

    /* stop dark band in lower half (undocumented register) */
    xf86I2CReadByte(Output->I2CDev, 0x15, &tmp);
    xf86I2CWriteByte(Output->I2CDev, 0x15, tmp & 0xF8);

    xf86I2CWriteByte(Output->I2CDev, 0x1C, 0x48);
    xf86I2CWriteByte(Output->I2CDev, 0x1D, 0x40);
    xf86I2CWriteByte(Output->I2CDev, 0x1E, 0xF2);
    xf86I2CWriteByte(Output->I2CDev, 0x1F, 0x80);
    xf86I2CWriteByte(Output->I2CDev, 0x20, 0x40);
    xf86I2CWriteByte(Output->I2CDev, 0x21, 0x00);
    xf86I2CWriteByte(Output->I2CDev, 0x22, 0x00);

    xf86I2CWriteByte(Output->I2CDev, 0x48, 0x10);
    xf86I2CWriteByte(Output->I2CDev, 0x48, 0x18);
}

/*
 *
 */
static void
CH7xxxTVPower(struct ViaOutput *Output, Bool On)
{
    CARD8 save;

    VIAFUNC(Output->scrnIndex);

    if (On) {
        /* POUTE */
        xf86I2CReadByte(Output->I2CDev, 0x1E, &save);
        xf86I2CWriteByte(Output->I2CDev, 0x1E, save | 0x02);

	/* Plainly enables all DACs - should be more specific. */
	xf86I2CWriteByte(Output->I2CDev, 0x49, 0x20);
    } else {
	xf86I2CWriteByte(Output->I2CDev, 0x49, 0x3E);

        /* !POUTE */
        xf86I2CReadByte(Output->I2CDev, 0x1E, &save);
        xf86I2CWriteByte(Output->I2CDev, 0x1E, save & ~0x02);
    }
}

/*
 *
 */
struct ViaOutput *
ViaCH7xxxInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    struct ViaOutput *Output;
    struct CH7xxxOutputPrivate *Private;
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x4B, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Unable to read from %s Slave %d.\n",
                   pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return FALSE;
    }

    switch (buf) {
    case 0x17:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected Chrontel CH7011 TV Encoder\n");
        pDev->DevName = "CH7011";

        Output = xnfcalloc(1, sizeof(struct ViaOutput));

        Output->Next = NULL;
        Output->Prev = NULL;

        Output->scrnIndex = pScrn->scrnIndex;
        Output->Name = "CH7011";
        Output->I2CDev = pDev;
        Output->Type = OUTPUT_TV;

        Private = CH7xxxPrivateCreate(Output);

        Output->Options = CH7xxxGetOptions(pScrn, Private);

        Private->Device = CH7011;

        Output->ClockMaster = TRUE;
        CH7011TVStandardSet(Output, Private->Standard);

        Output->Save = CH7011Save;
        Output->Restore = CH7011Restore;
        Output->Sense = CH7011DACSense;
        Output->ModeValid = CH7011TVModeValid;
        Output->Mode = CH7011TVMode;
        Output->Power = CH7xxxTVPower;
        Output->PrintRegs = CH7xxxPrintRegs;

        return Output;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unknown TV Encoder found at %s %X.\n",
                   pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }
}
