/*
# X-BASED THREED
#
#  Threed.c
#
###
#
#  Copyright (c) 1994 - 2007	David Albert Bagley, bagleyd@tux.org
#
#		   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "useful",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Threed */

#include "file.h"
#include "sound.h"
#include "ThreedP.h"

#ifdef WINVER
#ifndef INIFILE
#define INIFILE "wthreed.ini"
#endif
#ifndef DATAFILE
#define DATAFILE "c:\\Windows\threed.dat"
#endif

#define SECTION "setup"
#else

#ifndef DATAFILE
#define DATAFILE "threed.dat"
#endif

static void InitializeThreeD(Widget request, Widget renew);
static void ExposeThreeD(Widget renew, XEvent *event, Region region);
static void ResizeThreeD(ThreeDWidget w);
static void DestroyThreeD(Widget old);
static Boolean SetValuesThreeD(Widget current, Widget request, Widget renew);
static void QuitThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void HideThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void SurfaceThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void ObjectThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void SpeedThreeD(ThreeDWidget w, XEvent *event, char **args, int nArgs);
static void SlowThreeD(ThreeDWidget w, XEvent *event, char **args, int nArgs);
static void SoundThreeD(ThreeDWidget w, XEvent *event, char **args, int nArgs);
static void EnterThreeD(ThreeDWidget w, XEvent *event, char **args,
	int nArgs);
static void LeaveThreeD(ThreeDWidget w, XEvent *event, char **args,
	int nArgs);
static void MoveRaisePsiThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveRaisePhiThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveLowerPhiThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveLowerThetaThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveLowerPsiThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveRaiseThetaThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveRaisePhiThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveLowerThetaLowerPhiThreeD(ThreeDWidget w, XEvent *event,
	char **args, int n_args);
static void MoveLowerThetaRaisePhiThreeD(ThreeDWidget w, XEvent *event,
	char **args, int n_args);
static void MoveRaiseThetaLowerPhiThreeD(ThreeDWidget w, XEvent *event,
	char **args, int n_args);
static void MoveRaiseThetaRaisePhiThreeD(ThreeDWidget w, XEvent *event,
	char **args, int n_args);
static void MoveLeftThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveRightThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveUpThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveDownThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveOutThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void MoveInThreeD(ThreeDWidget w, XEvent *event, char **args,
	int n_args);
static void SelectThreeD(ThreeDWidget w, XEvent *event, char **args,
	int nArgs);
static void MotionThreeD(ThreeDWidget w, XEvent *event, char **args,
	int nArgs);
static void ReleaseThreeD(ThreeDWidget w, XEvent *event, char **args,
	int nArgs);

static char defaultTranslationsThreeD[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>KP_Divide: MoveRaisePsi()\n\
 <KeyPress>R5: MoveRaisePsi()\n\
 <KeyPress>Home: MoveRaiseThetaRaisePhi()\n\
 <KeyPress>KP_7: MoveRaiseThetaRaisePhi()\n\
 <KeyPress>R7: MoveRaiseThetaRaisePhi()\n\
 <KeyPress>Up: MoveRaisePhi()\n\
 <KeyPress>osfUp: MoveRaisePhi()\n\
 <KeyPress>KP_Up: MoveRaisePhi()\n\
 <KeyPress>KP_8: MoveRaisePhi()\n\
 <KeyPress>R8: MoveRaisePhi()\n\
 <Btn4Down>: MoveRaisePhi()\n\
 <KeyPress>Prior: MoveLowerThetaRaisePhi()\n\
 <KeyPress>KP_9: MoveLowerThetaRaisePhi()\n\
 <KeyPress>R9: MoveLowerThetaRaisePhi()\n\
 <KeyPress>Left: MoveRaiseTheta()\n\
 <KeyPress>osfLeft: MoveRaiseTheta()\n\
 <KeyPress>KP_Left: MoveRaiseTheta()\n\
 <KeyPress>KP_4: MoveRaiseTheta()\n\
 <KeyPress>R10: MoveRaiseTheta()\n\
 <KeyPress>Begin: MoveLowerPsi()\n\
 <KeyPress>KP_5: MoveLowerPsi()\n\
 <KeyPress>R11: MoveLowerPsi()\n\
 <KeyPress>Right: MoveLowerTheta()\n\
 <KeyPress>osfRight: MoveLowerTheta()\n\
 <KeyPress>KP_Right: MoveLowerTheta()\n\
 <KeyPress>KP_6: MoveLowerTheta()\n\
 <KeyPress>R12: MoveLowerTheta()\n\
 <KeyPress>End: MoveRaiseThetaLowerPhi()\n\
 <KeyPress>KP_1: MoveRaiseThetaLowerPhi()\n\
 <KeyPress>R13: MoveRaiseThetaLowerPhi()\n\
 <KeyPress>Down: MoveLowerPhi()\n\
 <KeyPress>osfDown: MoveLowerPhi()\n\
 <KeyPress>KP_Down: MoveLowerPhi()\n\
 <KeyPress>KP_2: MoveLowerPhi()\n\
 <KeyPress>R14: MoveLowerPhi()\n\
 <Btn5Down>: MoveLowerPhi()\n\
 <KeyPress>Next: MoveLowerThetaLowerPhi()\n\
 <KeyPress>KP_3: MoveLowerThetaLowerPhi()\n\
 <KeyPress>R15: MoveLowerThetaLowerPhi()\n\
 <KeyPress>l: MoveLeft()\n\
 <KeyPress>r: MoveRight()\n\
 <KeyPress>u: MoveUp()\n\
 <KeyPress>d: MoveDown()\n\
 <KeyPress>o: MoveOut()\n\
 <KeyPress>i: MoveIn()\n\
 <Btn1Down>: Select()\n\
 <Btn1Motion>: Motion()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>s: Surface()\n\
 <KeyPress>b: Object()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListThreeD[] = {
	{(char *) "Quit", (XtActionProc) QuitThreeD},
	{(char *) "Hide", (XtActionProc) HideThreeD},
	{(char *) "MoveRaisePsi", (XtActionProc) MoveRaisePsiThreeD},
	{(char *) "MoveLowerPhi", (XtActionProc) MoveLowerPhiThreeD},
	{(char *) "MoveLowerTheta", (XtActionProc) MoveLowerThetaThreeD},
	{(char *) "MoveLowerPsi", (XtActionProc) MoveLowerPsiThreeD},
	{(char *) "MoveRaiseTheta", (XtActionProc) MoveRaiseThetaThreeD},
	{(char *) "MoveRaisePhi", (XtActionProc) MoveRaisePhiThreeD},
	{(char *) "MoveLowerThetaLowerPhi",
		(XtActionProc) MoveLowerThetaLowerPhiThreeD},
	{(char *) "MoveLowerThetaRaisePhi",
		(XtActionProc) MoveLowerThetaRaisePhiThreeD},
	{(char *) "MoveRaiseThetaLowerPhi",
		(XtActionProc) MoveRaiseThetaLowerPhiThreeD},
	{(char *) "MoveRaiseThetaRaisePhi",
		(XtActionProc) MoveRaiseThetaRaisePhiThreeD},
	{(char *) "MoveLeft", (XtActionProc) MoveLeftThreeD},
	{(char *) "MoveRight", (XtActionProc) MoveRightThreeD},
	{(char *) "MoveUp", (XtActionProc) MoveUpThreeD},
	{(char *) "MoveDown", (XtActionProc) MoveDownThreeD},
	{(char *) "MoveOut", (XtActionProc) MoveOutThreeD},
	{(char *) "MoveIn", (XtActionProc) MoveInThreeD},
	{(char *) "Select", (XtActionProc) SelectThreeD},
	{(char *) "Motion", (XtActionProc) MotionThreeD},
	{(char *) "Release", (XtActionProc) ReleaseThreeD},
	{(char *) "Surface", (XtActionProc) SurfaceThreeD},
	{(char *) "Object", (XtActionProc) ObjectThreeD},
	{(char *) "Speed", (XtActionProc) SpeedThreeD},
	{(char *) "Slow", (XtActionProc) SlowThreeD},
	{(char *) "Sound", (XtActionProc) SoundThreeD},
	{(char *) "Enter", (XtActionProc) EnterThreeD},
	{(char *) "Leave", (XtActionProc) LeaveThreeD}
};

static XtResource resourcesThreeD[] = {
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(ThreeDWidget, core.width),
	 XtRString, (caddr_t) "400"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(ThreeDWidget, core.height),
	 XtRString, (caddr_t) "200"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.frameColor),
	 XtRString, (caddr_t) "cyan" /* XtDefaultForeground*/},
	{XtNwhiteBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[WHITE_BRUSH]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNltgrayBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[LTGRAY_BRUSH]),
	 XtRString, (caddr_t) "Green"},
	{XtNgrayBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[GRAY_BRUSH]),
	 XtRString, (caddr_t) "Red"},
	{XtNdkgrayBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[DKGRAY_BRUSH]),
	 XtRString, (caddr_t) "Blue"},
	{XtNblackBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[BLACK_BRUSH]),
	 XtRString, (caddr_t) "Black"},
	{XtNanotherBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[ANOTHER_BRUSH]),
	 XtRString, (caddr_t) "Brown"},
	{XtNborderColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.borderColor),
	 XtRString, (caddr_t) "gray75" /* XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.delay),
	 XtRString, (caddr_t) "50"}, /* DEFAULTDELAY */
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNbumpSound, XtCBumpSound, XtRString, sizeof (String),
	 XtOffset(ThreeDWidget, threed.bumpSound),
	 XtRString, (caddr_t) BUMPSOUND},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(ThreeDWidget, threed.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNxDistance, XtCXDistance, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.distance.x),
	 XtRString, (caddr_t) "0"},
	{XtNyDistance, XtCYDistance, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.distance.y),
	 XtRString, (caddr_t) "0"},
	{XtNzDistance, XtCZDistance, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.distance.z),
	 XtRString, (caddr_t) "50"},
	{XtNthetaDegrees, XtCThetaDegrees, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.angle.theta),
	 XtRString, (caddr_t) "0"},
	{XtNphiDegrees, XtCPhiDegrees, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.angle.phi),
	 XtRString, (caddr_t) "0"},
	{XtNpsiDegrees, XtCPsiDegrees, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.angle.psi),
	 XtRString, (caddr_t) "0"},
	{XtNsurface, XtCSurface, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.surface),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_SURFACE */
	{XtNobject, XtCObject, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.object),
	 XtRString, (caddr_t) "0"},
	{XtNobjectName, XtCObjectName, XtRString, sizeof (String),
	 XtOffset(ThreeDWidget, threed.name),
	 XtRString, (caddr_t) "None"},
	{XtNobjectNumber, XtCObjectNumber, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.numObjects),
	 XtRString, (caddr_t) "0"},
	{XtNobjectList, XtCObjectList, XtRString, sizeof (String),
	 XtOffset(ThreeDWidget, threed.list),
	 XtRString, (caddr_t) "None"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(ThreeDWidget, threed.select), XtRCallback, NULL}
};

ThreeDClassRec threedClassRec = {
	{
	 (WidgetClass) & widgetClassRec,	/* superclass */
	 (char *) "ThreeD",	/* class name */
	 sizeof (ThreeDRec),	/* widget size */
	 NULL,		/* class initialize */
	 NULL,		/* class part initialize */
	 FALSE,		/* class inited */
	 (XtInitProc) InitializeThreeD,	/* initialize */
	 NULL,		/* initialize hook */
	 XtInheritRealize,	/* realize */
	 actionsListThreeD,	/* actions */
	 XtNumber(actionsListThreeD),	/* num actions */
	 resourcesThreeD,	/* resources */
	 XtNumber(resourcesThreeD),	/* num resources */
	 NULLQUARK,	/* xrm class */
	 TRUE,		/* compress motion */
	 TRUE,		/* compress exposure */
	 TRUE,		/* compress enterleave */
	 TRUE,		/* visible interest */
	 (XtWidgetProc) DestroyThreeD,	/* destroy */
	 (XtWidgetProc) ResizeThreeD,	/* resize */
	 (XtExposeProc) ExposeThreeD,	/* expose */
	 (XtSetValuesFunc) SetValuesThreeD,	/* set values */
	 NULL,		/* set values hook */
	 XtInheritSetValuesAlmost,	/* set values almost */
	 NULL,		/* get values hook */
	 NULL,		/* accept focus */
	 XtVersion,	/* version */
	 NULL,		/* callback private */
	 defaultTranslationsThreeD,	/* tm table */
	 NULL,		/* query geometry */
	 NULL,		/* display accelerator */
	 NULL		/* extension */
	 }
	,
	{
	 0		/* ignore */
	 }
};

WidgetClass threedWidgetClass = (WidgetClass) & threedClassRec;

static void
setThreed(ThreeDWidget w, int reason)
{
	threedCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}

#endif

#define RADIANS(x) (M_PI*(x)/ST_ANGLE)
#define DEGREES(x) ((x)/M_PI*ST_ANGLE)

#ifdef DEPTHINDEX
static Object3D *OBJ;
#endif

static void
CheckGraphicsThreeD(ThreeDWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->threed.distance.x < MINDISTANCE ||
			w->threed.distance.x > MAXDISTANCE) {
		w->threed.distance.x = (MAXDISTANCE + MINDISTANCE) / 2;
		intCat(&buf1,
			"Distance out of bounds, use ",
			MINDEPTH);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXDEPTH);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->threed.distance.x);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	if (w->threed.distance.y < MINDISTANCE ||
			w->threed.distance.y > MAXDISTANCE) {
		w->threed.distance.y = (MAXDISTANCE + MINDISTANCE) / 2;
		intCat(&buf1,
			"Distance out of bounds, use ",
			MINDEPTH);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXDEPTH);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->threed.distance.y);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	if (w->threed.distance.z < MINDEPTH ||
			w->threed.distance.z > MAXDEPTH) {
		w->threed.distance.z = (MAXDEPTH + MINDEPTH) / 2;
		intCat(&buf1,
			"Distance out of bounds, use ",
			MINDEPTH);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXDEPTH);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->threed.distance.z);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	if (w->threed.angle.theta < MINDEGREES ||
			w->threed.angle.theta >= NUM_DEGREES) {
		w->threed.angle.theta = (NUM_DEGREES +
			w->threed.angle.theta) % NUM_DEGREES;
	}
	if (w->threed.angle.phi < MINDEGREES ||
			w->threed.angle.phi >= NUM_DEGREES) {
		w->threed.angle.phi = (NUM_DEGREES +
			w->threed.angle.phi) % NUM_DEGREES;
	}
	if (w->threed.angle.psi < MINDEGREES ||
			w->threed.angle.psi >= NUM_DEGREES) {
		w->threed.angle.psi = (NUM_DEGREES +
			w->threed.angle.psi) % NUM_DEGREES;
	}
	if (w->threed.numObjects == 0) {
		DISPLAY_WARNING("No objects");
	} else if (w->threed.object < 0 ||
			w->threed.object >= w->threed.numObjects) {
		w->threed.object = 0;
		intCat(&buf1, "Object type number out of bounds, use 0..",
			w->threed.numObjects - 1);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->threed.object);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}

}

#ifdef __cplusplus
extern "C" {
#endif
static int
#ifdef WINVER
__cdecl
#endif
CompareObject(const void *elem1, const void *elem2)
#ifdef DEPTHINDEX
{
	double z1, z2;

	z1 = OBJ->vertex[((Surface *) elem1)->depthIndex].eye.z;
	z2 = OBJ->vertex[((Surface *) elem2)->depthIndex].eye.z;
	if (z1 > z2)
		return -1;
	if (z2 > z1)
		return 1;
	return 0;
}
#else
{
	Surface *surface1, *surface2;

	surface1 = (Surface *) elem1;
	surface2 = (Surface *) elem2;
	if (surface1->averageDepth > surface2->averageDepth)
		return -1;
	if (surface2->averageDepth > surface1->averageDepth)
		return 1;
	return 0;
}
#endif
#ifdef __cplusplus
}
#endif

#define EPSILON 0.000001
#define ASPECTRATIO 1.0
#define viewDistance ((w->core.width + w->core.height) / 2)

static void
RealizeObject(ThreeDWidget w, Object3D *obj)
{
	int vert, surf;
	double z;

	if (w->threed.numObjects == 0)
		return;
	rotateObject(obj, &(w->threed.deltaAngle));
	w->threed.deltaAngle.theta = 0;
	w->threed.deltaAngle.phi = 0;
	w->threed.deltaAngle.psi = 0;
	for (vert = 0; vert < obj->numVertices; vert++) {
		Vertex *vertex = &(obj->vertex[vert]);

 		vertex->eye.x = obj->local[vert].x +
			w->threed.distance.x - obj->origin.x;
		vertex->eye.y = obj->local[vert].y +
			w->threed.distance.y - obj->origin.y;
		vertex->eye.z = obj->local[vert].z +
			w->threed.distance.z - obj->origin.z;
		/* Avoid divide by zero */
		if (vertex->eye.z < EPSILON && vertex->eye.z >= 0.0)
			vertex->eye.z = EPSILON;
		if (vertex->eye.z < 0.0 && vertex->eye.z > -EPSILON)
			vertex->eye.z = -EPSILON;
		vertex->screen.x = (int) (w->threed.center.x +
			vertex->eye.x *
			viewDistance / vertex->eye.z);
		vertex->screen.y = (int) (w->threed.center.y -
			ASPECTRATIO * vertex->eye.y *
			viewDistance / vertex->eye.z);
#ifdef DEBUG
		(void) printf("%d: x %g, y %g, z %g\n",
			vert, vertex->eye.x, vertex->eye.y, vertex->eye.z);
#endif
	}
#ifndef DEPTHINDEX
	for (surf = 0; surf < obj->numSurfaces; surf++) {
		Surface *surface = &(obj->surface[surf]);
		int mapIndex = surface->mapIndex;

		z = 0.0;
		for (vert = 0; vert < surface->numVertices;
				vert++, mapIndex++) {
			z += obj->vertex[obj->map[mapIndex]].eye.z;
		}
		surface->averageDepth = z / ((double) surface->numVertices);
	}
#endif
}

static void
DrawObject(ThreeDWidget w, Object3D *obj)
{
	int surf, vert, vertex, mapIndex, clickSurface = -1;
	Point points[MAXPOLYGONVERTICES + 1];
	Pixmap *dr;
	double s1 = 0.0, s2 = 0.0, s3 = 0.0;
	Point3D *v1, *v2, *v3;
	GC lineGC, fillGC;

	dr = &(w->threed.bufferObjects[0]);
	/* compute position of object in world */
	/* if (w->threed.surface) */ {
#ifdef DEPTHINDEX
 		OBJ = obj;
#endif
		(void) qsort((void *) obj->surface,
			(unsigned int) obj->numSurfaces,
			sizeof(Surface), CompareObject);
	}
	for (surf = 0; surf < obj->numSurfaces; surf++) {
		mapIndex = obj->surface[surf].mapIndex;

		if (obj->surface[surf].clipped)
			continue;
		/* (void) printf("polygon #%d\n", surf); */
		v1 = &obj->vertex[obj->map[mapIndex]].eye;
		v2 = &obj->vertex[obj->map[mapIndex + 1]].eye;
		v3 = &obj->vertex[obj->map[mapIndex + 2]].eye;
		s1 = -v1->x * (v2->y * v3->z - v3->y * v2->z);
		s2 = -v2->x * (v3->y * v1->z - v1->y * v3->z);
		s3 = -v3->x * (v1->y * v2->z - v2->y * v1->z);
		if (w->threed.surface && -s1 - s2 - s3 > 0.0) {
			obj->surface[surf].visible = -1;
			continue;
		} else if (-s1 - s2 - s3 == 0.0) {
			obj->surface[surf].visible = 0;
		} else {
			obj->surface[surf].visible = 1;
		}
		for (vert = 0; vert < obj->surface[surf].numVertices;
				vert++, mapIndex++) {
			vertex = obj->map[mapIndex];
			if (vert < MAXPOLYGONVERTICES) {
				points[vert].x = obj->vertex[vertex].screen.x;
				points[vert].y = obj->vertex[vertex].screen.y;
			} else {
				DISPLAY_WARNING("Ignoring extra polygon points");
				break;
			}
		}
		if (pointInPolygon(&(w->threed.currentPosition),
				&(points[0]), vert)) {
			clickSurface = surf;
		}
	}
	for (surf = 0; surf < obj->numSurfaces; surf++) {
		mapIndex = obj->surface[surf].mapIndex;
		if (obj->surface[surf].visible == -1 ||
				obj->surface[surf].clipped)
			continue;
		for (vert = 0; vert < obj->surface[surf].numVertices;
				vert++, mapIndex++) {
			vertex = obj->map[mapIndex];
			if (vert < MAXPOLYGONVERTICES) {
				/* Its not dynamic but we really after
				   speed here */
				points[vert].x = obj->vertex[vertex].screen.x;
				points[vert].y = obj->vertex[vertex].screen.y;
			} else {
				break;
			}
		}
		points[vert].x = points[0].x;
		points[vert].y = points[0].y;
		if (surf == clickSurface) {
			fillGC = w->threed.frameGC;
		} else {
			fillGC = w->threed.shadeGC[obj->surface[surf].color];
		}
		lineGC = w->threed.shadeGC[obj->surface[surf].color];
		if (!w->threed.surface || obj->surface[surf].visible == 0) {
			POLYLINE(w, *dr, fillGC,
				&points[0], vert + 1, True);
		} else {
			POLYGON(w, *dr, fillGC, lineGC,
				&points[0], vert, obj->convex, True);
		}
	}
#ifdef USE_SOUND
	if (w->threed.sound) {
		static int oldSurface = -1;

		if (clickSurface != oldSurface && clickSurface >= 0) {
			playSound((char *) BUMPSOUND);
			oldSurface = clickSurface;
		} else if (clickSurface < 0) {
			oldSurface = -1;
		}
	}
#endif

}

static void
CopyFromBuffer(ThreeDWidget w)
{
#ifdef WINVER
	w->core.hOldBitmap = (HBITMAP) SelectObject(w->core.memDC,
		w->threed.bufferObjects[0]);
	BitBlt(w->core.hDC,
		0, 0, /* dest */
		w->core.width, w->core.height,
		w->core.memDC,
		0, 0, /* src */
		SRCCOPY);
	(void) SelectObject(w->core.memDC, w->core.hOldBitmap);
#else
	XSetGraphicsExposures(XtDisplay(w), w->threed.graphicsGC, False);
	XCopyArea(XtDisplay(w), w->threed.bufferObjects[0], XtWindow(w),
		w->threed.graphicsGC,
		0, 0, /* src */
		w->core.width, w->core.height,
		0, 0); /* dest */
#endif
}

#if 0
static void
EraseFrame(ThreeDWidget w, Pixmap dr, Boolean focus)
{
	DRAWRECTANGLE(w, dr, (focus) ? w->threed.frameGC : w->threed.inverseGC,
		0, 0, w->core.width - 1, w->core.height - 1);
	FILLRECTANGLE(w, dr, w->threed.inverseGC,
		1, 1, w->core.width - 2, w->core.height - 2);
}
#endif

static void
DrawFrame(ThreeDWidget w, Pixmap dr, Boolean focus)
{
	DRAWRECTANGLE(w, dr, (focus) ? w->threed.frameGC : w->threed.borderGC,
			0, 0, w->core.width - 1, w->core.height - 1);
}

static void
SpeedObjects(ThreeDWidget w)
{
	w->threed.delay -= 5;
	if (w->threed.delay < 0)
		w->threed.delay = 0;
#ifdef HAVE_MOTIF
	setThreed(w, ACTION_SPEED);
#endif
}

static void
SlowObjects(ThreeDWidget w)
{
	w->threed.delay += 5;
#ifdef HAVE_MOTIF
	setThreed(w, ACTION_SPEED);
#endif
}

static void
SoundObjects(ThreeDWidget w)
{
	w->threed.sound = !w->threed.sound;
}

static char *
readLine(FILE *fp, char *string, char special)
{
#define MAXLINE 1024
	char buffer[MAXLINE];
	int length, i = 0, j = 0;
	Boolean parsed = False;

	for (;;) {
		if (!fgets(buffer, MAXLINE, fp)) {
			return NULL;
		}
		length = strlen(buffer);
		buffer[length - 1] = '\0';
		i = 0;
		while (special != '\0' && buffer[i] != special &&
				buffer[i] != '\0') {
			if (buffer[i] == '#' ||
					buffer[i] == '!' ||
					buffer[i] == ';') {
				i = length - 1;
			} else
				i++;
		}
		if (special != '\0' && buffer[i] == special) {
			i++;
			special = '\0';
		}
		while (buffer[i] == ' ')
			i++;
		parsed = False;
		j = 0;
		while (!parsed) {
			if (buffer[i] == '#' ||
					buffer[i] == '!' ||
					buffer[i] == ';') {
				string[j] = '\0';
				parsed = True;
			} else {
				string[j] = buffer[i];
				if (string[j] == '\0')
					parsed = True;
				i++;
				j++;
			}
		}
		if (strlen(string)) {
			return string;
		}
	}
}

static void
readCommon(ThreeDWidget w, FILE *fp, char *buffer)
{
	char *string;
	int i, j, k, l, depthIndex;
	double scale;

	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.numObjects));
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.distance.x));
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.distance.y));
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.distance.z));
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.angle.theta));
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.angle.phi));
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.angle.psi));
	w->threed.deltaAngle.theta = 0;
	w->threed.deltaAngle.phi = 0;
	w->threed.deltaAngle.psi = 0;
	w->threed.objects = (Object3D *) malloc(w->threed.numObjects *
		sizeof (Object3D));
#ifdef DEBUG
	(void) fprintf(stderr, "objects%c%d\n", SYMBOL, w->threed.numObjects);
	(void) fprintf(stderr, "distanceX%c%d\n", SYMBOL,
		w->threed.distance.x);
	(void) fprintf(stderr, "distanceY%c%d\n", SYMBOL,
		w->threed.distance.y);
	(void) fprintf(stderr, "distanceZ%c%d\n", SYMBOL,
		w->threed.distance.z);
	(void) fprintf(stderr, "angleTheta%c%d\n", SYMBOL,
		w->threed.angle.theta);
	(void) fprintf(stderr, "anglePhi%c%d\n", SYMBOL,
		w->threed.angle.phi);
	(void) fprintf(stderr, "anglePsi%c%d\n", SYMBOL,
		w->threed.angle.psi);
#endif
	for (i = 0; i < w->threed.numObjects; i++) {
		Object3D *obj = &(w->threed.objects[i]);

		string = readLine(fp, buffer, SYMBOL);
		obj->name = (char *) malloc(strlen(string) + 1);
		(void) strcpy(obj->name, string);
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%lg", &(obj->origin.x));
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%lg", &(obj->origin.y));
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%lg", &(obj->origin.z));
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%lg", &scale);
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%d", &j);
		obj->convex = (j != 0);
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%d", &(obj->numVertices));
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%d", &(obj->numSurfaces));
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%d", &(obj->numEdges));
		obj->local = NULL;
		obj->local = (Point3D *) malloc(obj->numVertices *
			sizeof (Point3D));
		obj->world = NULL;
		obj->world = (Point3D *) malloc(obj->numVertices *
			sizeof (Point3D)); /* rui */
		obj->vertex = NULL;
		obj->vertex = (Vertex *) malloc(obj->numVertices *
			sizeof (Vertex)); /* rua */
		obj->surface = NULL;
		obj->surface = (Surface *) malloc(obj->numSurfaces *
			sizeof (Surface));
		/* numEdges is 2 * the actual edges of the polyhedron */
		obj->map = NULL;
		obj->map = (int *) malloc(obj->numEdges *
			sizeof (int));
#ifdef DEBUG
		(void) fprintf(stderr, "\nNAME%c%s\n", SYMBOL, obj->name);
		(void) fprintf(stderr, "originX%c%lg\n", SYMBOL,
			obj->origin.x);
		(void) fprintf(stderr, "originY%c%lg\n", SYMBOL,
			obj->origin.y);
		(void) fprintf(stderr, "originZ%c%lg\n", SYMBOL,
			obj->origin.z);
		(void) fprintf(stderr, "Scale%c%g\n", SYMBOL, scale);
		(void) fprintf(stderr, "Convex%c%d\n", SYMBOL, obj->convex);
		(void) fprintf(stderr, "Vertices%c%d\n", SYMBOL,
			obj->numVertices);
		(void) fprintf(stderr, "Surfaces%c%d\n", SYMBOL,
			obj->numSurfaces);
		(void) fprintf(stderr, "Edges%c%d\n", SYMBOL,
			obj->numEdges);
#endif
		string = readLine(fp, buffer, SYMBOL);
		for (j = 0; j < obj->numVertices; j++) {
			(void) sscanf(string, "%lg %lg %lg",
				&(obj->local[j].x),
				&(obj->local[j].y),
				&(obj->local[j].z));
			obj->local[j].x *= scale;
			obj->local[j].y *= scale;
			obj->local[j].z *= scale;
			if (j < obj->numVertices - 1)
				string = readLine(fp, buffer, '\0');
		}
#ifdef DEBUG
		(void) fprintf(stderr, "\nVERTICES%c\n", SYMBOL);
		for (j = 0; j < obj->numVertices; j++)
			(void) fprintf(stderr, "%g\t%g\t%g\n",
				obj->local[j].x / scale,
				obj->local[j].y / scale,
				obj->local[j].z / scale);
#endif
		string = readLine(fp, buffer, SYMBOL);
		l = 0;
		for (j = 0; j < obj->numSurfaces; j++) {
			char *str;

			str = string;
			(void) sscanf(string, "%d",
				&(obj->surface[j].numVertices));
			for (k = 0; k < obj->surface[j].numVertices; k++) {
				while (*str != ' ' &&
						*str != '\t' && *str != '\0')
					str++;
				(void) sscanf(str, "%d", &(obj->map[l]));
				obj->map[l++]--;
				if (*str != '\0')
					str++;
			}
			if (j < obj->numSurfaces - 1)
				string = readLine(fp, buffer, '\0');

		}
#ifdef DEBUG
		(void) fprintf(stderr, "\nEDGES%c\n", SYMBOL);
		l = 0;
		for (j = 0; j < obj->numSurfaces; j++) {
			(void) fprintf(stderr, "%d\t",
				obj->surface[j].numVertices);
			for (k = 0; k < obj->surface[j].numVertices; k++)
				(void) fprintf(stderr, "%d\t",
					obj->map[l++] + 1);
			(void) fprintf(stderr, "\n");
		}
#endif
		string = readLine(fp, buffer, SYMBOL);
		for (j = 0; j < obj->numSurfaces; j++) {
			(void) sscanf(string, "%d %d %d",
				&(obj->surface[j].mapIndex),
				&depthIndex,
				&(obj->surface[j].color));
			if (j < obj->numSurfaces - 1)
				string = readLine(fp, buffer, '\0');
#ifdef DEPTHINDEX
			obj->surface[j].depthIndex = depthIndex;
#else
			depthIndex = 0;
#endif
		}
#ifdef DEBUG
		(void) fprintf(stderr, "\nSURFACES%c\n", SYMBOL);
		for (j = 0; j < obj->numSurfaces; j++)
			(void) fprintf(stderr, "%d\t%d\t%d\n",
				obj->surface[j].mapIndex,
#ifdef DEPTHINDEX
				obj->surface[j].depthIndex,
#else
				depthIndex,
#endif
				obj->surface[j].color);
#endif
		for (j = 0; j < obj->numSurfaces; j++) {
			Vector3D u, v, normal;
			int vertex[3];
			int *vertexList = &(obj->map[obj->surface[j].mapIndex]);

			obj->surface[j].twoSided = False;
			obj->surface[j].visible = 1;
			obj->surface[j].clipped = False;
			obj->surface[j].active = True;
			if (obj->surface[j].numVertices >= 3) {
				vertex[0] = *vertexList;
				vertexList++;
				vertex[1] = *vertexList;
				vertexList++;
				vertex[2] = *vertexList;
				createVector3D(
					&(obj->local[vertex[0]]),
					&(obj->local[vertex[1]]),
					&u);
				createVector3D(
					&(obj->local[vertex[0]]),
					&(obj->local[vertex[2]]),
					&v);
				crossProduct3D(&v, &u, &normal);
				obj->surface[j].normalLength =
					magnitudeVector3D(&normal);
			} else {
				obj->surface[j].normalLength = 0.0;
			}
		}
		obj->radius = maximumObjectRadius(obj);
	}
}

#ifdef WINVER
static FILE *
OpenIni(const HWND hWnd, const char *filename)
{
	unsigned short wReturn;
	char buf[256], szBuf[144];
	FILE *fp;

#ifdef DEBUG
	(void) fprintf(stderr, "INI=%s\n", filename);
#endif
	if ((fp = fopen(filename, "r")) == NULL) {
		wReturn = GetWindowsDirectory((LPSTR) szBuf, sizeof (szBuf));
		if (!wReturn || wReturn > sizeof (szBuf)) {
			(void) MessageBox(hWnd,
				(!wReturn) ? "threed: function failed" :
			"threed: buffer is too small", "GetWindowsDirectory",
					MB_ICONEXCLAMATION);
#ifdef DEBUG
			(void) fclose(fp);
#endif
			exit(1);
		}
		(void) sprintf(buf, "%s\\%s", szBuf, filename);
#ifdef DEBUG
		(void) fprintf(stderr, "INI=%s\n", buf);
#endif
		if ((fp = fopen(buf, "r")) == NULL) {
			(void) sprintf(buf,
				"%s: does not exist in \".\" or windows directory", filename);
			(void) MessageBox(hWnd, buf, "OpenIni", MB_ICONEXCLAMATION);
#ifdef DEBUG
			(void) fclose(fp);
#endif
			exit(1);
		}
	}
#ifdef DEBUG
	(void) fprintf(stderr, "Exiting OpenIni\n");
#endif

	return fp;
}
/*-
   Not using GetProfile[String,Int] because most of the data is on
   multiple lines, repeated, and dynamic.  Data must be in proper order
   i.e. information left of "=" is not read.
 */
static void
ReadThreeD(ThreeDWidget w)
{
	FILE *fp;
	char buffer[MAXLINE];
	char *string;
	int i;
	struct tagColor {
		int red, green, blue;
	} color;

	fp = OpenIni(w->core.hWnd, DATAFILE);
	for (i = 0; i <= COLORS; i++) {
		string = readLine(fp, buffer, SYMBOL);
		(void) sscanf(string, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->threed.shadeGC[i] = RGB(color.red, color.green, color.blue);
	}
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.delay));
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &i);
	w->threed.sound = (i != 0);
	string = readLine(fp, buffer, SYMBOL);
	(void) strcpy(w->threed.bumpSound, string);
	string = readLine(fp, buffer, SYMBOL);
	(void) strcpy(w->threed.moveSound, string);
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &i);
	w->threed.surface = (i != 0);
	string = readLine(fp, buffer, SYMBOL);
	(void) sscanf(string, "%d", &(w->threed.object));
#ifdef DEBUG
	for (i = 0; i <= COLORS; i++)
		(void) fprintf(stderr, "brush%d%c%d %d %d\n", i, SYMBOL,
			GetRValue(w->threed.shadeGC[i]),
			GetGValue(w->threed.shadeGC[i]),
			GetBValue(w->threed.shadeGC[i]));
	(void) fprintf(stderr, "Surface%c%d\n", SYMBOL,
		(w->threed.surface) ? 1 : 0 );
	(void) fprintf(stderr, "Object%c%d\n", SYMBOL, w->threed.object);
#endif
	readCommon(w, fp, buffer);
	(void) fclose(fp);
}

void
DestroyThreeD(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else

static void
ReadThreeD(ThreeDWidget w)
{
	FILE *fp;
	char buffer[MAXLINE];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, DATAFILE);
	lname = buf1;
	stringCat(&buf1, DATAPATH, FINALDELIM);
	stringCat(&buf2, buf1, DATAFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
	}
	readCommon(w, fp, buffer);
	(void) fclose(fp);
}

static Boolean
SetValuesThreeD(Widget current, Widget request, Widget renew)
{
	ThreeDWidget c = (ThreeDWidget) current, w = (ThreeDWidget) renew;
	XGCValues values;
	XtGCMask valueMask;
	Boolean redraw = FALSE;

	if (w->threed.numObjects == 0)
		return False;
	CheckGraphicsThreeD(w);
	if (w->threed.foreground != c->threed.foreground) {
		valueMask = GCForeground | GCBackground;
		values.foreground = w->threed.foreground;
		values.background = w->core.background_pixel;
		XtReleaseGC(renew, w->threed.graphicsGC);
		w->threed.graphicsGC = XtGetGC(renew, valueMask, &values);
		redraw = TRUE;
	}
	if (w->threed.distance.x != c->threed.distance.x ||
			w->threed.distance.y != c->threed.distance.y ||
			w->threed.distance.z != c->threed.distance.z ||
			w->threed.angle.theta != c->threed.angle.theta ||
			w->threed.angle.phi != c->threed.angle.phi ||
			w->threed.angle.psi != c->threed.angle.psi ||
			w->threed.surface != c->threed.surface ||
			w->threed.object != c->threed.object) {
		w->threed.name = w->threed.objects[w->threed.object].name;
#ifdef DEBUG
		(void) printf("%s\n", w->threed.name);
#endif
		w->threed.deltaAngle.theta = w->threed.angle.theta -
			c->threed.angle.theta;
		w->threed.deltaAngle.phi = w->threed.angle.phi -
			c->threed.angle.phi;
		w->threed.deltaAngle.psi = w->threed.angle.psi -
			c->threed.angle.psi;
		ResizeThreeD(w);
		redraw = TRUE;
	}
	if (w->threed.menu != ACTION_IGNORE) {
		int menu = w->threed.menu;

		w->threed.menu = ACTION_IGNORE;
		switch (menu) {
		case ACTION_SPEED:
			SpeedObjects(w);
			break;
		case ACTION_SLOW:
			SlowObjects(w);
			break;
		case ACTION_SOUND:
			SoundObjects(w);
			break;
		default:
			break;
		}
	}
	return (redraw);
}

static void
SetAllColors(ThreeDWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int color;

	valueMask = GCForeground | GCBackground;
	if (w->threed.reverse) {
		values.foreground = w->threed.foreground;
		values.background = w->threed.background;
	} else {
		values.foreground = w->threed.background;
		values.background = w->threed.foreground;
	}
	if (w->threed.inverseGC)
		XtReleaseGC((Widget) w, w->threed.inverseGC);
	w->threed.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->threed.reverse) {
		values.background = w->threed.foreground;
		values.foreground = w->threed.background;
	} else {
		values.background = w->threed.background;
		values.foreground = w->threed.foreground;
	}
	if (w->threed.graphicsGC)
		XtReleaseGC((Widget) w, w->threed.graphicsGC);
	w->threed.graphicsGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->threed.mono) {
		if (w->threed.reverse) {
			values.background = w->threed.foreground;
			values.foreground = w->threed.background;
		} else {
			values.background = w->threed.background;
			values.foreground = w->threed.foreground;
		}
	} else {
		values.foreground = w->threed.frameColor;
		values.background = w->threed.borderColor;
	}
	if (w->threed.frameGC)
		XtReleaseGC((Widget) w, w->threed.frameGC);
	w->threed.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->threed.mono) {
		if (w->threed.reverse) {
			values.foreground = w->threed.foreground;
			values.background = w->threed.background;
		} else {
			values.foreground = w->threed.background;
			values.background = w->threed.foreground;
		}
	} else {
		values.foreground = w->threed.borderColor;
		values.background = w->threed.frameColor;
	}
	if (w->threed.borderGC)
		XtReleaseGC((Widget) w, w->threed.borderGC);
	w->threed.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (color = 0; color < COLORS; color++) {
		if (!w->threed.mono) {
			values.foreground = w->threed.shadeColor[color];
		}
		if (w->threed.shadeGC[color])
			XtReleaseGC((Widget) w, w->threed.shadeGC[color]);
		w->threed.shadeGC[color] = XtGetGC((Widget) w, valueMask,
			&values);
	}
}

static void
QuitThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}

static void
DestroyThreeD(Widget old)
{
	ThreeDWidget w = (ThreeDWidget) old;
	int color;

	for (color = 0; color < COLORS; color++)
		XtReleaseGC(old, w->threed.shadeGC[color]);
	XtReleaseGC(old, w->threed.graphicsGC);
	XtReleaseGC(old, w->threed.borderGC);
	XtReleaseGC(old, w->threed.frameGC);
	XtReleaseGC(old, w->threed.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->threed.select);
}
#endif

static void
RepaintGraphicsThreeD(ThreeDWidget w)
{
	int sel;

#ifdef WINVER
	if (w->core.memDC == NULL) {
		w->core.memDC = CreateCompatibleDC(w->core.hDC);
		if (w->core.memDC == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
#else
	Display *display = XtDisplay(w);
	Window window = XtWindow(w);
	XWindowAttributes xgwa;

	(void) XGetWindowAttributes(display, window, &xgwa);
	if (w->threed.oldColormap == None) {
		w->threed.mono = (xgwa.depth < 2 || w->threed.mono);
		w->threed.oldColormap = xgwa.colormap;
	}
#endif
	if (w->threed.numObjects == 0)
		return;
	for (sel = 0; sel < 2; sel++) {
#ifdef WINVER
		if (w->threed.bufferObjects[sel] != NULL) {
			DeleteObject(w->threed.bufferObjects[sel]);
			w->threed.bufferObjects[sel] = NULL;
		}
		if ((w->threed.bufferObjects[sel] = CreateCompatibleBitmap(
				w->core.hDC,
				w->core.width, w->core.height)) == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
#else
		if (w->threed.bufferObjects[sel] != None) {
			XFreePixmap(display, w->threed.bufferObjects[sel]);
			w->threed.bufferObjects[sel] = None;
		}
		if ((w->threed.bufferObjects[sel] = XCreatePixmap(display,
				window, w->core.width, w->core.height,
				xgwa.depth)) == None) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
#endif
	}
#ifndef WINVER
	SetAllColors(w);
#endif
	/*EraseFrame(w, 0, w->threed.focus);*/
	/*DrawAllBufferedGraphics(w);
	DrawAllGraphics(w);
#ifndef WINVER
	XClearWindow(XtDisplay(w), XtWindow(w));
#endif */
#ifndef WINVER
	/* Clear buffer */
	XFillRectangle(XtDisplay(w), w->threed.bufferObjects[0],
		w->threed.inverseGC, 0, 0, w->core.width, w->core.height);
#endif
	RealizeObject(w, &(w->threed.objects[w->threed.object]));
	DrawObject(w, &(w->threed.objects[w->threed.object]));
	CopyFromBuffer(w);
}

static void
ResizeThreeD(ThreeDWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->threed.center.x = w->core.width / 2;
	w->threed.center.y = w->core.height / 2;
	w->threed.size = MIN(w->threed.center.x, w->threed.center.y);
}

#ifndef WINVER
static
#endif
void
InitializeThreeD(
#ifdef WINVER
ThreeDWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int i;

#ifdef WINVER
	w->threed.mono = DEFAULTMONO;
	w->threed.reverse = DEFAULTREVERSE;
	w->threed.bufferObjects[0] = NULL;
	w->threed.bufferObjects[1] = NULL;
#else
	ThreeDWidget w = (ThreeDWidget) renew;
	int color;

	w->threed.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->threed.mono);
	w->threed.bufferObjects[0] = None;
	w->threed.bufferObjects[1] = None;
	w->threed.colormap = None;
	w->threed.oldColormap = None;
	w->threed.fontInfo = NULL;
	for (color = 0; color < COLORS; color++)
		w->threed.shadeGC[color] = NULL;
	w->threed.graphicsGC = NULL;
	w->threed.borderGC = NULL;
	w->threed.frameGC = NULL;
	w->threed.inverseGC = NULL;
#endif
	createTrigTables();
	ReadThreeD(w);
	CheckGraphicsThreeD(w);
	if (w->threed.numObjects > 0) {
		w->threed.name = w->threed.objects[w->threed.object].name;
		if (!(w->threed.list = (char **) calloc((unsigned int)
				w->threed.numObjects, sizeof (char **)))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		for (i = 0; i < w->threed.numObjects; i++)
			w->threed.list[i] = w->threed.objects[i].name;
	}
	w->threed.deltaAngle.theta = w->threed.angle.theta;
	w->threed.deltaAngle.phi = w->threed.angle.phi;
	w->threed.deltaAngle.psi = w->threed.angle.psi;
	w->threed.currentPosition.x = 2 * w->core.width;
	w->threed.currentPosition.y = 2 * w->core.height;
	ResizeThreeD(w);
#ifdef DEBUG
	(void) printf("%s\n", w->threed.name);
#endif
#ifdef WINVER
	brush = CreateSolidBrush(w->threed.inverseGC);
	SETBACK(w->core.hWnd, brush);
#else
	SetAllColors(w);
#endif
}

#ifndef WINVER
static
#endif
void
ExposeThreeD(
#ifdef WINVER
ThreeDWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifndef WINVER
	ThreeDWidget w = (ThreeDWidget) renew;

	if (!w->core.visible)
		return;
#endif
	DrawFrame(w, 0, w->threed.focus);
	RepaintGraphicsThreeD(w);
}

#ifndef WINVER
static
#endif
void
HideThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	setThreed(w, ACTION_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectThreeD(ThreeDWidget w
#ifdef WINVER
, const int x, const int y/*, const int control*/
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	/*int control = (int) (event->xkey.state & ControlMask);*/
#endif
	w->threed.currentPosition.x = x;
	w->threed.currentPosition.y = y;
	RepaintGraphicsThreeD(w);
}

#ifndef WINVER
static
#endif
void
MotionThreeD(ThreeDWidget w
#ifdef WINVER
, const int x, const int y/*, const int shift, const int control*/
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	/*int shift = (int) (event->xbutton.state & (ShiftMask | LockMask));
	int control = (int) (event->xkey.state & ControlMask);*/
#endif
	int oldAngleTheta = w->threed.angle.theta;
	int oldAnglePhi = w->threed.angle.phi;

	w->threed.angle.theta += w->threed.currentPosition.x - x;
	w->threed.angle.phi += w->threed.currentPosition.y - y;
	w->threed.angle.theta = (64 * NUM_DEGREES + w->threed.angle.theta) %
		NUM_DEGREES;
	w->threed.angle.phi = (64 * NUM_DEGREES + w->threed.angle.phi) %
		NUM_DEGREES;
	w->threed.deltaAngle.theta = w->threed.angle.theta - oldAngleTheta;
	w->threed.deltaAngle.phi = w->threed.angle.phi - oldAnglePhi;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
	w->threed.currentPosition.x = x;
	w->threed.currentPosition.y = y;
}

#ifndef WINVER
static
#endif
void
ReleaseThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->threed.currentPosition.x = 2 * w->core.width;
	w->threed.currentPosition.y = 2 * w->core.height;
	RepaintGraphicsThreeD(w);
}

void
SpeedThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SpeedObjects(w);
}

void
SlowThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SlowObjects(w);
}

void
SoundThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SoundObjects(w);
}

#ifdef WINVER
void
setSurfaceThreeD(ThreeDWidget w , Boolean surface)
{
	w->threed.surface = surface;
}

void
setObjectThreeD(ThreeDWidget w, int object3D)
{
	w->threed.object = object3D;
}

#else

static
void
SurfaceThreeD(ThreeDWidget w
, XEvent *event, char **args, int n_args
)
{
	setThreed(w, ACTION_SURFACE);
}

static
void
ObjectThreeD(ThreeDWidget w
, XEvent *event, char **args, int n_args
)
{
	setThreed(w, ACTION_OBJECT);
}

void
EnterThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->threed.focus = True;
	DrawFrame(w, 0, w->threed.focus);
}

void
LeaveThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->threed.focus = False;
	DrawFrame(w, 0, w->threed.focus);
}

static void
MoveLowerPhiThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	int oldAngle = w->threed.angle.phi;

	w->threed.angle.phi -= DELTADEGREES;
	if (w->threed.angle.phi < MINDEGREES)
		w->threed.angle.phi += NUM_DEGREES;
	w->threed.deltaAngle.phi = w->threed.angle.phi - oldAngle;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveLowerThetaThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	int oldAngle = w->threed.angle.theta;

	w->threed.angle.theta -= DELTADEGREES;
	if (w->threed.angle.theta < MINDEGREES)
		w->threed.angle.theta += NUM_DEGREES;
	w->threed.deltaAngle.theta = w->threed.angle.theta - oldAngle;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRaiseThetaThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	int oldAngle = w->threed.angle.theta;

	w->threed.angle.theta += DELTADEGREES;
	if (w->threed.angle.theta > MAXDEGREES)
		w->threed.angle.theta -= NUM_DEGREES;
	w->threed.deltaAngle.theta = w->threed.angle.theta - oldAngle;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRaisePhiThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	int oldAngle = w->threed.angle.phi;

	w->threed.angle.phi += DELTADEGREES;
	if (w->threed.angle.phi > MAXDEGREES)
		w->threed.angle.phi -= NUM_DEGREES;
	w->threed.deltaAngle.phi = w->threed.angle.phi - oldAngle;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveLowerPsiThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	int oldAngle = w->threed.angle.psi;

	w->threed.angle.psi -= DELTADEGREES;
	if (w->threed.angle.psi < MINDEGREES)
		w->threed.angle.psi += NUM_DEGREES;
	w->threed.deltaAngle.psi = w->threed.angle.psi - oldAngle;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRaisePsiThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	int oldAngle = w->threed.angle.psi;

	w->threed.angle.psi += DELTADEGREES;
	if (w->threed.angle.psi > MAXDEGREES)
		w->threed.angle.psi -= NUM_DEGREES;
	w->threed.deltaAngle.psi = w->threed.angle.psi - oldAngle;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveLowerThetaLowerPhiThreeD(ThreeDWidget w, XEvent *event,
		char **args, int n_args)
{
	int oldAngleTheta = w->threed.angle.theta;
	int oldAnglePhi = w->threed.angle.phi;

	w->threed.angle.theta -= DELTADEGREES;
	w->threed.angle.phi -= DELTADEGREES;
	if (w->threed.angle.theta < MINDEGREES)
		w->threed.angle.theta += NUM_DEGREES;
	if (w->threed.angle.phi < MINDEGREES)
		w->threed.angle.phi += NUM_DEGREES;
	w->threed.deltaAngle.theta = w->threed.angle.theta - oldAngleTheta;
	w->threed.deltaAngle.phi = w->threed.angle.phi - oldAnglePhi;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveLowerThetaRaisePhiThreeD(ThreeDWidget w, XEvent *event,
		char **args, int n_args)
{
	int oldAngleTheta = w->threed.angle.theta;
	int oldAnglePhi = w->threed.angle.phi;

	w->threed.angle.theta -= DELTADEGREES;
	w->threed.angle.phi += DELTADEGREES;
	if (w->threed.angle.theta < MINDEGREES)
		w->threed.angle.theta += NUM_DEGREES;
	if (w->threed.angle.phi > MAXDEGREES)
		w->threed.angle.phi -= NUM_DEGREES;
	w->threed.deltaAngle.theta = w->threed.angle.theta - oldAngleTheta;
	w->threed.deltaAngle.phi = w->threed.angle.phi - oldAnglePhi;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRaiseThetaLowerPhiThreeD(ThreeDWidget w, XEvent *event,
		char **args, int n_args)
{
	int oldAngleTheta = w->threed.angle.theta;
	int oldAnglePhi = w->threed.angle.phi;

	w->threed.angle.theta += DELTADEGREES;
	w->threed.angle.phi -= DELTADEGREES;
	if (w->threed.angle.theta > MAXDEGREES)
		w->threed.angle.theta -= NUM_DEGREES;
	if (w->threed.angle.phi < MINDEGREES)
		w->threed.angle.phi += NUM_DEGREES;
	w->threed.deltaAngle.theta = w->threed.angle.theta - oldAngleTheta;
	w->threed.deltaAngle.phi = w->threed.angle.phi - oldAnglePhi;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRaiseThetaRaisePhiThreeD(ThreeDWidget w, XEvent *event,
		char **args, int n_args)
{
	int oldAngleTheta = w->threed.angle.theta;
	int oldAnglePhi = w->threed.angle.phi;

	w->threed.angle.theta += DELTADEGREES;
	w->threed.angle.phi += DELTADEGREES;
	if (w->threed.angle.theta > MAXDEGREES)
		w->threed.angle.theta -= NUM_DEGREES;
	if (w->threed.angle.phi > MAXDEGREES)
		w->threed.angle.phi -= NUM_DEGREES;
	w->threed.deltaAngle.theta = w->threed.angle.theta - oldAngleTheta;
	w->threed.deltaAngle.phi = w->threed.angle.phi - oldAnglePhi;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRightThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	if (w->threed.distance.x < MAXDISTANCE)
		w->threed.distance.x += DELTADISTANCE;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveLeftThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	if (w->threed.distance.x > MINDISTANCE)
		w->threed.distance.x -= DELTADISTANCE;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveUpThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	if (w->threed.distance.y < MAXDISTANCE)
		w->threed.distance.y += DELTADISTANCE;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveDownThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	if (w->threed.distance.y > MINDISTANCE)
		w->threed.distance.y -= DELTADISTANCE;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveOutThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	if (w->threed.distance.z < MAXDEPTH)
		w->threed.distance.z += DELTADEPTH;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveInThreeD(ThreeDWidget w, XEvent *event, char **args, int n_args)
{
	if (w->threed.distance.z > MINDEPTH)
		w->threed.distance.z -= DELTADEPTH;
	setThreed(w, ACTION_SET);
	RepaintGraphicsThreeD(w);
}
#endif
