/*-
# X-BASED DINOSAUR CUBE
#
#  DinoGL.c
#
###
#
#  Copyright (c) 2007	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 *event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Thanks goes also to Brian Paul for making it possible and inexpensive
 * to use OpenGL at home.
 *
 * Based on rubik.c by Marcelo F. Vianna <mfvianna@centroin.com.br>
 */

#ifdef HAVE_OPENGL

/* Methods file for DinoGL */

#include "DinoP.h"
#include "DinoGLP.h"

static float front_shininess[] =
{60.0};
static float front_specular[] =
{0.7, 0.7, 0.7, 1.0};
static float ambient[] =
{0.0, 0.0, 0.0, 1.0};
static float diffuse[] =
{1.0, 1.0, 1.0, 1.0};
static float position0[] =
{1.0, 1.0, 1.0, 0.0};
static float position1[] =
{-1.0, -1.0, 1.0, 0.0};
static float lmodel_ambient[] =
{0.5, 0.5, 0.5, 1.0};
static float lmodel_twoside[] =
{GL_TRUE};

static float MaterialMagenta[] =
{0.7, 0.0, 0.7, 1.0};
static float MaterialCyan[] =
{0.0, 0.7, 0.7, 1.0};
static float MaterialGray[] =
{0.2, 0.2, 0.2, 1.0};

/*-
 * Color labels mapping:
 * =====================
 *
 *        +------+
 *        |  0   |
 *        |      |
 *        | TOP  |
 *        |3(0) 1|
 *        | RED  |
 *        |  2   |
 * +------+------+------+
 * |  0   |  0   |  0   |
 * |      |      |      |
 * | LEFT |FRONT |RIGHT |
 * |3(1) 1|3(2) 1|3(3) 1|
 * | BLUE |YELLOW|GREEN |
 * |  2   |  2   |  2   |
 * +------+------+------+
 *        |  0   |
 *        |      |
 *        |BOTTOM|
 *        |3(4) 1|
 *        |WHITE |
 *        |  2   |
 *        +------+         +------+
 *        |  0   |         |\ 0  /|
 *        |      |         | \  / |
 *        | BACK |         | xxxx |
 *        |3(5) 1|         |3(N) 1|
 *        |ORANGE|         | /  \ |
 *        |  2   |         |/ 2  \|
 *        +------+         +------+
 *
 *  Map to 3d
 *  FRONT  => X, Y
 *  BACK   => X, Y
 *  LEFT   => Z, Y
 *  RIGHT  => Z, Y
 *  TOP    => X, Z
 *  BOTTOM => X, Z
 */

static Boolean madeCurrent = False;

#ifdef WINVER
static HGLRC hRC = NULL;
#else
static GLXContext *glXContext = (GLXContext *) NULL;
static Boolean SetValuesPuzzleGL(Widget current, Widget request, Widget renew);
static void ResizePuzzleGL(DinoGLWidget w);
static void InitializePuzzleGL(Widget request, Widget renew);
static void ExposePuzzleGL(Widget renew, XEvent *event, Region region);
static void MovePuzzleGLTl(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLTop(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLTr(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLLeft(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLRight(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLBl(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLBottom(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleGLBr(DinoGLWidget w,
	XEvent *event, char **args, int nArgs);

static char defaultTranslationsPuzzleGL[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>2: Period2()\n\
 <KeyPress>3: Period3()\n\
 <KeyPress>b: Both()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPuzzleGL[] =
{
	{(char *) "Quit", (XtActionProc) QuitPuzzle},
	{(char *) "Hide", (XtActionProc) HidePuzzle},
	{(char *) "MoveTl", (XtActionProc) MovePuzzleGLTl},
	{(char *) "MoveTop", (XtActionProc) MovePuzzleGLTop},
	{(char *) "MoveTr", (XtActionProc) MovePuzzleGLTr},
	{(char *) "MoveLeft", (XtActionProc) MovePuzzleGLLeft},
	{(char *) "MoveRight", (XtActionProc) MovePuzzleGLRight},
	{(char *) "MoveBl", (XtActionProc) MovePuzzleGLBl},
	{(char *) "MoveBottom", (XtActionProc) MovePuzzleGLBottom},
	{(char *) "MoveBr", (XtActionProc) MovePuzzleGLBr},
	{(char *) "Select", (XtActionProc) SelectPuzzle},
	{(char *) "Release", (XtActionProc) ReleasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) PracticePuzzleMaybe},
	{(char *) "Practice2", (XtActionProc) PracticePuzzle2},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizePuzzleMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizePuzzle2},
	{(char *) "Get", (XtActionProc) GetPuzzle},
	{(char *) "Write", (XtActionProc) WritePuzzle},
	{(char *) "Undo", (XtActionProc) UndoPuzzle},
	{(char *) "Redo", (XtActionProc) RedoPuzzle},
	{(char *) "Clear", (XtActionProc) ClearPuzzle},
	{(char *) "Randomize", (XtActionProc) RandomizePuzzle},
	{(char *) "Solve", (XtActionProc) SolvePuzzle},
	{(char *) "Practice", (XtActionProc) PracticePuzzle},
	{(char *) "Orientize", (XtActionProc) OrientizePuzzle},
	{(char *) "Period2", (XtActionProc) Period2ModePuzzle},
	{(char *) "Period3", (XtActionProc) Period3ModePuzzle},
	{(char *) "Both", (XtActionProc) BothModePuzzle},
	{(char *) "View", (XtActionProc) ViewPuzzle},
	{(char *) "Speed", (XtActionProc) SpeedPuzzle},
	{(char *) "Slow", (XtActionProc) SlowPuzzle},
	{(char *) "Sound", (XtActionProc) SoundPuzzle},
	{(char *) "Enter", (XtActionProc) EnterPuzzle},
	{(char *) "Leave", (XtActionProc) LeavePuzzle}
};

static XtResource resourcesPuzzleGL[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(DinoWidget, core.width),
	 XtRString, (caddr_t) "250"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(DinoWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[1]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[2]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[4]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[5]),
	 XtRString, (caddr_t) "Orange"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNview, XtCView, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.view),
	 XtRString, (caddr_t) "1"},
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.mode),
	 XtRString, (caddr_t) "3"}, /*DEFAULTMODE */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULTORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULTPRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNstyle, XtCStyle, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.style),
	 XtRString, (caddr_t) "-1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
         XtOffset(DinoWidget, dino.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(DinoWidget, dino.select),
	 XtRCallback, (caddr_t) NULL}
};

DinoGLClassRec dinoGLClassRec =
{
	{
		(WidgetClass) & dinoClassRec,	/* superclass */
		(char *) "DinoGL",	/* class name */
		sizeof (DinoGLRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePuzzleGL,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPuzzleGL,	/* actions */
		XtNumber(actionsListPuzzleGL),	/* num actions */
		resourcesPuzzleGL,	/* resources */
		XtNumber(resourcesPuzzleGL),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizePuzzleGL,	/* resize */
		(XtExposeProc) ExposePuzzleGL,	/* expose */
		(XtSetValuesFunc) SetValuesPuzzleGL,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPuzzleGL,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass dinoGLWidgetClass = (WidgetClass) & dinoGLClassRec;
#endif

#ifdef DEBUG
void
printCube(DinoGLWidget w)
{
	int face, position;

	for (face = 0; face < MAXFACES; face++) {
		for (position = 0; position < MAXORIENT; position++) {
			(void) printf("%d %d  ",
				w->dino.cubeLoc[face][position].face,
				w->dino.cubeLoc[face][position].rotation);
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}
#endif

static void
pickColor(DinoGLWidget w, int C, int mono)
{
	float Material[] = {1.0, 1.0, 1.0, 1.0};

	switch (C) {
	case TOP_FACE:
	case LEFT_FACE:
	case FRONT_FACE:
	case RIGHT_FACE:
	case BOTTOM_FACE:
	case BACK_FACE:
		break;
	case TEST_FACE:
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			MaterialMagenta);
		return;
	default:
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			MaterialCyan);
		return;
	}
        {
#ifdef WINVER
	struct tagColor {
		int red, green, blue;
	} color;

	color.red = GetRValue(w->dino.faceGC[C]);
	color.green = GetGValue(w->dino.faceGC[C]);
	color.blue = GetBValue(w->dino.faceGC[C]);
#define MAXCOLOR 0xFF
#else
	XColor color;

	color.pixel = w->dino.faceColor[C];
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
#define MAXCOLOR 0xFFFF
#endif
#ifdef DEBUG
	(void) printf("i%d %d %d\n", color.red, color.green, color.blue);
#endif
	if (mono) {
		/* really GrayScale */
		float intensity = 0.3 * color.red + 0.59 * color.green +
			0.11 * color.blue;

#ifdef DEBUG
		(void) printf("m%g\n", intensity);
#endif
		Material[0] *= (float) intensity / MAXCOLOR;
		Material[1] *= (float) intensity / MAXCOLOR;
		Material[2] *= (float) intensity / MAXCOLOR;
	} else {
		Material[0] *= (float) color.red / MAXCOLOR;
		Material[1] *= (float) color.green / MAXCOLOR;
		Material[2] *= (float) color.blue / MAXCOLOR;
	}
	}
#ifdef DEBUG
	(void) printf("f%g %g %g\n", Material[0], Material[1], Material[2]);
#endif
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, Material);
}

static Boolean
draw_stickerless_cubit(Boolean sep)
{
	if (sep) {
		glBegin(GL_TRIANGLES);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
		/* Internal to cubit */
		glNormal3f(0.0, 1.0, 1.0);
		glVertex3f(-CUBEROUND, CUBELEN, -CUBELEN);
		glVertex3f(CUBEROUND, CUBELEN, -CUBEFULLROUND - CUBELEN);
		glVertex3f(0, CUBELEN - CUBEROUND, -CUBELEN);
		glNormal3f(0.0, 1.0, 1.0);
		glVertex3f(CUBELEN, CUBEROUND, -CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN, -CUBEROUND, -CUBELEN);
		glVertex3f(CUBELEN - CUBEROUND, 0, -CUBELEN);
		glNormal3f(0.0, 1.0, 1.0);
		glVertex3f(CUBELEN, -CUBEROUND, -CUBELEN);
		glVertex3f(CUBEROUND, CUBELEN, CUBEFULLROUND - CUBELEN);
		glVertex3f(0, CUBELEN - CUBEROUND, -CUBELEN);
		glNormal3f(0.0, 1.0, 1.0);
		glVertex3f(CUBELEN, CUBEROUND, CUBEFULLROUND - CUBELEN);
		glVertex3f(-CUBEROUND, CUBELEN, -CUBELEN);
		glVertex3f(CUBELEN - CUBEROUND, 0, -CUBELEN);
		glNormal3f(1.0, 1.0, 0.0);
		glVertex3f(CUBEROUND, CUBELEN, -CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBEROUND, CUBELEN, CUBEFULLROUND - CUBELEN);
		glVertex3f(0, CUBELEN - CUBEROUND, -CUBELEN);
		glNormal3f(1.0, 1.0, 0.0);
		glVertex3f(CUBELEN, CUBEROUND, CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN, CUBEROUND, -CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN - CUBEROUND, 0, -CUBELEN);
		glEnd();
	} else {
		glBegin(GL_QUADS);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
		/* Edge of cubit */
		glNormal3f(1.0, 1.0, 0.0);
		glVertex3f(CUBEROUND, CUBELEN, -CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBEROUND, CUBELEN, CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN, CUBEROUND, CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN, CUBEROUND, -CUBEFULLROUND - CUBELEN);
		/* Internal to cubit */
		glNormal3f(0.0, 1.0, 1.0);
		glVertex3f(CUBEROUND, CUBELEN, -CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN, CUBEROUND, -CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN, -CUBEROUND, -CUBELEN);
		glVertex3f(-CUBEROUND, CUBELEN, -CUBELEN);
		glNormal3f(0.0, 1.0, 1.0);
		glVertex3f(CUBEROUND, CUBELEN, CUBEFULLROUND - CUBELEN);
		glVertex3f(CUBELEN, CUBEROUND, CUBEFULLROUND - CUBELEN);
		glVertex3f(-CUBEROUND, CUBELEN, -CUBELEN);
		glVertex3f(CUBELEN, -CUBEROUND, -CUBELEN);
		glEnd();
	}
	glBegin(GL_TRIANGLES);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
	/* Put sticker here */
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(CUBELEN, CUBEROUND, -CUBEFULLROUND - CUBELEN);
	glVertex3f(CUBELEN, CUBEROUND, CUBEFULLROUND - CUBELEN);
	glVertex3f(CUBELEN, -CUBEROUND, -CUBELEN);
	glNormal3f(0.0, 1.0, 0.0);
	glVertex3f(-CUBEROUND, CUBELEN, -CUBELEN);
	glVertex3f(CUBEROUND, CUBELEN, CUBEFULLROUND - CUBELEN);
	glVertex3f(CUBEROUND, CUBELEN, -CUBEFULLROUND - CUBELEN);
	glEnd();
	return True;
}

static void
DrawOrientLine(DinoGLWidget w, int face, int rotation,
		float x, float y, float z)
{
	int err = 0;
	char *buf1, *buf2;

	glTranslatef(x, y, z);
	glBegin(GL_QUADS);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MaterialGray);
	switch (face) {
	case 0:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, -STICKERSHORT);
			glVertex3f(-CUT, 0.0, -STICKERSHORT);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(STICKERSHORT, 0.0, CUT);
			glVertex3f(STICKERSHORT, 0.0, -CUT);
			break;
		case 2:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, STICKERSHORT);
			glVertex3f(CUT, 0.0, STICKERSHORT);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-STICKERSHORT, 0.0, -CUT);
			glVertex3f(-STICKERSHORT, 0.0, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 1:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, STICKERSHORT, CUT);
			glVertex3f(0.0, STICKERSHORT, -CUT);
			break;
		case 1:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, STICKERSHORT);
			glVertex3f(0.0, CUT, STICKERSHORT);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -STICKERSHORT, -CUT);
			glVertex3f(0.0, -STICKERSHORT, CUT);
			break;
		case 3:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, CUT, -STICKERSHORT);
			glVertex3f(0.0, -CUT, -STICKERSHORT);
			break;
		default:
			err = 1;
		}
		break;
	case 2:
		switch (rotation) {
		case 0:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, STICKERSHORT, 0.0);
			glVertex3f(-CUT, STICKERSHORT, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(STICKERSHORT, -CUT, 0.0);
			glVertex3f(STICKERSHORT, CUT, 0.0);
			break;
		case 2:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, -STICKERSHORT, 0.0);
			glVertex3f(CUT, -STICKERSHORT, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-STICKERSHORT, CUT, 0.0);
			glVertex3f(-STICKERSHORT, -CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	case 3:
		switch (rotation) {
		case 0:
			glVertex3f(0.0, CUT, -CUT);
			glVertex3f(0.0, STICKERSHORT, -CUT);
			glVertex3f(0.0, STICKERSHORT, CUT);
			glVertex3f(0.0, CUT, CUT);
			break;
		case 1:
			glVertex3f(0.0, -CUT, -CUT);
			glVertex3f(0.0, -CUT, -STICKERSHORT);
			glVertex3f(0.0, CUT, -STICKERSHORT);
			glVertex3f(0.0, CUT, -CUT);
			break;
		case 2:
			glVertex3f(0.0, -CUT, CUT);
			glVertex3f(0.0, -STICKERSHORT, CUT);
			glVertex3f(0.0, -STICKERSHORT, -CUT);
			glVertex3f(0.0, -CUT, -CUT);
			break;
		case 3:
			glVertex3f(0.0, CUT, CUT);
			glVertex3f(0.0, CUT, STICKERSHORT);
			glVertex3f(0.0, -CUT, STICKERSHORT);
			glVertex3f(0.0, -CUT, CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 4:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, 0.0, CUT);
			glVertex3f(CUT, 0.0, STICKERSHORT);
			glVertex3f(-CUT, 0.0, STICKERSHORT);
			glVertex3f(-CUT, 0.0, CUT);
			break;
		case 1:
			glVertex3f(CUT, 0.0, -CUT);
			glVertex3f(STICKERSHORT, 0.0, -CUT);
			glVertex3f(STICKERSHORT, 0.0, CUT);
			glVertex3f(CUT, 0.0, CUT);
			break;
		case 2:
			glVertex3f(-CUT, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -STICKERSHORT);
			glVertex3f(CUT, 0.0, -STICKERSHORT);
			glVertex3f(CUT, 0.0, -CUT);
			break;
		case 3:
			glVertex3f(-CUT, 0.0, CUT);
			glVertex3f(-STICKERSHORT, 0.0, CUT);
			glVertex3f(-STICKERSHORT, 0.0, -CUT);
			glVertex3f(-CUT, 0.0, -CUT);
			break;
		default:
			err = 1;
		}
		break;
	case 5:
		switch (rotation) {
		case 0:
			glVertex3f(CUT, -CUT, 0.0);
			glVertex3f(CUT, -STICKERSHORT, 0.0);
			glVertex3f(-CUT, -STICKERSHORT, 0.0);
			glVertex3f(-CUT, -CUT, 0.0);
			break;
		case 1:
			glVertex3f(CUT, CUT, 0.0);
			glVertex3f(STICKERSHORT, CUT, 0.0);
			glVertex3f(STICKERSHORT, -CUT, 0.0);
			glVertex3f(CUT, -CUT, 0.0);
			break;
		case 2:
			glVertex3f(-CUT, CUT, 0.0);
			glVertex3f(-CUT, STICKERSHORT, 0.0);
			glVertex3f(CUT, STICKERSHORT, 0.0);
			glVertex3f(CUT, CUT, 0.0);
			break;
		case 3:
			glVertex3f(-CUT, -CUT, 0.0);
			glVertex3f(-STICKERSHORT, -CUT, 0.0);
			glVertex3f(-STICKERSHORT, CUT, 0.0);
			glVertex3f(-CUT, CUT, 0.0);
			break;
		default:
			err = 1;
		}
		break;
	}
	glEnd();
	if (err == 1) {
		intCat(&buf1, "draw_orient_line: face ", face);
		stringCat(&buf2, buf1, ", rotation ");
		free(buf1);
		intCat(&buf1, buf2, rotation);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
}

static void
DrawCubit(DinoGLWidget w,
	   int back, int front, int left, int right, int bottom, int top)
{
	Boolean mono = w->dino.mono;

	if (back != NO_FACE) { /* Orange */
		glPushMatrix();
		pickColor(w, back, mono);
		glNormal3f(0.0, 0.0, -1.0);
		if (top != NO_FACE) {
			glTranslatef(-CUBELEN, -CUBEROUND, -STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, OFFSET, 0.0);
			glVertex3f(-STICKERFULLSHORT, STICKERFULLLONG, 0.0);
			glVertex3f(STICKERFULLSHORT, STICKERFULLLONG, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BACK_FACE,
					w->dino.cubeLoc[BACK_FACE][0].rotation,
					0.0, CUBEROUND - 2.0 * CUBELEN,
					-CUTDEPTH + STICKERDEPTH);
			}
		}
		if (left != NO_FACE) {
			glTranslatef(CUBEROUND, CUBELEN, -STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(-OFFSET, 0.0, 0.0);
			glVertex3f(-STICKERFULLLONG, -STICKERFULLSHORT, 0.0);
			glVertex3f(-STICKERFULLLONG, STICKERFULLSHORT, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BACK_FACE,
					w->dino.cubeLoc[BACK_FACE][1].rotation,
					-CUBEROUND + 2.0 * CUBELEN, 0.0,
					-CUTDEPTH + STICKERDEPTH);
			}
		}
		if (bottom != NO_FACE) {
			glTranslatef(CUBELEN, CUBEROUND, -STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, -OFFSET, 0.0);
			glVertex3f(STICKERFULLSHORT, -STICKERFULLLONG, 0.0);
			glVertex3f(-STICKERFULLSHORT, -STICKERFULLLONG, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BACK_FACE,
					w->dino.cubeLoc[BACK_FACE][2].rotation,
					0.0, -CUBEROUND + 2.0 * CUBELEN,
					-CUTDEPTH + STICKERDEPTH);
			}
		}
		if (right != NO_FACE) {
			glTranslatef(-CUBEROUND, -CUBELEN, -STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(OFFSET, 0.0, 0.0);
			glVertex3f(STICKERFULLLONG, STICKERFULLSHORT, 0.0);
			glVertex3f(STICKERFULLLONG, -STICKERFULLSHORT, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BACK_FACE,
					w->dino.cubeLoc[BACK_FACE][3].rotation,
					CUBEROUND - 2.0 * CUBELEN, 0.0,
					-CUTDEPTH + STICKERDEPTH);
			}
		}
		glPopMatrix();
	}
	if (front != NO_FACE) { /* Yellow */
		glPushMatrix();
		pickColor(w, front, mono);
		glNormal3f(0.0, 0.0, 1.0);
		if (top != NO_FACE) {
			glTranslatef(CUBELEN, -CUBEROUND, STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, OFFSET, 0.0);
			glVertex3f(STICKERFULLSHORT, STICKERFULLLONG, 0.0);
			glVertex3f(-STICKERFULLSHORT, STICKERFULLLONG, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, FRONT_FACE,
					w->dino.cubeLoc[FRONT_FACE][2].rotation,
					0.0, CUBEROUND - 2.0 * CUBELEN,
					CUTDEPTH - STICKERDEPTH);
			}
		}
		if (left != NO_FACE) {
			glTranslatef(CUBEROUND, -CUBELEN, STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(-OFFSET, 0.0, 0.0);
			glVertex3f(-STICKERFULLLONG, STICKERFULLSHORT, 0.0);
			glVertex3f(-STICKERFULLLONG, -STICKERFULLSHORT, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, FRONT_FACE,
					w->dino.cubeLoc[FRONT_FACE][1].rotation,
					-CUBEROUND + 2.0 * CUBELEN, 0.0,
					CUTDEPTH - STICKERDEPTH);
			}
		}
		if (bottom != NO_FACE) {
			glTranslatef(-CUBELEN, CUBEROUND, STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, -OFFSET, 0.0);
			glVertex3f(-STICKERFULLSHORT, -STICKERFULLLONG, 0.0);
			glVertex3f(STICKERFULLSHORT, -STICKERFULLLONG, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, FRONT_FACE,
					w->dino.cubeLoc[FRONT_FACE][0].rotation,
					0.0, -CUBEROUND + 2.0 * CUBELEN,
					CUTDEPTH - STICKERDEPTH);
			}
		}
		if (right != NO_FACE) {
			glTranslatef(-CUBEROUND, CUBELEN, STICKERDEPTH);
			glBegin(GL_TRIANGLES);
			glVertex3f(OFFSET, 0.0, 0.0);
			glVertex3f(STICKERFULLLONG, -STICKERFULLSHORT, 0.0);
			glVertex3f(STICKERFULLLONG, STICKERFULLSHORT, 0.0);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, FRONT_FACE,
					w->dino.cubeLoc[FRONT_FACE][3].rotation,
					CUBEROUND - 2.0 * CUBELEN, 0.0,
					CUTDEPTH - STICKERDEPTH);
			}
		}
		glPopMatrix();
	}
	if (left != NO_FACE) { /* Blue */
		glPushMatrix();
		pickColor(w, left, mono);
		glNormal3f(-1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glTranslatef(-STICKERDEPTH, -CUBELEN, -CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, OFFSET);
			glVertex3f(0.0, -STICKERFULLSHORT, STICKERFULLLONG);
			glVertex3f(0.0, STICKERFULLSHORT, STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, LEFT_FACE,
					w->dino.cubeLoc[LEFT_FACE][3].rotation,
					-CUTDEPTH + STICKERDEPTH,
					0.0,
					CUBEROUND - 2.0 * CUBELEN);
			}
		}
		if (top != NO_FACE) {
			glTranslatef(-STICKERDEPTH, -CUBEROUND, CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, OFFSET, 0.0);
			glVertex3f(0.0, STICKERFULLLONG, STICKERFULLSHORT);
			glVertex3f(0.0, STICKERFULLLONG, -STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, LEFT_FACE,
					w->dino.cubeLoc[LEFT_FACE][0].rotation,
					-CUTDEPTH + STICKERDEPTH,
					CUBEROUND,
					0.0);
			}
		}
		if (back != NO_FACE) {
			glTranslatef(-STICKERDEPTH, CUBELEN, CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, -OFFSET);
			glVertex3f(0.0, STICKERFULLSHORT, -STICKERFULLLONG);
			glVertex3f(0.0, -STICKERFULLSHORT, -STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, LEFT_FACE,
					w->dino.cubeLoc[LEFT_FACE][1].rotation,
					-CUTDEPTH + STICKERDEPTH,
					0.0,
					-CUBEROUND + 2.0 * CUBELEN);
			}
		}
		if (bottom != NO_FACE) {
			glTranslatef(-STICKERDEPTH, CUBEROUND, -CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, -OFFSET, 0.0);
			glVertex3f(0.0, -STICKERFULLLONG, -STICKERFULLSHORT);
			glVertex3f(0.0, -STICKERFULLLONG, STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, LEFT_FACE,
					w->dino.cubeLoc[LEFT_FACE][2].rotation,
					-CUTDEPTH + STICKERDEPTH,
					-CUBEROUND,
					0.0);
			}
		}
		glPopMatrix();
	}
	if (right != NO_FACE) { /* Green */
		glPushMatrix();
		pickColor(w, right, mono);
		glNormal3f(1.0, 0.0, 0.0);
		if (front != NO_FACE) {
			glTranslatef(STICKERDEPTH, CUBELEN, -CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, OFFSET);
			glVertex3f(0.0, STICKERFULLSHORT, STICKERFULLLONG);
			glVertex3f(0.0, -STICKERFULLSHORT, STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, RIGHT_FACE,
					w->dino.cubeLoc[RIGHT_FACE][1].rotation,
					CUTDEPTH - STICKERDEPTH,
					CUBEROUND - CUBELEN,
					CUBEROUND - 2.0 * CUBELEN);
			}
		}
		if (top != NO_FACE) {
			glTranslatef(STICKERDEPTH, -CUBEROUND, -CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, OFFSET, 0.0);
			glVertex3f(0.0, STICKERFULLLONG, -STICKERFULLSHORT);
			glVertex3f(0.0, STICKERFULLLONG, STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, RIGHT_FACE,
					w->dino.cubeLoc[RIGHT_FACE][0].rotation,
					CUTDEPTH - STICKERDEPTH,
					CUBELEN,
					-CUBELEN + CUBEROUND);
			}
		}
		if (back != NO_FACE) {
			glTranslatef(STICKERDEPTH, -CUBELEN, CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, -OFFSET);
			glVertex3f(0.0, -STICKERFULLSHORT, -STICKERFULLLONG);
			glVertex3f(0.0, STICKERFULLSHORT, -STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, RIGHT_FACE,
					w->dino.cubeLoc[RIGHT_FACE][3].rotation,
					CUTDEPTH - STICKERDEPTH,
					0.0, -CUBEROUND + 2.0 * CUBELEN);
			}
		}
		if (bottom != NO_FACE) {
			glTranslatef(STICKERDEPTH, CUBEROUND, CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, -OFFSET, 0.0);
			glVertex3f(0.0, -STICKERFULLLONG, STICKERFULLSHORT);
			glVertex3f(0.0, -STICKERFULLLONG, -STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, RIGHT_FACE,
					w->dino.cubeLoc[RIGHT_FACE][2].rotation,
					CUTDEPTH - STICKERDEPTH,
					-CUBELEN, 0.0);
			}
		}
		glPopMatrix();
	}
	if (bottom != NO_FACE) { /* White */
		glPushMatrix();
		pickColor(w, bottom, mono);
		glNormal3f(0.0, -1.0, 0.0);
		if (left != NO_FACE) {
			glTranslatef(CUBEROUND, -STICKERDEPTH, -CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(-OFFSET, 0.0, 0.0);
			glVertex3f(-STICKERFULLLONG, 0.0, STICKERFULLSHORT);
			glVertex3f(-STICKERFULLLONG, 0.0, -STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->dino.cubeLoc[BOTTOM_FACE][3].rotation,
					-CUBEROUND,
					-CUTDEPTH + STICKERDEPTH, 0.0);
			}
		}
		if (front != NO_FACE) {
			glTranslatef(-CUBELEN, -STICKERDEPTH, -CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, OFFSET);
			glVertex3f(STICKERFULLSHORT, 0.0, STICKERFULLLONG);
			glVertex3f(-STICKERFULLSHORT, 0.0, STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->dino.cubeLoc[BOTTOM_FACE][0].rotation,
					0.0, -CUTDEPTH + STICKERDEPTH,
					CUBEROUND);
			}
		}
		if (right != NO_FACE) {
			glTranslatef(-CUBEROUND, -STICKERDEPTH, CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(OFFSET, 0.0, 0.0);
			glVertex3f(STICKERFULLLONG, 0.0, -STICKERFULLSHORT);
			glVertex3f(STICKERFULLLONG, 0.0, STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->dino.cubeLoc[BOTTOM_FACE][1].rotation,
					CUBEROUND,
					-CUTDEPTH + STICKERDEPTH, 0.0);
			}
		}
		if (back != NO_FACE) {
			glTranslatef(CUBELEN, -STICKERDEPTH, CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, -OFFSET);
			glVertex3f(-STICKERFULLSHORT, 0.0, -STICKERFULLLONG);
			glVertex3f(STICKERFULLSHORT, 0.0, -STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, BOTTOM_FACE,
					w->dino.cubeLoc[BOTTOM_FACE][2].rotation,
					0.0, -CUTDEPTH + STICKERDEPTH,
					-CUBEROUND);
			}
		}
		glPopMatrix();
	}
	if (top != NO_FACE) { /* Red */
		glPushMatrix();
		pickColor(w, top, mono);
		glNormal3f(0.0, 1.0, 0.0);
		if (left != NO_FACE) {
			glTranslatef(CUBEROUND, STICKERDEPTH, CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(-OFFSET, 0.0, 0.0);
			glVertex3f(-STICKERFULLLONG, 0.0, -STICKERFULLSHORT);
			glVertex3f(-STICKERFULLLONG, 0.0, STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, TOP_FACE,
					w->dino.cubeLoc[TOP_FACE][3].rotation,
					-CUBELEN,
					CUTDEPTH - STICKERDEPTH, 0.0);
			}
		}
		if (front != NO_FACE) {
			glTranslatef(CUBELEN, STICKERDEPTH, -CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, OFFSET);
			glVertex3f(-STICKERFULLSHORT, 0.0, STICKERFULLLONG);
			glVertex3f(STICKERFULLSHORT, 0.0, STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, TOP_FACE,
					w->dino.cubeLoc[TOP_FACE][2].rotation,
					0.0, CUTDEPTH - STICKERDEPTH,
					CUBELEN);
			}
		}
		if (right != NO_FACE) {
			glTranslatef(-CUBEROUND, STICKERDEPTH, -CUBELEN);
			glBegin(GL_TRIANGLES);
			glVertex3f(OFFSET, 0.0, 0.0);
			glVertex3f(STICKERFULLLONG, 0.0, STICKERFULLSHORT);
			glVertex3f(STICKERFULLLONG, 0.0, -STICKERFULLSHORT);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, TOP_FACE,
					w->dino.cubeLoc[TOP_FACE][1].rotation,
					CUBELEN,
					CUTDEPTH - STICKERDEPTH, 0.0);
			}
		}
		if (back != NO_FACE) {
			glTranslatef(-CUBELEN, STICKERDEPTH, CUBEROUND);
			glBegin(GL_TRIANGLES);
			glVertex3f(0.0, 0.0, -OFFSET);
			glVertex3f(STICKERFULLSHORT, 0.0, -STICKERFULLLONG);
			glVertex3f(-STICKERFULLSHORT, 0.0, -STICKERFULLLONG);
			glEnd();
			if (w->dino.orient) {
				DrawOrientLine(w, TOP_FACE,
					w->dino.cubeLoc[TOP_FACE][0].rotation,
					0.0, CUTDEPTH - STICKERDEPTH,
					-CUBELEN);
			}
		}
		glPopMatrix();
	}
}

static Boolean
DrawCube(DinoGLWidget w)
{
#define S1 1
#define DRAW_STICKERLESS_CUBIT(s) if (!draw_stickerless_cubit(s)) return False
	DinoSlice slice;
	GLfloat rotateStep;
	Boolean s;

	/*if (w->dinoGL.movement.face == NO_FACE) */ {
		slice.face = NO_FACE;
		slice.rotation = NO_ROTATION;
#ifdef FIXME
	} else {
		convertMove(w, w->dinoGL.movement, &slice);
#endif
	}
	rotateStep = (slice.rotation == CCW) ? w->dinoGL.rotateStep :
		-w->dinoGL.rotateStep;
	s = (w->dino.mode != PERIOD3);

/*-
 * The glRotatef() routine transforms the coordinate system for every future
 * vertex specification (this is not so simple, but by now comprehending this
 * is sufficient). So if you want to rotate the inner slice, you can draw
 * one slice, rotate the anglestep for the centerslice, draw the inner slice,
 * rotate reversely and draw the other slice.
 * There is a sequence for drawing cubies for each axis being moved...
 */
	switch (slice.face) {
	case NO_FACE:
	case TOP_FACE:	/* BOTTOM_FACE too */
		glPushMatrix();
		glRotatef(rotateStep, 0, 1, 0);

		glTranslatef(-0.5, -0.5, -0.5);
		glTranslatef(0, 0, S1);
		glPushMatrix();
		glRotatef(180.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT(s); /* White Blue */
		glPopMatrix();
		DrawCubit(w, NO_FACE, NO_FACE,
			w->dino.cubeLoc[LEFT_FACE][2].face,
			NO_FACE,
			w->dino.cubeLoc[BOTTOM_FACE][3].face,
			NO_FACE); /* LB */
		glTranslatef(0, 0, -S1);
		glPushMatrix();
		glRotatef(90.0, 0, -1, 0);
		glRotatef(180.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT(s); /* White Orange */
		glPopMatrix();
		DrawCubit(w, w->dino.cubeLoc[BACK_FACE][0].face,
			NO_FACE, NO_FACE, NO_FACE,
			w->dino.cubeLoc[BOTTOM_FACE][2].face,
			NO_FACE); /* BB */
		glTranslatef(S1, 0, S1);
		glPushMatrix();
		glRotatef(90.0, 0, 1, 0);
		glRotatef(180.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT(s); /* White Yellow */
		glPopMatrix();
		DrawCubit(w, NO_FACE,
			w->dino.cubeLoc[FRONT_FACE][2].face,
			NO_FACE, NO_FACE,
			w->dino.cubeLoc[BOTTOM_FACE][0].face,
			NO_FACE); /* FB */
		glTranslatef(0, 0, -S1);
		glPushMatrix();
		glRotatef(180.0, 0, 1, 0);
		glRotatef(180.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT(s); /* White Green */
		glPopMatrix();
		DrawCubit(w, NO_FACE, NO_FACE, NO_FACE,
			w->dino.cubeLoc[RIGHT_FACE][2].face,
			w->dino.cubeLoc[BOTTOM_FACE][1].face,
			NO_FACE); /* RB */


		glTranslatef(-S1, 0, 0);
		glPushMatrix();
		glRotatef(90.0, 1, 0, 0);
		glRotatef(180.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT(s); /* Blue Orange */
		glPopMatrix();
		DrawCubit(w, w->dino.cubeLoc[BACK_FACE][3].face,
			NO_FACE,
			w->dino.cubeLoc[LEFT_FACE][3].face,
			NO_FACE, NO_FACE, NO_FACE); /* BL */
		glTranslatef(0, S1, S1);
		glPushMatrix();
		glRotatef(90.0, -1, 0, 0);
		glRotatef(180.0, 0, 0, 1);
		DRAW_STICKERLESS_CUBIT(s); /* Blue Yellow */
		glPopMatrix();
		DrawCubit(w, NO_FACE,
			w->dino.cubeLoc[FRONT_FACE][3].face,
			w->dino.cubeLoc[LEFT_FACE][1].face,
			NO_FACE, NO_FACE, NO_FACE); /* FL */
		glTranslatef(S1, 0, -S1);
		glPushMatrix();
		glRotatef(90.0, -1, 0, 0);
		DRAW_STICKERLESS_CUBIT(s); /* Green Orange */
		glPopMatrix();
		DrawCubit(w, w->dino.cubeLoc[BACK_FACE][1].face,
			NO_FACE, NO_FACE,
			w->dino.cubeLoc[RIGHT_FACE][1].face,
			NO_FACE, NO_FACE); /* BR */
		glTranslatef(0, -S1, S1);
		glPushMatrix();
		glRotatef(90.0, 1, 0, 0);
		DRAW_STICKERLESS_CUBIT(s); /* Green Yellow */
		glPopMatrix();
		DrawCubit(w, NO_FACE,
			w->dino.cubeLoc[FRONT_FACE][1].face,
			NO_FACE,
			w->dino.cubeLoc[RIGHT_FACE][3].face,
			NO_FACE, NO_FACE); /* FR */


		glTranslatef(-S1, S1, -S1);
		glPushMatrix();
		glRotatef(180.0, 0, 1, 0);
		DRAW_STICKERLESS_CUBIT(s); /* Red Blue */
		glPopMatrix();
		DrawCubit(w,
			NO_FACE, NO_FACE,
			w->dino.cubeLoc[LEFT_FACE][0].face,
			NO_FACE, NO_FACE,
			w->dino.cubeLoc[TOP_FACE][3].face); /* LT */
		glTranslatef(S1, 0, 0);
		glPushMatrix();
		glRotatef(90.0, 0, 1, 0);
		DRAW_STICKERLESS_CUBIT(s); /* Red Orange */
		glPopMatrix();
		DrawCubit(w,
			w->dino.cubeLoc[BACK_FACE][2].face,
			NO_FACE, NO_FACE, NO_FACE, NO_FACE,
			w->dino.cubeLoc[TOP_FACE][0].face); /* BT */
		glTranslatef(-S1, 0, S1);
		glPushMatrix();
		glRotatef(-90.0, 0, 1, 0);
		DRAW_STICKERLESS_CUBIT(s); /* Red Yellow */
		glPopMatrix();
		DrawCubit(w, NO_FACE,
			w->dino.cubeLoc[FRONT_FACE][0].face,
			NO_FACE, NO_FACE, NO_FACE,
			w->dino.cubeLoc[TOP_FACE][2].face); /* FT */
		glTranslatef(S1, 0, 0);
		glPushMatrix();
		DRAW_STICKERLESS_CUBIT(s); /* Red Green */
		glPopMatrix();
		DrawCubit(w, NO_FACE, NO_FACE, NO_FACE,
			w->dino.cubeLoc[RIGHT_FACE][0].face,
			NO_FACE,
			w->dino.cubeLoc[TOP_FACE][1].face); /* RT */
		glPopMatrix();
		break;
	}
	return True;
#undef S1
}

void
DrawAllPiecesGL(DinoGLWidget w)
{
#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("Draw All GL error");
	}
#endif
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPushMatrix();
	glTranslatef(0.0, 0.0, -10.0);
	if (w->core.height > w->core.width)
		glScalef(SCALE, SCALE * w->core.width / w->core.height, SCALE);
	else
		glScalef(SCALE * w->core.height / w->core.width, SCALE, SCALE);
	switch (w->dino.view) {
	case 0: /* >-  vertical */
		glRotatef(w->dinoGL.step / 2, 0, 1, 0);
		glRotatef(w->dinoGL.step / 2, 1, 0, 0);
		break;
	case 1: /* Y  !vertical */
		glRotatef(w->dinoGL.step / 2, 1, 0, 0);
		glRotatef(w->dinoGL.step / 2, 0, 1, 0);
		break;
	case 2: /* -<  vertical */
		glRotatef(2 * w->dinoGL.step, 0, 0, 1);
		glRotatef(5 * w->dinoGL.step / 2, 0, 1, 0);
		glRotatef(w->dinoGL.step / 2, 1, 0, 0);
		break;
	case 3: /* ^  !vertical */
		glRotatef(5 * w->dinoGL.step / 2, 0, 1, 0);
		glRotatef(w->dinoGL.step / 2, 1, 0, 1);
		break;
	default:
		break;
	}
	if (!DrawCube(w)) {
		DISPLAY_WARNING("DrawCube error");
	}
	glPopMatrix();
	glFlush();
#ifdef WINVER
	SwapBuffers(w->core.hDC);
#else
	glXSwapBuffers(XtDisplay(w), XtWindow(w));
#endif
}

void
DrawFrameGL(DinoGLWidget w, Boolean focus)
{
	/* UNUSED */
}

Boolean
SelectPiecesGL(DinoGLWidget w, int x, int y, int *face, int *position)
{
	/* UNUSED */
	return True;
}

#ifdef WINVER
Boolean setupPixelFormat(DinoGLWidget w, BYTE type, DWORD flags)
{
	PIXELFORMATDESCRIPTOR pfd;
	int pixelFormat;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | flags;
	pfd.dwLayerMask = PFD_MAIN_PLANE;
	/*pfd.cColorBits = 8;
	pfd.cDepthBits = 16;*/
	pfd.iPixelType = type;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;

	if ((pixelFormat = ChoosePixelFormat(w->core.hDC, &pfd)) == 0) {
		DWORD err;
		char *buf1;

		err = GetLastError();
		/* 6 ERROR_INVALID_HANDLE */
		intCat(&buf1, "ChoosePixelFormat failed ", err);
		DISPLAY_WARNING(buf1);
		free(buf1);
		return FALSE;
	}

	if (SetPixelFormat(w->core.hDC, pixelFormat, &pfd) == FALSE) {
		MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
		return FALSE;
	}
	DescribePixelFormat(w->core.hDC, pixelFormat,
		sizeof(PIXELFORMATDESCRIPTOR), &pfd);
	return TRUE;
}

#else

static Boolean
SetValuesPuzzleGL(Widget current, Widget request, Widget renew)
{
	DinoGLWidget c = (DinoGLWidget) current, w = (DinoGLWidget) renew;
	Boolean redraw = False;

	if (w->dino.view != c->dino.view) {
		ResizePuzzleGL(w);
		redraw = True;
	}
#if 0
	if (w->dinoGL.cubeSizex.x != c->dinoGL.cubeSizex.x ||
	    w->dinoGL.cubeSizey.x != c->dinoGL.cubeSizey.x ||
	    w->dinoGL.cubeSizez.x != c->dinoGL.cubeSizez.x ||
	    w->dinoGL.cubeSizex.y != c->dinoGL.cubeSizex.y ||
	    w->dinoGL.cubeSizey.y != c->dinoGL.cubeSizey.y ||
	    w->dinoGL.cubeSizez.y != c->dinoGL.cubeSizez.y) {
		ResizePuzzleGL(w);
		redraw = True;
	}
#endif
	return (redraw);
}

static GLXContext *
initGL(DinoGLWidget w) {
	XVisualInfo xviIn, *xviOut;
	int numVis;

	/*XGetWindowAttributes(XtDisplay(w), XtWindow(w), &xgwa); */
	xviIn.screen = DefaultScreen(XtDisplay(w));
	xviIn.visualid = XVisualIDFromVisual(DefaultVisual(XtDisplay(w),
		xviIn.screen));
	xviOut = XGetVisualInfo(XtDisplay(w), VisualScreenMask | VisualIDMask,
		&xviIn, &numVis);
	if (!xviOut) {
		XtWarning("Could not get XVisualInfo");
		return (GLXContext *) NULL;
	}
	if (glXContext)
		free(glXContext);
	if ((glXContext = (GLXContext *) malloc(sizeof (GLXContext))) ==
			NULL) {
		DISPLAY_ERROR("Not enough memory for glx info, exiting.");
	}
	*glXContext = glXCreateContext(XtDisplay(w), xviOut, 0, GL_TRUE);
	(void) XFree((char *) xviOut);
	if (!*glXContext) {
		XtWarning("Could not create rendering context");
		return (GLXContext *) NULL;
	}
	return glXContext;
}
#endif

#ifndef WINVER
static
#endif
void
ResizePuzzleGL(DinoGLWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	glViewport(0, 0, (GLint) w->core.width, (GLint) w->core.height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-1.0, 1.0, -1.0, 1.0, 5.0, 15.0);
	glMatrixMode(GL_MODELVIEW);
}

static void
makeCurrentGL(DinoGLWidget w)
{
	GLboolean   rgbaMode;

#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("GL error");
	}
#endif
	madeCurrent = True;
	/* True Color junk */
	glGetBooleanv(GL_RGBA_MODE, &rgbaMode);
	if (!rgbaMode) {
#ifdef WINVER
		glClearIndex(0.0);
#else
		glIndexi(WhitePixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
		glClearIndex((float) BlackPixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
#endif
	}
	ResizePuzzleGL(w);

	glDrawBuffer(GL_BACK);
	glClearDepth(1.0);
	glClearColor(0.0, 0.0, 0.0, 1.0);
	glColor3f(1.0, 1.0, 1.0);

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, position0);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, position1);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lmodel_twoside);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_NORMALIZE);
	glEnable(GL_CULL_FACE);
	glShadeModel(GL_FLAT);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, front_shininess);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, front_specular);

	w->dinoGL.angleStep = 90.0;
	w->dinoGL.step = 90;
#if 0
	w->dinoGL.vx = 0.05;
	if (LRAND() & 1)
		w->dinoGL.vx *= -1.0;
	w->dinoGL.vy = 0.05;
	if (LRAND() & 1)
		w->dinoGL.vy *= -1.0;
#endif
	w->dinoGL.rotateStep = 0.0;
}

#ifndef WINVER
static void
InitializePuzzleGL(Widget request, Widget renew)
{
	DinoGLWidget w = (DinoGLWidget) renew;

	SetAllColors((DinoWidget) w);
	(void) initGL(w);
	w->dino.dim = 4; /* 2 and 3 already taken */
	ResizePuzzleGL(w);
}
#endif

#ifndef WINVER
static
#endif
void
ExposePuzzleGL(
#ifdef WINVER 
DinoGLWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifdef WINVER 
	if (hRC == NULL) {
#if 1
		BYTE type = PFD_TYPE_RGBA;
#else
		BYTE type = PFD_TYPE_COLORINDEX;
#endif
		DWORD flags = PFD_DOUBLEBUFFER;
		(void) setupPixelFormat(w, type, flags);
		hRC = wglCreateContext(w->core.hDC);
	}
#else
	DinoGLWidget w = (DinoGLWidget) renew;

	if (!w->core.visible)
		return;
#endif
	if (!madeCurrent) {
		makeCurrentGL(w);
	}
	DrawFrameGL(w, w->dino.focus);
	DrawAllPiecesGL(w);
}

#ifndef WINVER
static void
MovePuzzleGLTl(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		TL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLTop(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		TOP,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLTr(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		TR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLLeft(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLRight(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLBl(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		BL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLBottom(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzleGLBr(DinoGLWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y,
		BR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}
#endif
#endif
