#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <gtk/gtk.h>
#include <glib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/****************************************************************************/

static int time_parse(const char *time_text) {
	int time = 0, slice;
	size_t len;
	gboolean was_dot = FALSE;
	char *duptext, *temp, sep = ':';

	duptext = g_strdup(time_text);

	temp = duptext;
	while (sep) {
		len = strcspn(temp, ":.");

		if (sep == ':') { was_dot = FALSE; time *= 60; }
		else if (sep == '.') { was_dot = TRUE; time *= 1000; }

		sep = temp[len];
		temp[len] = '\0';

		slice = strtol(temp, NULL, 10);
		time += slice;

		temp += len + 1;
	}

	if (!was_dot) time *= 1000;
	time *= 32;

	g_free(duptext);

	return time;
}

static void time_snprint(char *buffer, size_t bufflen, int time) {
	snprintf(buffer, bufflen, "%d:%02d:%02d.%03d",
		time / 32000 / 60 / 60,
		time / 32000 / 60 % 60,
		time / 32000 % 60,
		time / 32 % 1000
	);
}

static void time_insert(
	GtkEditable *w,
	const gchar *text,
	gint length,
	gint *position,
	gpointer data
) {
	int i;

	for (i = 0; i < length; i++) {
		if (text[i] != ':' && text[i] != '.' && !isdigit(text[i])) {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(w), "insert-text");
			return;
		}
	}
}

static void time_changed(GtkEntry *w, gpointer timep) {
	int *time = timep;
	char entry_text[64];

	*time = time_parse(gtk_entry_get_text(w));
	time_snprint(entry_text, sizeof(entry_text), *time);

	gtk_signal_handler_block_by_func(
		GTK_OBJECT(w),
		GTK_SIGNAL_FUNC(time_changed),
		timep
	);

	gtk_entry_set_text(w, entry_text);

	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(w),
		GTK_SIGNAL_FUNC(time_changed),
		timep
	);
}

static void time_changed_focus(GtkEntry *w, GdkEventFocus *event, gpointer timep) {
	time_changed(w, timep);
}

GtkWidget *util_widget_entry_time(int *time) {
	GtkWidget *widget;
	char entry_text[64];

	widget = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(widget), 20);
	gtk_widget_set_usize(widget, 120, -1);
	time_snprint(entry_text, sizeof(entry_text), *time);
	gtk_entry_set_text(GTK_ENTRY(widget), entry_text);
	gtk_signal_connect(GTK_OBJECT(widget), "activate", GTK_SIGNAL_FUNC(time_changed), time);
	gtk_signal_connect(GTK_OBJECT(widget), "focus-out-event", GTK_SIGNAL_FUNC(time_changed_focus), time);
	gtk_signal_connect(GTK_OBJECT(widget), "insert-text", GTK_SIGNAL_FUNC(time_insert), time);

	return widget;
}

/****************************************************************************/

static void number_insert(
	GtkEditable *w,
	const gchar *text,
	gint length,
	gint *position,
	gpointer *data
) {
	int i;

	for (i = 0; i < length; i++) {
		if (!isdigit(text[i])) {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(w), "insert-text");
			return;
		}
	}
}

static void number_changed(GtkEntry *w, gpointer nump) {
	int *num = nump;
	char entry_text[64];

	*num = strtol(gtk_entry_get_text(w), NULL, 10);
	if (*num < 0) *num = 0;

	snprintf(entry_text, sizeof(entry_text), "%d", *num);

	gtk_signal_handler_block_by_func(
		GTK_OBJECT(w),
		GTK_SIGNAL_FUNC(number_changed),
		nump
	);

	gtk_entry_set_text(w, entry_text);

	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(w),
		GTK_SIGNAL_FUNC(number_changed),
		nump
	);
}

GtkWidget *util_widget_entry_number(int *number, int min, int max, int sizex) {
	GtkWidget *widget;
	char entry_text[64];

	widget = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(widget), 10);
	snprintf(entry_text, sizeof(entry_text), "%d", *number);
	gtk_entry_set_text(GTK_ENTRY(widget), entry_text);
	gtk_signal_connect(GTK_OBJECT(widget), "changed", GTK_SIGNAL_FUNC(number_changed), number);
	gtk_signal_connect(GTK_OBJECT(widget), "insert-text", GTK_SIGNAL_FUNC(number_insert), number);
	gtk_widget_set_usize(widget, sizex, -1);
	
	return widget;
}

/****************************************************************************/

static void string_changed(GtkEditable *w, gpointer strp) {
	char **str = strp;
	if (*str) g_free(*str);
	*str = gtk_editable_get_chars(w, 0, -1);
}

GtkWidget *util_widget_entry_string(char **str) {
	GtkWidget *widget;

	widget = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(widget), *str?*str:"");
	gtk_signal_connect(GTK_OBJECT(widget), "changed", GTK_SIGNAL_FUNC(string_changed), str);

	return widget;
}

/****************************************************************************/

GtkWidget *util_widget_label(char *label, GtkWidget *child) {
	GtkWidget *hbox, *widget;

	hbox = gtk_hbox_new(FALSE, 5);
	widget = gtk_label_new(label);
	gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), child, TRUE, TRUE, 0);

	return hbox;
}
