/*
 *  XMMS Crossfade Plugin
 *  Copyright (C) 2000-2007  Peter Eisenlohr <peter@eisenlohr.org>
 *
 *  based on the original OSS Output Plugin
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 *  USA.
 */

/*
 *  Volume to standard (16bit-le stereo)
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include "volume.h"
#include <string.h>
#include <math.h>		/* for rintf */

#undef VERBOSE

static void
set_factor(volume_context_t *vc, gfloat factor)
{
#if 0
	if (factor < 0.01f)
		factor = 0.01f;
	if (factor > 4.0f)
		factor = 4.0f;
#endif
	vc->factor_l = factor;
	vc->factor_r = factor;
#if defined(VERBOSE)
	DEBUG(("[crossfade] volume_set_factor: new factor=%.3f\n", factor));
#endif
}

void volume_set(volume_context_t *vc, gint l, gint r)
{
#if 1
	vc->factor_l = volume_compute_factor(l, 50);
	vc->factor_r = volume_compute_factor(r, 50);
#else
	vc->factor_l = (gfloat)l / 100;
	vc->factor_r = (gfloat)r / 100;
#endif
#if defined(VERBOSE)
	DEBUG(("[crossfade] volume_set: left=%.3f right=%.3f\n",
	       vc->factor_l, vc->factor_r));
#endif
}

void
volume_init(volume_context_t *vc)
{
	memset(vc, 0, sizeof(*vc));
	vc->factor_l = 1.0f;
	vc->factor_r = 1.0f;
	vc->active = FALSE;
}

void
volume_set_active(volume_context_t *vc, gboolean active)
{
	vc->active = active;
#if defined(VERBOSE)
	DEBUG(("[crossfade] volume_set_active: active=%d\n", vc->active));
#endif
}

void
volume_set_target_rms(volume_context_t *vc, gint target_rms)
{
	vc->target_rms = target_rms;
	if (vc->active && (vc->song_rms == 0))
	{
		DEBUG(("[crossfade] volume_set_target_rms: WARNING: song_rms=0!\n"));
		vc->factor_l = 1.0f;
		vc->factor_r = 1.0f;
		return;
	}
	set_factor(vc, (gfloat) vc->target_rms / (gfloat) vc->song_rms);
}

void
volume_set_song_rms(volume_context_t *vc, gint song_rms)
{
	vc->song_rms = song_rms;
	set_factor(vc, (gfloat) vc->target_rms / (gfloat) vc->song_rms);
}

gfloat
volume_compute_factor(gint percent, gint dB_range)
{
	gfloat dB;

	if (percent >= 100)
		return 1;
	if (percent <= 0)
		return 0;
	dB = (percent - 100) / 100.0 * dB_range;
	return pow(10, dB / 20);
}

void
volume_flow(volume_context_t *vc, gint16 *in, gint length)
{
	struct timeval tv;
	glong dt;
	gint out;

	if (!vc->active)
		return;

	length /= 4;
	while (length--)
	{
		out = (gint) rintf((gfloat) *in * vc->factor_l);
		if      (out >  32767) { out =  32767; vc->clips++; }
		else if (out < -32768) { out = -32768; vc->clips++; }
		*in++ = out;

		out = (gint) rintf((gfloat) *in * vc->factor_r);
		if      (out >  32767) { out =  32767; vc->clips++; }
		else if (out < -32768) { out = -32768; vc->clips++; }
		*in++ = out;
	}

	gettimeofday(&tv, NULL);
	dt = (tv.tv_sec - vc->tv_last.tv_sec)  * 1000
	  + (tv.tv_usec - vc->tv_last.tv_usec) / 1000;

	if (((dt < 0) || (dt > 1000)) && (vc->clips > 0))
	{
		DEBUG(("[crossfade] volume_flow: %d samples clipped!\n", vc->clips));
		vc->clips = 0;
		vc->tv_last = tv;
	}
}

void
volume_free(volume_context_t *vc)
{
}
