/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: xpages.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : image pages routines                                     *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : XMdcPagesSelected()          - Handle selected page      *
 *                XMdcPagesGoTo()              - Go to a specified page    *
 *                XMdcPagesCreateMenu()        - Create pages menu         *
 *                XMdcPagesNext()              - Go to next     page       *
 *                XMdcPagesPrev()              - Go to previous page       *
 *                XMdcPagesSelCallbackApply()  - Apply new page display    *
 *                XMdcPagesSel()               - Select page display       *
 *                XMdcPagesGetNrImages()       - Get images per page       *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: xpages.c,v 1.21 2007/11/09 23:03:50 enlf Exp $
 */

/*
   Copyright (C) 1997-2007 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include <stdio.h>

#include "xmedcon.h"

/****************************************************************************
                              D E F I N E S
****************************************************************************/

static GtkWidget *wpages=NULL;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

void XMdcPagesSelected(GtkWidget *widget, Uint32 *pagenr)
{
  if ( (Uint32)(*pagenr) != my.curpage) {

    my.curpage = (Uint32)(*pagenr);
    MdcDebugPrint("one-based selected page = %u",my.curpage + 1);

    XMdcMainWidgetsInsensitive();

    XMdcRemovePreviousColorMap();
    XMdcRemovePreviousImages();
    XMdcBuildColorMap();
    XMdcBuildCurrentImages();

    my.prevpage = my.curpage;

    XMdcMainWidgetsResensitive();
  }
}

gboolean XMdcPagesGoTo(GtkWidget *spinner, gpointer data) 
{
  GtkWidget *menu;
  GtkWidget *active;
  GtkSpinButton *spin = GTK_SPIN_BUTTON(spinner);
  Uint32 page, newpage;

  page = (Uint32) gtk_spin_button_get_value_as_int(spin); newpage = page - 1;

  if ( page == 0 || page > my.number_of_pages ) {

    XMdcDisplayWarn("Need a number between [1 , %u]",my.number_of_pages);

  }else{

    gtk_option_menu_set_history(GTK_OPTION_MENU(my.pagemenu),newpage);
    menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(my.pagemenu));
    gtk_menu_set_active(GTK_MENU(menu),newpage);
    active = gtk_menu_get_active(GTK_MENU(menu));
    gtk_menu_item_activate(GTK_MENU_ITEM(active));

  }

  return(TRUE);

}

GtkWidget *XMdcPagesCreateMenu(void)
{
  GtkWidget *menu;
  GtkWidget *menuitem;
  GSList *group;
  Uint32 i;

  menu = gtk_menu_new();
  group= NULL;

  for(i=0; i<my.number_of_pages; i++) {

    sprintf(mdcbufr,"Page: %4u/%u",i+1,my.number_of_pages);
    menuitem = gtk_radio_menu_item_new_with_label(group,mdcbufr);
    gtk_signal_connect(GTK_OBJECT (menuitem), "activate",
                        GTK_SIGNAL_FUNC(XMdcPagesSelected),
                         (Uint32 *)&my.pagenumber[i]);
    group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menuitem));
    gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(menuitem),TRUE);
    gtk_menu_append(GTK_MENU(menu),menuitem);
    gtk_widget_show(menuitem);
  }

  return(menu);

}

void XMdcPagesNext(void)
{
  GtkWidget *menu;
  GtkWidget *active;
  Uint32 nextpage;

  nextpage = my.curpage + 1;

  if (nextpage < my.number_of_pages) {
    gtk_option_menu_set_history(GTK_OPTION_MENU(my.pagemenu),nextpage);
    menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(my.pagemenu));
    gtk_menu_set_active(GTK_MENU(menu),nextpage);
    active = gtk_menu_get_active(GTK_MENU(menu));
    gtk_menu_item_activate(GTK_MENU_ITEM(active));
  }
} 

void XMdcPagesPrev(void)
{
  GtkWidget *menu;
  GtkWidget *active;
  Uint32 prevpage;

  if (my.curpage > 0) {
    prevpage = my.curpage - 1;
    gtk_option_menu_set_history(GTK_OPTION_MENU(my.pagemenu),prevpage);
    menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(my.pagemenu));
    gtk_menu_set_active(GTK_MENU(menu),prevpage);
    active = gtk_menu_get_active(GTK_MENU(menu));
    gtk_menu_item_activate(GTK_MENU_ITEM(active));
  } 
}

void XMdcPagesSelCallbackApply(GtkWidget *widget, gpointer data)
{
  Int8 type=XMDC_PAGES_FRAME_BY_FRAME;

  MdcDebugPrint("pages layout: ");
  if (GTK_TOGGLE_BUTTON(sPagesSelection.FrameByFrame)->active)         {
    MdcDebugPrint("\tper frame"); type = XMDC_PAGES_FRAME_BY_FRAME;
  }else if (GTK_TOGGLE_BUTTON(sPagesSelection.SliceBySlice)->active)   {
    MdcDebugPrint("\tper slice"); type = XMDC_PAGES_SLICE_BY_SLICE;
  }else if (GTK_TOGGLE_BUTTON(sPagesSelection.ScreenFull)->active)     {
    MdcDebugPrint("\tscreen full"); type = XMDC_PAGES_SCREEN_FULL;
  }

  if (type != sPagesSelection.CurType) {
    sPagesSelection.CurType = type;
    if (XMDC_FILE_OPEN == MDC_YES) {
      /* prevent useless reprocessing */
      if (my.fi->number == 1) return;

      XMdcProgressBar(MDC_PROGRESS_BEGIN,0.,"Redisplay images:");
      XMdcViewerHide(); 
      XMdcViewerEnableAutoShrink();
      XMdcViewerReset();
      XMdcDisplayImages();
      XMdcProgressBar(MDC_PROGRESS_END,0.,NULL);
    }
  } 
}

void XMdcPagesSel(void)
{

  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *box3;
  GtkWidget *box4;
  GtkWidget *frame;
  GtkWidget *button;
  GtkWidget *separator;
  GSList *group;

  if (wpages == NULL) {

    wpages = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    gtk_signal_connect(GTK_OBJECT(wpages),"destroy",
                       GTK_SIGNAL_FUNC(XMdcMedconQuit),NULL);
    gtk_signal_connect(GTK_OBJECT(wpages),"delete_event",
                       GTK_SIGNAL_FUNC(XMdcHandlerToHide),NULL);

    gtk_window_set_title(GTK_WINDOW(wpages),"Pages Selection");

    gtk_container_set_border_width (GTK_CONTAINER (wpages), 0);

    box1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (wpages), box1);
    gtk_widget_show(box1);

    /* create upper box - Initial Resize */
    box2 = gtk_vbox_new (FALSE, 5);
    gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER(box2), 5);
    gtk_widget_show(box2);
 
    box3 = gtk_hbox_new (FALSE, 5);
    gtk_box_pack_start(GTK_BOX(box2), box3, TRUE, TRUE, 0);
    gtk_widget_show(box3);

    frame = gtk_frame_new("Display Pages");
    gtk_box_pack_start(GTK_BOX (box3), frame, TRUE, TRUE, 0);
    gtk_widget_show(frame);

    box4 = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(frame), box4);
    gtk_container_set_border_width(GTK_CONTAINER(box4), 5);
    gtk_widget_show(box4);
 
    button = gtk_radio_button_new_with_label(NULL, "frame by frame (volume)");
    gtk_box_pack_start(GTK_BOX(box4), button, TRUE, TRUE, 0);
    if (sPagesSelection.CurType == XMDC_PAGES_FRAME_BY_FRAME);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_widget_show(button);
    sPagesSelection.FrameByFrame = button;

    group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
    button = gtk_radio_button_new_with_label(group, "slice by slice (image)");
    gtk_box_pack_start(GTK_BOX(box4), button, TRUE, TRUE, 0);
    if (sPagesSelection.CurType == XMDC_PAGES_SLICE_BY_SLICE)
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_widget_show(button);
    sPagesSelection.SliceBySlice = button;

    group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
    button = gtk_radio_button_new_with_label(group, "whole screen filled");
    gtk_box_pack_start (GTK_BOX(box4), button, TRUE, TRUE, 0);
    if (sPagesSelection.CurType == XMDC_PAGES_SCREEN_FULL)
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_widget_show(button);
    sPagesSelection.ScreenFull = button;

    /* create horizontal separator */
    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, FALSE, 0);
    gtk_widget_show (separator);

    /* create bottom button box */
    box2 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start(GTK_BOX(box1), box2, TRUE, TRUE, 2);
    gtk_widget_show(box2);
 
    button = gtk_button_new_with_label("Apply");
    gtk_box_pack_start(GTK_BOX(box2), button, TRUE, TRUE, 2);
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(gtk_widget_hide), GTK_OBJECT(wpages));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(XMdcPagesSelCallbackApply), NULL);
    gtk_widget_show(button);

    button = gtk_button_new_with_label ("Cancel");
    gtk_box_pack_start(GTK_BOX(box2), button, TRUE, TRUE, 2);
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(gtk_widget_hide),GTK_OBJECT(wpages));
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(XMdcViewerShow), NULL);
    gtk_widget_show(button);
 
  }else{
    /* set buttons to appropriate state */
    GtkWidget *b1, *b2, *b3;
   
    gtk_widget_hide(wpages);
 
    b1 = sPagesSelection.FrameByFrame;
    b2 = sPagesSelection.SliceBySlice;
    b3 = sPagesSelection.ScreenFull;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b1),FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b2),FALSE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b3),FALSE);
    switch (sPagesSelection.CurType) {
      case XMDC_PAGES_FRAME_BY_FRAME :
          gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b1),TRUE); break;
      case XMDC_PAGES_SLICE_BY_SLICE :
          gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b2),TRUE); break;
      case XMDC_PAGES_SCREEN_FULL    :
          gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b3),TRUE); break;
    }

  } 

  XMdcShowWidget(wpages);
}

Uint32 XMdcPagesGetNrImages(void)
{
  Uint32 nr=0;

  switch (sPagesSelection.CurType) {
    case XMDC_PAGES_FRAME_BY_FRAME: nr = my.fi->dim[3];
        break;
    case XMDC_PAGES_SLICE_BY_SLICE: nr = 1;
        break;
    case XMDC_PAGES_SCREEN_FULL   : nr = my.fi->number;
        break;
  }

  return(nr);
}
