/*  xhangglider
 *  Copyright (C) 1999 Yasuhiro Take
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <X11/Xlib.h>

static Display *dpy;
static GC gc, bmapgc;

#include "def.h"

int td_initialize(Display *display, Pixmap clip)
{
  dpy = (Display *)display;
  gc = XCreateGC(dpy, DefaultRootWindow(dpy), 0, 0);

  if (clip) {
    bmapgc = XCreateGC(dpy, clip, 0, 0);
    XSetFunction(dpy, bmapgc, GXset);
  }

  return 0;
}

int td_rotate_polygons(int r, double v_cos, double v_sin, PolygonData *pd,
		       int x_dim, int y_dim)
{
  int i, j, ii;
  double rd;
  float x, y;

  if (r <= 360) {
    rd = (double)r / 180.0 * M_PI;

    v_cos = cos(rd);
    v_sin = sin(rd);
  }    
  
  for (i = 0; i < pd->n_pol; i++) {
    ii = pd->pol[i].type;
    
    for (j = 0; j < ii; j++) {
      x = pd->pol[i].vertix[x_dim][j];
      y = pd->pol[i].vertix[y_dim][j];

      pd->pol[i].vertix[x_dim][j] = v_cos * x - v_sin * y;
      pd->pol[i].vertix[y_dim][j] = v_sin * x + v_cos * y;
    }
  }

  return 0;
}

int td_convert_coord(Polygon *pol, XPoint *points, int n_points, int zoom,
		     int x, int y)
{
  int i;

  for (i = 0; i < n_points; i++) {
    points[i].x = (int)((pol->vertix[0][i] + 0.5) * zoom) + x;
    points[i].y = (int)(((-pol->vertix[2][i]) + 0.5) * zoom) + y;
  }

  points[n_points].x = points[0].x;
  points[n_points].y = points[0].y;
  
  return 0;
}

int td_query_distance_of_each(Polygon *pol, int n_pol, int dim)
{
  float d;
  int cnt = 0;
  int i, ii, j;
  
  for (i = 0; i < n_pol; i++) {
    if ((ii = pol[i].type) == T_UNUSED)
      d = INVALID_VALUE;
    else {
      d = 0.0;
      for (j = 0; j < ii; j++)
	d += pol[i].vertix[dim][j];
      d /= (float)(ii);
      cnt++;
    }
    
    pol[i].distance = d;
  }

  return cnt;
}

static int sort_direction;

static int compare_polygons(void *v1, void *v2)
{
  Polygon *p1, *p2;

  p1 = (Polygon *)v1;
  p2 = (Polygon *)v2;

  if (p1->distance < p2->distance)
    return -(sort_direction);
  if (p1->distance > p2->distance)
    return (sort_direction);
  return 0;
}

int td_sort_polygons(Polygon *pol, int n_pol, int direction)
{
  sort_direction = direction;
  qsort(pol, n_pol, sizeof(Polygon), compare_polygons);
  return 0;
}

int td_draw_polygons(Drawable d, Pixmap clip, PolygonData *pd, int zoom,
		     int x, int y)
{
  int i, r;
  XPoint points[5];
  int n_points;
  Polygon *pol;
  int n_pol;

  pol = pd->pol;
  n_pol = pd->n_pol;
  
  if ((i = td_query_distance_of_each(pol, n_pol, DIM_Y)) == 0)
    return 0;
  
  td_sort_polygons(pol, n_pol, SORT_DIRECTION_MAX_FIRST);

  for (i = 0; i < n_pol; i++) {
    if (pol[i].type != T_UNUSED) {
      n_points = pol[i].type;

      if (pol[i].type == T_BALL)
	n_points = 2;
      
      td_convert_coord(&(pol[i]), points, n_points, zoom, x, y);

      XSetForeground(dpy, gc, pol[i].color.pixel);

      switch (pol[i].type) {
	case T_QUAD:
	case T_TRI:
	  XFillPolygon(dpy, d, gc, points, n_points, Complex, CoordModeOrigin);
	  XDrawLines(dpy, d, gc, points, n_points + 1, CoordModeOrigin);
	  
	  if (clip) {
	    XFillPolygon(dpy, clip, bmapgc, points, n_points, Complex,
			 CoordModeOrigin);
	    XDrawLines(dpy, clip, bmapgc, points, n_points + 1,
		       CoordModeOrigin);
	  }

	  
	  break;
	case T_LINE:
	  XDrawLines(dpy, d, gc, points, n_points, CoordModeOrigin);
	  
	  if (clip)
	    XDrawLines(dpy, clip, bmapgc, points, n_points, CoordModeOrigin);

	  break;
	case T_BALL:
	  r = points[1].x - zoom / 2 - x;

	  XFillArc(dpy, d, gc, points[0].x - r, points[0].y - r, r * 2, r * 2,
		   0, 360*64);
	  if (clip)
	    XFillArc(dpy, clip, bmapgc, points[0].x - r, points[0].y - r,
		     r * 2, r * 2, 0, 360*64);
      }
    }
  }
  
  return 0;
}


    
    
  
  
  
