/*
 *  xfce4-volstatus-icon
 *
 *  Copyright (c) 2006 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <ghal/ghal.h>

#include <libxfcegui4/libxfcegui4.h>

#include "xfce-volstatus-common.h"
#include "xfce-volstatus-dialog.h"

#define BORDER 8

struct _XfceVolstatusDialog
{
    XfceTitledDialog parent;
    
    XfceVolstatusIcon *icon;
    
    GList *drives;
    
    GtkTreeStore *ts;
    GtkWidget *treeview;
    
    GtkWidget *remove_btn;
    GtkWidget *name_lbl;
    GtkWidget *kind_lbl;
    GtkWidget *mntpt_lbl;
};

typedef struct _XfceVolstatusDialogClass
{
    XfceTitledDialogClass parent;
} XfceVolstatusDialogClass;

enum
{
    COL_PIXBUF = 0,
    COL_NAME,
    COL_GHAL_DEVICE,
};


static void xfce_volstatus_dialog_class_init(XfceVolstatusDialogClass *klass);

static void xfce_volstatus_dialog_init(XfceVolstatusDialog *dialog);
static void xfce_volstatus_dialog_finalize(GObject *obj);

static void xfce_volstatus_dialog_drive_added(XfceVolstatusIcon *icon,
                                              GHalDrive *drive,
                                              gboolean has_mounted_volume,
                                              gpointer user_data);
static void xfce_volstatus_dialog_drive_removed(XfceVolstatusIcon *icon,
                                                GHalDrive *drive,
                                                gpointer user_data);
static void xfce_volstatus_dialog_drive_status_changed(XfceVolstatusIcon *icon,
                                                       GHalDrive *drive,
                                                       gboolean has_mounted_volume,
                                                       gpointer user_data);

static void xfce_volstatus_dialog_volume_added(GHalDrive *drive,
                                               GHalVolume *volume,
                                               gpointer user_data);
static void xfce_volstatus_dialog_volume_removed(GHalDrive *drive,
                                                 GHalVolume *volume,
                                                 gpointer user_data);


G_DEFINE_TYPE(XfceVolstatusDialog, xfce_volstatus_dialog, XFCE_TYPE_TITLED_DIALOG)


static void
xfce_volstatus_dialog_class_init(XfceVolstatusDialogClass *klass)
{
    GObjectClass *object_class = (GObjectClass *)klass;
    
    object_class->finalize = xfce_volstatus_dialog_finalize;
}

static void
xfce_volstatus_dialog_init(XfceVolstatusDialog *dialog)
{
    
}

static void
xfce_volstatus_dialog_finalize(GObject *obj)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(obj);
    GList *l;
    
    TRACE("entering");
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(dialog->icon),
                                         G_CALLBACK(xfce_volstatus_dialog_drive_added),
                                         dialog);
    g_signal_handlers_disconnect_by_func(G_OBJECT(dialog->icon),
                                         G_CALLBACK(xfce_volstatus_dialog_drive_removed),
                                         dialog);
    g_signal_handlers_disconnect_by_func(G_OBJECT(dialog->icon),
                                         G_CALLBACK(xfce_volstatus_dialog_drive_status_changed),
                                         dialog);
    
    g_object_unref(G_OBJECT(dialog->ts));
    
    for(l = dialog->drives; l; l = l->next) {
        if(!GHAL_IS_DRIVE(l->data))
            continue;
        
        g_signal_handlers_disconnect_by_func(G_OBJECT(l->data),
                                             G_CALLBACK(xfce_volstatus_dialog_volume_added),
                                             dialog);
        g_signal_handlers_disconnect_by_func(G_OBJECT(l->data),
                                             G_CALLBACK(xfce_volstatus_dialog_volume_removed),
                                             dialog);
    }
    g_list_foreach(dialog->drives, (GFunc)g_object_unref, NULL);
    g_list_free(dialog->drives);
    
    G_OBJECT_CLASS(xfce_volstatus_dialog_parent_class)->finalize(obj);
}



static const gchar *
xfce_volstatus_drive_type_to_string(GHalDrive *drive)
{
    LibHalDriveType drive_type = ghal_drive_get_drive_type(drive);
    
    switch(drive_type) {
        case LIBHAL_DRIVE_TYPE_CDROM:
            {
                GList *volumes, *l;
                GHalVolumeDisc *disc = NULL;
                LibHalVolumeDiscType disc_type;
                const gchar *type_str = NULL;
                
                volumes = ghal_drive_list_volumes(drive);
                if(G_UNLIKELY(!volumes))
                    goto cdrom_type_out;
                
                for(l = volumes; l; l = l->next) {
                    disc = l->data;
                    if(!GHAL_IS_VOLUME_DISC(disc))
                        continue;
                    
                    if(ghal_volume_disc_has_audio(disc))
                        type_str =  _("Audio CD");
                    else {
                        disc_type = ghal_volume_disc_get_disc_type(disc);
                        switch(disc_type) {
                            case LIBHAL_VOLUME_DISC_TYPE_CDROM:
                                goto cdrom_type_out;
                            case LIBHAL_VOLUME_DISC_TYPE_CDR:
                                type_str = _("Recordable CD");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_CDRW:
                                type_str = _("Rewritable CD");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_DVDROM:
                                type_str = _("DVD");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_DVDRAM:
                                type_str =  _("Rewritable DVD (DVD-RAM)");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_DVDR:
                            case LIBHAL_VOLUME_DISC_TYPE_DVDPLUSR:
                            case LIBHAL_VOLUME_DISC_TYPE_DVDPLUSR_DL:
                                type_str = _("Recordable DVD");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_DVDRW:
                            case LIBHAL_VOLUME_DISC_TYPE_DVDPLUSRW:
                                type_str = _("Rewritable DVD");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_BDROM:
                                type_str = _("Blu-ray Disc");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_BDR:
                                type_str = _("Recordable Blu-ray Disc");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_BDRE:
                                type_str = _("Rewritable Blu-ray Disc");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_HDDVDROM:
                                type_str = _("HD-DVD");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_HDDVDR:
                                type_str = _("Recordable HD-DVD");
                                break;
                            case LIBHAL_VOLUME_DISC_TYPE_HDDVDRW:
                                type_str = _("Rewritable HD-DVD");
                                break;
                            default:
                                type_str = _("Data CD");
                        }
                    }
                }
                
cdrom_type_out:
                g_list_foreach(volumes, (GFunc)g_object_unref, NULL);
                g_list_free(volumes);
                
                if(G_UNLIKELY(!type_str))
                    type_str =  _("Optical Drive");
                
                return type_str;
            }
            break;
        case LIBHAL_DRIVE_TYPE_FLOPPY:
            return _("Floppy Disk");
        case LIBHAL_DRIVE_TYPE_REMOVABLE_DISK:
            return _("Removable Disk");
        case LIBHAL_DRIVE_TYPE_DISK:
            return _("Hard Disk");
        case LIBHAL_DRIVE_TYPE_COMPACT_FLASH:
        case LIBHAL_DRIVE_TYPE_FLASHKEY:
            return _("Flash Disk");
        case LIBHAL_DRIVE_TYPE_PORTABLE_AUDIO_PLAYER:
            return _("Audio Player");
        case LIBHAL_DRIVE_TYPE_MEMORY_STICK:
            return _("Memory Stick");
        case LIBHAL_DRIVE_TYPE_SMART_MEDIA:
            return _("Smart Media Disk");
        case LIBHAL_DRIVE_TYPE_SD_MMC:
            return _("SD/MMC Disk");
        case LIBHAL_DRIVE_TYPE_CAMERA:
            return _("Camera");
        case LIBHAL_DRIVE_TYPE_ZIP:
            return _("Zip Disk");
        case LIBHAL_DRIVE_TYPE_JAZ:
            return _("Jaz Disk");
        default:
            return _("Unknown");
    }
}

static gboolean
xfce_volstatus_dialog_find_drive_iter(XfceVolstatusDialog *dialog,
                                      GHalDrive *drive,
                                      GtkTreeIter *iter)
{
    GHalDevice *a_device;
    
    /* FIXME: this is pretty inefficient */
    
    if(!gtk_tree_model_get_iter_first(GTK_TREE_MODEL(dialog->ts), iter))
        return FALSE;
    
    do {
        gtk_tree_model_get(GTK_TREE_MODEL(dialog->ts), iter,
                           COL_GHAL_DEVICE, &a_device,
                           -1);
        if(ghal_device_equal(GHAL_DEVICE(drive), a_device)) {
            g_object_unref(G_OBJECT(a_device));
            return TRUE;
        }
        g_object_unref(G_OBJECT(a_device));
    } while(gtk_tree_model_iter_next(GTK_TREE_MODEL(dialog->ts), iter));
    
    return FALSE;
}

static void
xfce_volstatus_dialog_add_drive_to_list(XfceVolstatusDialog *dialog,
                                        GHalDrive *drive)
{
    GList *icon_list, *volumes, *l;
    gchar *name;
    gint w, h;
    GdkPixbuf *pix = NULL;
    GtkTreeIter iter, iter_child;
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &w, &h);
    
    icon_list = ghal_drive_get_icon_list(drive);
    for(l = icon_list; !pix && l; l = l->next)
        pix = xfce_themed_icon_load((const gchar *)l->data, w);
    g_list_foreach(icon_list, (GFunc)g_free, NULL);
    g_list_free(icon_list);
    
    name = ghal_drive_get_display_name(drive);
    
    gtk_tree_store_append(dialog->ts, &iter, NULL);
    gtk_tree_store_set(dialog->ts, &iter,
                       COL_PIXBUF, pix,
                       COL_NAME, name,
                       COL_GHAL_DEVICE, drive,
                       -1);
    
    g_free(name);
    if(pix)
        g_object_unref(G_OBJECT(pix));
    
    g_signal_connect(G_OBJECT(drive), "volume-added",
                     G_CALLBACK(xfce_volstatus_dialog_volume_added), dialog);
    g_signal_connect(G_OBJECT(drive), "volume-removed",
                     G_CALLBACK(xfce_volstatus_dialog_volume_removed), dialog);
    
    volumes = ghal_drive_list_volumes(drive);
    for(l = volumes; l; l = l->next) {
        GHalVolume *volume = l->data;
        
        pix = NULL;
        icon_list = ghal_volume_get_icon_list(volume);
        for(l = icon_list; !pix && l; l = l->next)
            pix = xfce_themed_icon_load((const gchar *)l->data, w);
        g_list_foreach(icon_list, (GFunc)g_free, NULL);
        g_list_free(icon_list);
        
        name = ghal_volume_get_display_name(volume);

        gtk_tree_store_append(dialog->ts, &iter_child, &iter);
        gtk_tree_store_set(dialog->ts, &iter_child,
                           COL_PIXBUF, pix,
                           COL_NAME, name,
                           COL_GHAL_DEVICE, volume,
                           -1);

        g_free(name);
        if(pix)
            g_object_unref(G_OBJECT(pix));
    }
    g_list_foreach(volumes, (GFunc)g_object_unref, NULL);
    g_list_free(volumes);
}

static void
xfce_volstatus_dialog_drive_added(XfceVolstatusIcon *icon,
                                  GHalDrive *drive,
                                  gboolean has_mounted_volume,
                                  gpointer user_data)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(user_data);
    
    dialog->drives = g_list_prepend(dialog->drives,
                                    g_object_ref(G_OBJECT(drive)));
    
    if(has_mounted_volume)
        xfce_volstatus_dialog_add_drive_to_list(dialog, drive);
}

static void
xfce_volstatus_dialog_drive_removed(XfceVolstatusIcon *icon,
                                    GHalDrive *drive,
                                    gpointer user_data)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(user_data);
    GList *drive_node;
    GtkTreeIter iter;
    
    drive_node = g_list_find_custom(dialog->drives, drive, ghal_device_compare);
    if(!drive_node)
        return;
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(drive_node->data),
                                         G_CALLBACK(xfce_volstatus_dialog_volume_added),
                                         dialog);
    g_signal_handlers_disconnect_by_func(G_OBJECT(drive_node->data),
                                         G_CALLBACK(xfce_volstatus_dialog_volume_removed),
                                         dialog);
    
    g_object_unref(G_OBJECT(drive_node->data));
    dialog->drives = g_list_delete_link(dialog->drives, drive_node);
    
    if(xfce_volstatus_dialog_find_drive_iter(dialog, drive, &iter))
        gtk_tree_store_remove(dialog->ts, &iter);
}

static void
xfce_volstatus_dialog_drive_status_changed(XfceVolstatusIcon *icon,
                                           GHalDrive *drive,
                                           gboolean has_mounted_volume,
                                           gpointer user_data)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(user_data);
    
    if(!g_list_find_custom(dialog->drives, drive, ghal_device_compare)) {
        DBG("Got drive-status-changed for a drive not in our list!");
        return;
    }
    
    if(has_mounted_volume)
        xfce_volstatus_dialog_add_drive_to_list(dialog, drive);
    else {
        GtkTreeIter iter;
        if(xfce_volstatus_dialog_find_drive_iter(dialog, drive, &iter))
            gtk_tree_store_remove(dialog->ts, &iter);
    }
}

static void
xfce_volstatus_dialog_volume_added(GHalDrive *drive,
                                   GHalVolume *volume,
                                   gpointer user_data)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(user_data);
    GtkTreeIter iter, iter_child;
    GList *icon_list, *l;
    gchar *name;
    GdkPixbuf *pix = NULL;
    
    if(!xfce_volstatus_dialog_find_drive_iter(dialog, drive, &iter))
        return;
    
    icon_list = ghal_volume_get_icon_list(volume);
    if(icon_list) {
        gint w, h;
        
        gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &w, &h);
        for(l = icon_list; !pix && l; l = l->next)
            pix = xfce_themed_icon_load((const gchar *)l->data, w);
        g_list_foreach(icon_list, (GFunc)g_free, NULL);
        g_list_free(icon_list);
    }
    
    name = ghal_volume_get_display_name(volume);

    gtk_tree_store_append(dialog->ts, &iter_child, &iter);
    gtk_tree_store_set(dialog->ts, &iter_child,
                       COL_PIXBUF, pix,
                       COL_NAME, name,
                       COL_GHAL_DEVICE, volume,
                       -1);

    g_free(name);
    if(pix)
        g_object_unref(G_OBJECT(pix));
}

static void
xfce_volstatus_dialog_volume_removed(GHalDrive *drive,
                                     GHalVolume *volume,
                                     gpointer user_data)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(user_data);
    GtkTreeIter iter, iter_child;
    GHalDevice *v_device;
    
    if(!xfce_volstatus_dialog_find_drive_iter(dialog, drive, &iter))
        return;
    
    if(!gtk_tree_model_iter_children(GTK_TREE_MODEL(dialog->ts), &iter_child,
                                     &iter))
    {
        return;
    }
    
    /* FIXME: this is pretty inefficient */
    do {
        gtk_tree_model_get(GTK_TREE_MODEL(dialog->ts), &iter_child,
                           COL_GHAL_DEVICE, &v_device,
                           -1);
        if(ghal_device_equal(GHAL_DEVICE(volume), v_device)) {
            g_object_unref(G_OBJECT(v_device));
            gtk_tree_store_remove(dialog->ts, &iter);
            break;
        }
        g_object_unref(G_OBJECT(v_device));
    } while(gtk_tree_model_iter_next(GTK_TREE_MODEL(dialog->ts), &iter_child));
}

static void
xfce_volstatus_dialog_remove_clicked(GtkWidget *widget,
                                     gpointer user_data)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(user_data);
    GtkTreeSelection *sel;
    GtkTreeIter iter;
    GHalDevice *device = NULL;
    GHalDrive *drive = NULL;
    
    sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(dialog->treeview));
    if(!gtk_tree_selection_get_selected(sel, NULL, &iter))
        return;
    
    gtk_tree_model_get(GTK_TREE_MODEL(dialog->ts), &iter,
                       COL_GHAL_DEVICE, &device,
                       -1);
    if(!device)
        return;
    
    if(GHAL_IS_DRIVE(device)) {
        drive = GHAL_DRIVE(device);
        device = NULL;
    } else if(GHAL_IS_VOLUME(device)) {
        GHalVolume *volume = GHAL_VOLUME(device);
        drive = GHAL_DRIVE(ghal_volume_get_storage_device(volume));
        g_object_unref(G_OBJECT(device));
        device = NULL;
    }
    
    if(drive) {
        xfce_volstatus_remove_drive(drive);
        g_object_unref(G_OBJECT(drive));
    }
}

static void
xfce_volstatus_dialog_tree_selection_changed(GtkTreeSelection *sel,
                                             gpointer user_data)
{
    XfceVolstatusDialog *dialog = XFCE_VOLSTATUS_DIALOG(user_data);
    GtkTreeIter iter;
    GHalDevice *device = NULL;
    
    if(!gtk_tree_selection_get_selected(sel, NULL, &iter)) {
        gtk_label_set_text(GTK_LABEL(dialog->name_lbl), "");
        gtk_label_set_text(GTK_LABEL(dialog->kind_lbl), "");
        gtk_label_set_text(GTK_LABEL(dialog->mntpt_lbl), "");
        gtk_widget_set_sensitive(dialog->remove_btn, FALSE);
    } else {
        gchar *name = NULL;
        
        gtk_tree_model_get(GTK_TREE_MODEL(dialog->ts), &iter,
                           COL_NAME, &name,
                           COL_GHAL_DEVICE, &device,
                           -1);
        
        gtk_label_set_text(GTK_LABEL(dialog->name_lbl), name);
        g_free(name);
        
        if(GHAL_IS_DRIVE(device)) {
            const gchar *type = xfce_volstatus_drive_type_to_string(GHAL_DRIVE(device));
            
            gtk_label_set_text(GTK_LABEL(dialog->kind_lbl), type);
            gtk_label_set_text(GTK_LABEL(dialog->mntpt_lbl), _("n/a"));
        } else if(GHAL_IS_VOLUME(device)) {
            GHalVolume *volume = GHAL_VOLUME(device);
            LibHalVolumeUsage fsusage = ghal_volume_get_fsusage(volume);
            const gchar *mount_point, *fstype;
            gchar *str;
            
            switch(fsusage) {
                case LIBHAL_VOLUME_USAGE_MOUNTABLE_FILESYSTEM:
                    fstype = ghal_volume_get_fstype(volume);
                    if(fstype) {
                        str = g_strdup_printf(_("Filesystem (%s)"), fstype);
                        gtk_label_set_text(GTK_LABEL(dialog->kind_lbl), str);
                        g_free(str);
                    } else {
                        gtk_label_set_text(GTK_LABEL(dialog->kind_lbl),
                                           _("Filesystem"));
                    }
                    
                    if(ghal_volume_is_mounted(volume)) {
                        mount_point = ghal_volume_get_mount_point(volume);
                        if(!mount_point)
                            mount_point = _("(unknown)");
                    } else
                        mount_point = _("(n/a)");
                    gtk_label_set_text(GTK_LABEL(dialog->mntpt_lbl),
                                       mount_point);
                    break;
                
                case LIBHAL_VOLUME_USAGE_PARTITION_TABLE:
                    gtk_label_set_text(GTK_LABEL(dialog->kind_lbl),
                                       _("Partition Table"));
                    gtk_label_set_text(GTK_LABEL(dialog->mntpt_lbl),
                                       _("(n/a)"));
                    break;
                
                case LIBHAL_VOLUME_USAGE_RAID_MEMBER:
                    gtk_label_set_text(GTK_LABEL(dialog->kind_lbl),
                                       _("RAID Array Member"));
                    gtk_label_set_text(GTK_LABEL(dialog->mntpt_lbl),
                                       _("(n/a)"));
                    break;
                
                case LIBHAL_VOLUME_USAGE_CRYPTO:
                    /* FIXME: can these be mounted? */
                    gtk_label_set_text(GTK_LABEL(dialog->kind_lbl),
                                       _("Encrypted Filesystem"));
                    gtk_label_set_text(GTK_LABEL(dialog->mntpt_lbl),
                                       _("(n/a)"));
                    break;
                
                default:
                    gtk_label_set_text(GTK_LABEL(dialog->kind_lbl),
                                       _("Unknown"));
                    gtk_label_set_text(GTK_LABEL(dialog->mntpt_lbl),
                                       _("(n/a)"));
                    break;
            }
        }

        gtk_widget_set_sensitive(dialog->remove_btn, TRUE);
        g_object_unref(G_OBJECT(device));
    }
}

static void
xfce_volstatus_dialog_create(XfceVolstatusDialog *dialog)
{
    GtkWidget *topvbox, *hbox, *sw, *treeview, *btn, *table, *lbl;
    GtkTreeStore *ts;
    GtkTreeViewColumn *col;
    GtkCellRenderer *render;
    GtkTreeSelection *sel;
    
#if GTK_CHECK_VERSION(2, 12, 0)
    gtk_window_set_screen(GTK_WINDOW(dialog),
                          gtk_status_icon_get_screen(GTK_STATUS_ICON(dialog->icon)));
#endif
    
    topvbox = GTK_DIALOG(dialog)->vbox;
    gtk_container_set_border_width(GTK_CONTAINER(topvbox), BORDER);
    gtk_box_set_spacing(GTK_BOX(topvbox), BORDER);
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
                                        GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(sw);
    gtk_box_pack_start(GTK_BOX(topvbox), sw, TRUE, TRUE, 0);
    
    ts = gtk_tree_store_new(3, GDK_TYPE_PIXBUF, G_TYPE_STRING,
                            GHAL_TYPE_DEVICE);
    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(ts));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    
    col = gtk_tree_view_column_new();
    render = gtk_cell_renderer_pixbuf_new();
    gtk_tree_view_column_pack_start(col, render, FALSE);
    gtk_tree_view_column_set_attributes(col, render,
                                        "pixbuf", COL_PIXBUF, NULL);
    render = gtk_cell_renderer_text_new();
    gtk_tree_view_column_pack_start(col, render, TRUE);
    gtk_tree_view_column_set_attributes(col, render, "text", COL_NAME, NULL);
    g_object_set(G_OBJECT(render),
                 "ellipsize", PANGO_ELLIPSIZE_END,
                 "ellipsize-set", TRUE,
                 "editable", FALSE,
                 "editable-set", TRUE,
                 NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(topvbox), hbox, FALSE, FALSE, 0);
    
    btn = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
                     G_CALLBACK(xfce_volstatus_dialog_remove_clicked), dialog);
    
    table = gtk_table_new(3, 2, FALSE);
    gtk_widget_show(table);
    gtk_box_pack_start(GTK_BOX(topvbox), table, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Name:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_table_attach_defaults(GTK_TABLE(table), lbl, 0, 1, 0, 1);
    
    lbl = gtk_label_new(_("Kind:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_table_attach_defaults(GTK_TABLE(table), lbl, 0, 1, 1, 2);
    
    lbl = gtk_label_new(_("Mount Point:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_table_attach_defaults(GTK_TABLE(table), lbl, 0, 1, 2, 3);
    
    dialog->name_lbl = gtk_label_new("");
    gtk_misc_set_alignment(GTK_MISC(dialog->name_lbl), 0.0, 0.5);
    gtk_widget_show(dialog->name_lbl);
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->name_lbl, 1, 2, 0, 1);
    
    dialog->kind_lbl = gtk_label_new("");
    gtk_misc_set_alignment(GTK_MISC(dialog->kind_lbl), 0.0, 0.5);
    gtk_widget_show(dialog->kind_lbl);
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->kind_lbl, 1, 2, 1, 2);
    
    dialog->mntpt_lbl = gtk_label_new("");
    gtk_misc_set_alignment(GTK_MISC(dialog->mntpt_lbl), 0.0, 0.5);
    gtk_widget_show(dialog->mntpt_lbl);
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->mntpt_lbl, 1, 2, 2, 3);
    
    gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CLOSE,
                          GTK_RESPONSE_ACCEPT);
    g_signal_connect(G_OBJECT(dialog), "response",
                     G_CALLBACK(gtk_widget_destroy), NULL);
    
    dialog->ts = ts;
    dialog->treeview = treeview;
    dialog->remove_btn = btn;
    
    sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    gtk_tree_selection_set_mode(sel, GTK_SELECTION_SINGLE);
    g_signal_connect(G_OBJECT(sel), "changed",
                     G_CALLBACK(xfce_volstatus_dialog_tree_selection_changed),
                     dialog);
}



/*
 * public api
 */
GtkWidget *
xfce_volstatus_dialog_new(XfceVolstatusIcon *icon)
{
    XfceVolstatusDialog *dialog =  g_object_new(XFCE_TYPE_VOLSTATUS_DIALOG,
                                                "has-separator", FALSE,
                                                "title", _("Removable Volumes"),
                                                "subtitle", _("Use the list below to safely remove removable volumes."),
                                                NULL);
    GList *drives, *l;
    
    dialog->icon = icon;
    xfce_volstatus_dialog_create(dialog);
    
    drives = xfce_volstatus_icon_list_drives(dialog->icon);
    for(l = drives; l; l = l->next) {
        GList *volumes, *m;
        gboolean has_mounted_volume = FALSE;
        
        volumes = ghal_drive_list_volumes(l->data);
        for(m = volumes; m; m = m->next) {
            if(ghal_volume_is_mounted(m->data)) {
                has_mounted_volume = TRUE;
                break;
            }
        }
        g_list_foreach(volumes, (GFunc)g_object_unref, NULL);
        g_list_free(volumes);
        
        xfce_volstatus_dialog_drive_added(icon, l->data, has_mounted_volume,
                                          dialog);
    }
    g_list_free(drives);
    
    g_signal_connect(G_OBJECT(icon), "drive-added",
                     G_CALLBACK(xfce_volstatus_dialog_drive_added), dialog);
    g_signal_connect(G_OBJECT(icon), "drive-removed",
                     G_CALLBACK(xfce_volstatus_dialog_drive_removed), dialog);
    g_signal_connect(G_OBJECT(icon), "drive-status-changed",
                     G_CALLBACK(xfce_volstatus_dialog_drive_status_changed),
                     dialog);
    
    return GTK_WIDGET(dialog);
}
