/*
 *  ghal
 *
 *  Copyright (c) 2007 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ghal-context.h"
#include "ghal-private.h"
#include "ghal-drive.h"
#include "ghal-volume.h"
#include "ghal-volume-disc.h"

#include <dbus/dbus-glib-lowlevel.h>
#include <libhal.h>

struct _GHalContextPrivate
{
    LibHalContext *hal_ctx;
    DBusGConnection *dbus_gconn;
    GHashTable *devices;
};

enum
{
    SIG_DEVICE_ADDED = 0,
    SIG_DEVICE_REMOVED,
    N_SIGS,
};

static void ghal_context_class_init(GHalContextClass *klass);
static void ghal_context_init(GHalContext *context);

static void ghal_context_finalize(GObject *obj);

static guint ghc_signals[N_SIGS] = { 0, };
static GHashTable *ghc_singletons = NULL;


G_DEFINE_TYPE(GHalContext, ghal_context, G_TYPE_OBJECT)


static void
ghal_context_class_init(GHalContextClass *klass)
{
    GObjectClass *object_class = (GObjectClass *)klass;
    
    g_type_class_add_private(klass, sizeof(GHalContextPrivate));
    
    object_class->finalize = ghal_context_finalize;
    
    ghc_signals[SIG_DEVICE_ADDED] = g_signal_new("device-added",
                                                 GHAL_TYPE_CONTEXT,
                                                 G_SIGNAL_RUN_LAST,
                                                 G_STRUCT_OFFSET(GHalContextClass,
                                                                 device_added),
                                                 NULL, NULL,
                                                 g_cclosure_marshal_VOID__OBJECT,
                                                 G_TYPE_NONE, 1,
                                                 GHAL_TYPE_DEVICE);
    ghc_signals[SIG_DEVICE_REMOVED] = g_signal_new("device-removed",
                                                   GHAL_TYPE_CONTEXT,
                                                   G_SIGNAL_RUN_LAST,
                                                   G_STRUCT_OFFSET(GHalContextClass,
                                                                   device_removed),
                                                   NULL, NULL,
                                                   g_cclosure_marshal_VOID__OBJECT,
                                                   G_TYPE_NONE, 1,
                                                   GHAL_TYPE_DEVICE);
}

static void
ghal_context_init(GHalContext *context)
{
    context->priv = G_TYPE_INSTANCE_GET_PRIVATE(context, GHAL_TYPE_CONTEXT,
                                                GHalContextPrivate);
    context->priv->devices = g_hash_table_new_full(g_str_hash, g_str_equal,
                                                   (GDestroyNotify)g_free,
                                                   NULL);
}

#ifdef DEBUG
static void
ghal_context_print_devices(gpointer key,
                           gpointer value,
                           gpointer data)
{
    g_printerr("    %s\n", (gchar *)key);
}
#endif

static void
ghal_context_finalize(GObject *obj)
{
    GHalContext *context = GHAL_CONTEXT(obj);
    
    if(context->priv->devices) {
        guint ht_size = g_hash_table_size(context->priv->devices);
        if(ht_size) {
            g_warning("Attempt to finalize GHalContext without releasing %d GHalDevices",
                       ht_size);
#ifdef DEBUG
            g_printerr("Unreleased devices are:\n");
            g_hash_table_foreach(context->priv->devices,
                                 ghal_context_print_devices, NULL);
#endif
        }
        g_hash_table_destroy(context->priv->devices);
    }
    
    if(context->priv->hal_ctx) {
        libhal_ctx_shutdown(context->priv->hal_ctx, NULL);
        libhal_ctx_free(context->priv->hal_ctx);
    }
    
    G_OBJECT_CLASS(ghal_context_parent_class)->finalize(obj);
}



static void
ghal_context_check_volume_signal(GHalContext *context,
                                 GHalDevice *device,
                                 gboolean is_add)
{
    gchar *storage_udi;
    GHalDevice *storage_device;
    
    if(!GHAL_IS_VOLUME(device))
        return;
    
    storage_udi = ghal_device_get_property_string(device,
                                                  "block.storage_device",
                                                  NULL);
    if(!storage_udi)
        return;
    
    storage_device = ghal_context_get_device_for_udi(context, storage_udi);
    if(storage_device) {
        if(GHAL_IS_DRIVE(storage_device)) {
            if(is_add) {
                _ghal_drive_volume_added(GHAL_DRIVE(storage_device),
                                         GHAL_VOLUME(device));
            } else {
                _ghal_drive_volume_removed(GHAL_DRIVE(storage_device),
                                           GHAL_VOLUME(device));
            }
        }
        /* signal receivers should take reference if they want one */
        g_object_unref(G_OBJECT(storage_device));
    }
    
    g_free(storage_udi);
}

static void
ghal_context_device_added(LibHalContext *hal_ctx,
                          const char *udi)
{
    GHalContext *context = libhal_ctx_get_user_data(hal_ctx);
    GHalDevice *device = ghal_context_get_device_for_udi(context, udi);
    
    g_signal_emit(G_OBJECT(context), ghc_signals[SIG_DEVICE_ADDED], 0, device);
    
    ghal_context_check_volume_signal(context, device, TRUE);
    
    /* signal receivers should take reference if they want one */
    g_object_unref(G_OBJECT(device));
}

static void
ghal_context_device_removed(LibHalContext *hal_ctx,
                            const char *udi)
{
    GHalContext *context = libhal_ctx_get_user_data(hal_ctx);
    GHalDevice *device = ghal_context_get_device_for_udi(context, udi);
    
    ghal_context_check_volume_signal(context, device, FALSE);
    
    g_signal_emit(G_OBJECT(context), ghc_signals[SIG_DEVICE_REMOVED], 0,
                  device);
    
    /* signal receivers should take reference if they want one */
    g_object_unref(G_OBJECT(device));
}

static void
ghal_context_device_new_capability(LibHalContext *hal_ctx,
                                   const char *udi,
                                   const char *capability)
{
    GHalContext *context = libhal_ctx_get_user_data(hal_ctx);
    GHalDevice *device = _ghal_context_peek_device_cache(context, udi);
    
    if(!device)
        return;
    
    _ghal_device_new_capability(device, capability);
}

static void
ghal_context_device_lost_capability(LibHalContext *hal_ctx,
                                    const char *udi,
                                    const char *capability)
{
    GHalContext *context = libhal_ctx_get_user_data(hal_ctx);
    GHalDevice *device = _ghal_context_peek_device_cache(context, udi);
    
    if(!device)
        return;
    
    _ghal_device_lost_capability(device, capability);
}

#include <string.h>
static void
ghal_context_device_property_modified(LibHalContext *hal_ctx,
                                      const char *udi,
                                      const char *key,
                                      dbus_bool_t is_removed,
                                      dbus_bool_t is_added)
{
    GHalContext *context = libhal_ctx_get_user_data(hal_ctx);
    GHalDevice *device = _ghal_context_peek_device_cache(context, udi);
    
    if(!device)
        return;
    
    _ghal_device_property_changed(device, key,
                                  is_removed ? GHAL_PROPERTY_REMOVED
                                             : (is_added ? GHAL_PROPERTY_ADDED
                                                         : GHAL_PROPERTY_CHANGED));
}

static void
ghal_context_device_condition(LibHalContext *hal_ctx,
                              const char *udi,
                              const char *condition_name,
                              const char *condition_detail)
{
    GHalContext *context = libhal_ctx_get_user_data(hal_ctx);
    GHalDevice *device = _ghal_context_peek_device_cache(context, udi);
    
    if(!device)
        return;
    
    _ghal_device_condition(device, condition_name, condition_detail);
}

static gboolean
ghal_context_remove_ht(gpointer key,
                       gpointer value,
                       gpointer user_data)
{
    if(value == user_data)
        return TRUE;
    else
        return FALSE;
}

static void
ghal_context_weak_notify(gpointer data,
                         GObject *obj)
{
    if(!g_hash_table_foreach_remove(ghc_singletons, ghal_context_remove_ht, obj))
        g_warning("ghal_context_weak_notify(): did not find %p in hashtable", obj);
}

static void
ghal_context_device_weak_notify(gpointer data,
                                GObject *obj)
{
    GHalContext *context = data;
    g_hash_table_remove(context->priv->devices,
                        ghal_device_peek_udi((GHalDevice *)obj));
}




/*
 * internal private stuff
 */

LibHalContext *
_ghal_context_peek_libhal_context(GHalContext *context)
{
    return context->priv->hal_ctx;
}

DBusGConnection *
_ghal_context_peek_dbus_g_connection(GHalContext *context)
{
    return context->priv->dbus_gconn;
}

GHalDevice *
_ghal_context_peek_device_cache(GHalContext *context,
                                const gchar *udi)
{
    return g_hash_table_lookup(context->priv->devices, udi);
}


/*
 * public api
 */

GHalContext *
ghal_context_get(DBusGConnection *dbus_connection,
                 GError **error)
{
    GHalContext *context;
    DBusError derror;
    
    if(!ghc_singletons)
        ghc_singletons = g_hash_table_new(g_direct_hash, g_direct_equal);
    
    if(dbus_connection)
        context = g_hash_table_lookup(ghc_singletons, dbus_connection);
    else
        context = g_hash_table_lookup(ghc_singletons, (gpointer)(-1));
    
    if(context)
        return g_object_ref(G_OBJECT(context));
    
    context = g_object_new(GHAL_TYPE_CONTEXT, NULL);
    dbus_error_init(&derror);
    
    if(dbus_connection) {
        context->priv->hal_ctx = libhal_ctx_new();
        libhal_ctx_set_dbus_connection(context->priv->hal_ctx,
                                       dbus_g_connection_get_connection(dbus_connection));
        context->priv->dbus_gconn = dbus_connection;
        if(!libhal_ctx_init(context->priv->hal_ctx, &derror)) {
            libhal_ctx_free(context->priv->hal_ctx);
            context->priv->hal_ctx = NULL;
        }
    } else
        context->priv->hal_ctx = libhal_ctx_init_direct(&derror);
    
    if(G_LIKELY(context->priv->hal_ctx)) {
        libhal_ctx_set_device_added(context->priv->hal_ctx,
                                    ghal_context_device_added);
        libhal_ctx_set_device_removed(context->priv->hal_ctx,
                                      ghal_context_device_removed);
        libhal_ctx_set_device_new_capability(context->priv->hal_ctx,
                                             ghal_context_device_new_capability);
        libhal_ctx_set_device_lost_capability(context->priv->hal_ctx,
                                              ghal_context_device_lost_capability);
        libhal_ctx_set_device_property_modified(context->priv->hal_ctx,
                                                ghal_context_device_property_modified);
        libhal_ctx_set_device_condition(context->priv->hal_ctx,
                                        ghal_context_device_condition);
        libhal_ctx_set_user_data(context->priv->hal_ctx, context);
        
        if(libhal_device_property_watch_all(context->priv->hal_ctx, &derror)) {
            if(dbus_connection)
                g_hash_table_insert(ghc_singletons, dbus_connection, context);
            else
                g_hash_table_insert(ghc_singletons, (gpointer)(-1), context);
            g_object_weak_ref(G_OBJECT(context), ghal_context_weak_notify, NULL);
        } else {
            libhal_ctx_shutdown(context->priv->hal_ctx, NULL);
            libhal_ctx_free(context->priv->hal_ctx);
            context->priv->hal_ctx = NULL;
        }
    }
    
    if(G_UNLIKELY(!context->priv->hal_ctx)) {
        if(error)
            dbus_set_g_error(error, &derror);
        g_object_unref(G_OBJECT(context));
        context = NULL;
    }
    
    dbus_error_free(&derror);
    
    return context;
}

GHalDevice *
ghal_context_get_device_for_udi(GHalContext *context,
                                const gchar *udi)
{
    GHalDevice *device;
    dbus_bool_t is_storage, is_volume;
    
    device = g_hash_table_lookup(context->priv->devices, udi);
    if(device)
        return g_object_ref(G_OBJECT(device));
    
    /* we return special objects for devices that are also either drives
     * or storage volumes */
    is_storage = libhal_device_query_capability(context->priv->hal_ctx,
                                                udi, "storage", NULL);
    is_volume = libhal_device_query_capability(context->priv->hal_ctx,
                                               udi, "volume", NULL);
    
    if(is_storage)
        device = g_object_new(GHAL_TYPE_DRIVE, NULL);
    else if(is_volume) {
        if(libhal_device_get_property_bool(context->priv->hal_ctx, udi,
                                           "volume.is_disc", NULL))
        {
            device = g_object_new(GHAL_TYPE_VOLUME_DISC, NULL);
        } else
            device = g_object_new(GHAL_TYPE_VOLUME, NULL);
    } else
        device = g_object_new(GHAL_TYPE_DEVICE, NULL);
    
    _ghal_device_set_context(device, context);
    _ghal_device_set_udi(device, udi);
    
    g_object_weak_ref(G_OBJECT(device), ghal_context_device_weak_notify,
                      context);
    g_hash_table_insert(context->priv->devices, g_strdup(udi), device);
    
    return device;
}

GList *
ghal_context_get_all_devices(GHalContext *context,
                             GError **error)
{
    GList *devices = NULL;
    char **device_udis;
    int num_devices = 0, i;
    DBusError derror;
    
    dbus_error_init(&derror);
    device_udis = libhal_get_all_devices(context->priv->hal_ctx, &num_devices,
                                         &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
        num_devices = 0;
    }
    dbus_error_free(&derror);
    
    for(i = 0; i < num_devices; ++i) {
        GHalDevice *device = ghal_context_get_device_for_udi(context,
                                                             device_udis[i]);
        devices = g_list_prepend(devices, device);
    }
    
    if(device_udis)
        libhal_free_string_array(device_udis);
    
    return g_list_reverse(devices);
}

gboolean
ghal_context_device_exists(GHalContext *context,
                           const gchar *udi,
                           GError **error)
{
    gboolean ret = FALSE;
    DBusError derror;
    
    dbus_error_init(&derror);
    ret = libhal_device_exists(context->priv->hal_ctx, udi, &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
    }
    dbus_error_free(&derror);
    
    return ret;
}

GList *
ghal_context_find_device_by_capability(GHalContext *context,
                                       const gchar *capability,
                                       GError **error)
{
    GList *devices = NULL;
    char **device_udis;
    int num_devices = 0, i;
    DBusError derror;
    
    dbus_error_init(&derror);
    device_udis = libhal_find_device_by_capability(context->priv->hal_ctx,
                                                   capability,
                                                   &num_devices,
                                                   &derror);
    if(dbus_error_is_set(&derror)) {
        if(error)
            dbus_set_g_error(error, &derror);
        num_devices = 0;
    }
    dbus_error_free(&derror);
    
    for(i = 0; i < num_devices; ++i) {
        GHalDevice *device = ghal_context_get_device_for_udi(context,
                                                             device_udis[i]);
        devices = g_list_prepend(devices, device);
    }
    
    if(device_udis)
        libhal_free_string_array(device_udis);
    
    return devices;
}
