/* msm-hwrender.c
 *
 * Copyright (c) 2009, Code Aurora Forum. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Code Aurora nor
 *       the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/ioctl.h>
#include <errno.h>

#include "msm.h"
#include "msm-drm.h"
#include "msm-render.h"

#define MSM_MAX_BLITS 8

#define DEBUG_MSM_BLIT 1

static struct
{
   unsigned int count;
   struct mdp_blit_req req[MSM_MAX_BLITS];
} mdp_blit_list;

#if DEBUG_MSM_BLIT
static char * const GetPixelFormatString(char * const dest, const uint32_t pixelFormat)
{
   switch(pixelFormat)
   {
      case MDP_RGB_565:     strcpy(dest, "rgb_565"); break;
      case MDP_XRGB_8888:   strcpy(dest, "xrgb_8888"); break;
      case MDP_Y_CBCR_H2V2: strcpy(dest, "YCbCr_H2V2"); break;
      case MDP_ARGB_8888:   strcpy(dest, "argb_8888"); break;
      case MDP_RGB_888:     strcpy(dest, "rgb_888"); break;
      case MDP_Y_CRCB_H2V2: strcpy(dest, "YCrCb_H2V2"); break;
      case MDP_YCRYCB_H2V1: strcpy(dest, "YCrYCb_H2V1"); break;
      case MDP_Y_CRCB_H2V1: strcpy(dest, "YCrCb_H2V1"); break;
      case MDP_Y_CBCR_H2V1: strcpy(dest, "YCbCr_H2V1"); break;
      case MDP_RGBA_8888:   strcpy(dest, "rgba_8888"); break;
      case MDP_BGRA_8888:   strcpy(dest, "bgra_8888"); break;
      case MDP_BGR_565:     strcpy(dest, "bgr_565"); break;
      case MDP_FB_FORMAT:   strcpy(dest, "FBFormat"); break;
      default:              strcpy(dest, "INVALID-FORMAT"); break;
   }

   return dest;
}
#endif

static void DisplayMSMBlitError(MSMPtr pMsm,
				int count,
				struct mdp_blit_req *mdp_reqs)
{
#if DEBUG_MSM_BLIT
   // Report the error and give detailed info on exactly what was wrong.
   // We don't actually show complete blit request info: e.g. alpha, masks, formats, etc., are not shown.
   ErrorF("Error %d while executing MSMFB_BLIT (BlitCount = %d):\n", errno, count);
   int bl = 0;
   for (bl = 0; bl < count; bl++) {
      // Display parameters if they differ from default values re-display if they've changed from the previous setting.
      if ((bl == 0 && (mdp_reqs[bl].alpha != MDP_ALPHA_NOP
                       || mdp_reqs[bl].transp_mask != MDP_TRANSP_NOP
                       || mdp_reqs[bl].flags != 0
                       || mdp_reqs[bl].sharpening_strength != 64))
          || (bl > 0 && (mdp_reqs[bl].alpha != mdp_reqs[bl-1].alpha
                        || mdp_reqs[bl].transp_mask != mdp_reqs[bl-1].transp_mask
                        || mdp_reqs[bl].flags != mdp_reqs[bl-1].flags
                        || mdp_reqs[bl].sharpening_strength != mdp_reqs[bl-1].sharpening_strength))) {
         ErrorF("  Non-Default Params: ");
         if (mdp_reqs[bl].alpha != MDP_ALPHA_NOP
             || (bl > 0 && mdp_reqs[bl].alpha != mdp_reqs[bl-1].alpha))
            ErrorF("Alpha=0x%x, ", mdp_reqs[bl].alpha);
         if (mdp_reqs[bl].transp_mask != MDP_TRANSP_NOP
             || (bl > 0 && mdp_reqs[bl].transp_mask != mdp_reqs[bl-1].transp_mask))
            ErrorF("TranspMask=0x%x, ", mdp_reqs[bl].transp_mask);
         if (mdp_reqs[bl].flags != 0
             || (bl > 0 && mdp_reqs[bl].flags != mdp_reqs[bl-1].flags))
            ErrorF("Flags=0x%x, ", mdp_reqs[bl].flags);
         if (mdp_reqs[bl].sharpening_strength != 64
             || (bl > 0 && mdp_reqs[bl].sharpening_strength != mdp_reqs[bl-1].sharpening_strength))
            ErrorF("ShStrength=%d, ", mdp_reqs[bl].sharpening_strength);
         ErrorF("\n");
      }

      // Display the image portion of the request for the first request and re-display it when it changes.
      if (bl < 1
          || mdp_reqs[bl].src.offset != mdp_reqs[bl-1].src.offset
          || mdp_reqs[bl].src.width != mdp_reqs[bl-1].src.width
          || mdp_reqs[bl].src.height != mdp_reqs[bl-1].src.height
          || mdp_reqs[bl].src.format != mdp_reqs[bl-1].src.format
          || mdp_reqs[bl].dst.offset != mdp_reqs[bl-1].dst.offset
          || mdp_reqs[bl].dst.width != mdp_reqs[bl-1].dst.width
          || mdp_reqs[bl].dst.height != mdp_reqs[bl-1].dst.height
          || mdp_reqs[bl].dst.format != mdp_reqs[bl-1].dst.format) {
         char dstFormatString[100];
         char srcFormatString[100];
         GetPixelFormatString(dstFormatString, mdp_reqs[bl].dst.format);
         GetPixelFormatString(srcFormatString, mdp_reqs[bl].src.format);
         ErrorF("  Image: %u:(w=%u,h=%u)/%s --> %u:(w=%u,h=%u)/%s\n", mdp_reqs[bl].src.offset, mdp_reqs[bl].src.width, mdp_reqs[bl].src.height, srcFormatString,
                                                                      mdp_reqs[bl].dst.offset, mdp_reqs[bl].dst.width, mdp_reqs[bl].dst.height, dstFormatString);
      }

      // Display the rectangle coordinates and sizes that are actually copied.
      ErrorF("    CopyRect: (%u,%u) --> (%u,%u) of w=%u,h=%u", mdp_reqs[bl].src_rect.x, mdp_reqs[bl].src_rect.y,
                                                               mdp_reqs[bl].dst_rect.x, mdp_reqs[bl].dst_rect.y,
                                                               mdp_reqs[bl].src_rect.w, mdp_reqs[bl].src_rect.h);
      if (mdp_reqs[bl].src_rect.w != mdp_reqs[bl].dst_rect.w
          || mdp_reqs[bl].src_rect.h != mdp_reqs[bl].dst_rect.h) {
         ErrorF(" (scaled to w=%u,h=%u)", mdp_reqs[bl].src_rect.w, mdp_reqs[bl].src_rect.h);
      }

      // Check for a few invalid conditions and print errors for them if they are incorrect.
      if (mdp_reqs[bl].src_rect.x + mdp_reqs[bl].src_rect.w > mdp_reqs[bl].src.width)
         ErrorF("  -Err: sr.x+sr.w>s.width!");
      if (mdp_reqs[bl].dst_rect.x + mdp_reqs[bl].dst_rect.w > mdp_reqs[bl].dst.width)
         ErrorF("  -Err: dr.x+dr.w>d.width!");
      if (mdp_reqs[bl].src_rect.y + mdp_reqs[bl].src_rect.h > mdp_reqs[bl].src.height)
         ErrorF("  -Err: sr.y+sr.h>s.height!");
      if (mdp_reqs[bl].dst_rect.y + mdp_reqs[bl].dst_rect.h > mdp_reqs[bl].dst.height)
         ErrorF("  -Err: dr.y+dr.h>d.height!");
      ErrorF("\n");
   }
#else
      ErrorF("Error while executing MSMFB_BLIT\n");
#endif // DEBUG_MSM_BLIT
}

void hwBlitFlush(MSMPtr pMsm)
{
    if (ioctl(pMsm->fd, MSMFB_BLIT, &mdp_blit_list))
	DisplayMSMBlitError(pMsm, mdp_blit_list.count, &(mdp_blit_list.req[0]));

    mdp_blit_list.count = 0;
}

void hwBlitReset(void)
{
    mdp_blit_list.count = 0;
}

static
int formatToCpp(MSMPtr pMsm, int format)
{
    switch(format) {
    case MDP_XRGB_8888:
	return 4;
    case MDP_RGB_888:
	    return 3;
    case MDP_RGB_565:
	    return 2;
    case MDP_YCRYCB_H2V1:
    case MDP_Y_CRCB_H2V2:
    case MDP_Y_CBCR_H2V2:
	return 2;
    case MDP_FB_FORMAT:
	return (pMsm->mode_info.bits_per_pixel >> 3);
   }

   return 0;
}

void
hwBlit(MSMPtr pMsm, MSMBlitRec *blit, int flags)
{
    int index = mdp_blit_list.count;

    mdp_blit_list.req[index].flags = flags;
    mdp_blit_list.req[index].alpha = 0xFF;
    mdp_blit_list.req[index].transp_mask = 0xFFFFFFFF;

    /* The width is actually the pitch / cpp */

    mdp_blit_list.req[index].src.width = blit->src->pitch /
	formatToCpp(pMsm, blit->src->format);

    mdp_blit_list.req[index].src.height = blit->src->height;
    mdp_blit_list.req[index].src.format = blit->src->format;

    if (blit->src->flags & MSM_BLIT_FB) {
	mdp_blit_list.req[index].src.offset = blit->src->priv[0];
	mdp_blit_list.req[index].src.memory_id = pMsm->fd;
    }
    else if (blit->src->flags & MSM_BLIT_PMEM) {
	mdp_blit_list.req[index].src.offset = blit->src->priv[1];
	mdp_blit_list.req[index].src.memory_id = blit->src->priv[0];
    }
#if MDP_BLIT_REQ_VERSION >= 2
    else {
	struct msm_drm_bo *bo = (struct msm_drm_bo *) blit->src->priv[0];

	mdp_blit_list.req[index].flags |= MDP_BLIT_SRC_GEM;
	mdp_blit_list.req[index].src.offset = 0;
	mdp_blit_list.req[index].src.memory_id = pMsm->drmFD;
	mdp_blit_list.req[index].src.priv = bo->handle;
    }
#endif

    mdp_blit_list.req[index].dst.width = blit->dst->pitch /
	formatToCpp(pMsm, blit->dst->format);

    mdp_blit_list.req[index].dst.height = blit->dst->height;
    mdp_blit_list.req[index].dst.format = blit->dst->format;

    if (blit->dst->flags & MSM_BLIT_FB) {
	mdp_blit_list.req[index].dst.offset = blit->dst->priv[0];
	mdp_blit_list.req[index].dst.memory_id = pMsm->fd;
    }
#if MDP_BLIT_REQ_VERSION >= 2
    else {
	struct msm_drm_bo *bo = (struct msm_drm_bo *) blit->dst->priv[0];

	mdp_blit_list.req[index].flags |= MDP_BLIT_DST_GEM;
	mdp_blit_list.req[index].dst.offset = 0;
	mdp_blit_list.req[index].dst.memory_id = pMsm->drmFD;
	mdp_blit_list.req[index].dst.priv = bo->handle;
    }
#endif

    mdp_blit_list.req[index].src_rect.x = blit->srcRect->x;
    mdp_blit_list.req[index].src_rect.y = blit->srcRect->y;
    mdp_blit_list.req[index].src_rect.w = blit->srcRect->w;
    mdp_blit_list.req[index].src_rect.h = blit->srcRect->h;

    mdp_blit_list.req[index].dst_rect.x = blit->dstRect->x;
    mdp_blit_list.req[index].dst_rect.y = blit->dstRect->y;
    mdp_blit_list.req[index].dst_rect.w = blit->dstRect->w;
    mdp_blit_list.req[index].dst_rect.h = blit->dstRect->h;

    mdp_blit_list.count++;

    if (mdp_blit_list.count == MSM_MAX_BLITS)
	hwBlitFlush(pMsm);
}
