/*
 *	Xenophilia GTK+ Theme Engine
 *
 *  xeno_theme.c:
 *		Theme engine
 *
 *	Copyright  1999-2002 Johan Hanson <misagon@bahnhof.se>
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include <stdlib.h>

#include <gmodule.h>
#include <gtk/gtkrc.h>

#include "xeno_theme.h"
#include "xeno_rc_style.h"
#include "xeno_style.h"

#include "xeno_patches.h"

#define INSTALL_PATCHES	0

/* Theme functions to export */
#if XENO_GTK2
void		theme_init				(GTypeModule *module);
GtkRcStyle *theme_create_rc_style	(void);
#else
void		theme_init				(GtkThemeEngine *engine);
#endif

void		theme_exit				(void);

/* GtkThemeEngine */
#if !XENO_GTK2
static void		xeno_duplicate_style	(GtkStyle *dest, GtkStyle	*src);
static void		xeno_destroy_style		(GtkStyle *style);
#endif

/* Exported variables */
GtkThemeEngine *xeno_theme_engine		= XENO_THEME_ENGINE_NONE;
GdkVisual *		xeno_theme_visual		= NULL;
GdkColormap *	xeno_theme_colormap		= NULL;
gdouble			xeno_theme_gamma_exp	= 2.2;
gboolean		xeno_theme_use_gamma	= TRUE;
gboolean		xeno_theme_pseudocolor	= FALSE;


/*
 *	GtkThemeEngine
 */

#if XENO_GTK2
GtkRcStyle * theme_create_rc_style (void) {
	return (GtkRcStyle *) xeno_rc_style_new();
}

#else
static void
xeno_duplicate_style (GtkStyle *dest, GtkStyle *src) {
	g_return_if_fail (XENO_STYLE_IS_XENO(src));
	g_return_if_fail (XENO_STYLE_IS_XENO(dest));
	
	if (dest->engine_data) {
		g_warning ("xeno_duplicate_style(): dest has engine data\n");
		xeno_style_data_destroy (dest->engine_data);
		dest->engine_data = NULL;
	}
}

static void
xeno_unrealize_style (GtkStyle *style) {
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_STYLE_IS_XENO(style));
	
	if (style->engine_data) {
		xeno_style_data_destroy (style->engine_data);
		style->engine_data = NULL;
	}
}

static void
xeno_destroy_style (GtkStyle *style) {
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_STYLE_IS_XENO(style));
	
	if (style->engine_data) {
		g_warning ("xeno_destroy_style(): style has engine data\n");
		xeno_style_data_destroy (style->engine_data);
		style->engine_data = NULL;
	}
}
#endif

gboolean
theme_parse_boolean (const gchar *str, gboolean *return_p) {
	if (str != NULL) {
		if (   g_strcasecmp(str, "YES") == 0
			|| g_strcasecmp(str, "TRUE") == 0
			|| g_strcasecmp(str, "ON") == 0
			|| g_strcasecmp(str, "1") == 0)
		{
			*return_p = TRUE;
		} else if (   g_strcasecmp(str, "NO") == 0
				   || g_strcasecmp(str, "FALSE") == 0
				   || g_strcasecmp(str, "OFF") == 0
				   || g_strcasecmp(str, "0") == 0)
		{
			*return_p = FALSE;
		} else {
			return FALSE;
		}
		return TRUE;
	}
	return FALSE;
}

void theme_load () {
}

void theme_unload () {
}

#if XENO_GTK2
void theme_init (GTypeModule *module) {
#else
void theme_init (GtkThemeEngine *engine) {
#endif
	const char	*env_str;
	GdkVisual	*visual;
	
  #if XENO_GTK2
	/* by referencing ourselves, we stay in memory */
	g_type_module_use (module);
  #else
	/* Initialize theme engine structure */
	engine->parse_rc_style    = xeno_parse_rc_style;
	engine->merge_rc_style    = xeno_merge_rc_style;
	engine->rc_style_to_style = xeno_rc_style_to_style;
	engine->duplicate_style   = xeno_duplicate_style;
	engine->realize_style     = xeno_realize_style;
	engine->unrealize_style   = xeno_unrealize_style;
	engine->destroy_rc_style  = xeno_destroy_rc_style;
	engine->destroy_style     = xeno_destroy_style;
	engine->set_background    = NULL;
	
	if (xeno_theme_engine != XENO_THEME_ENGINE_NONE) {
		g_warning ("xeno_theme_engine != XENO_THEME_ENGINE_NONE");
	} else {
		xeno_theme_engine = engine;
	}
  #endif
	
	/* Test display and see if we need to reduce color usage */
	env_str = getenv ("XENO_THEME_CMAP");
	visual = gtk_widget_get_default_visual ();
	
	if (!theme_parse_boolean (env_str, &xeno_theme_pseudocolor)) {
		xeno_theme_pseudocolor = (   visual->type == GDK_VISUAL_PSEUDO_COLOR
								  || visual->type == GDK_VISUAL_DIRECT_COLOR);
	}
	
	xeno_theme_visual = visual;
	xeno_theme_colormap = gtk_widget_get_default_colormap();
	
	/* Get user-defined gamma or the default which is 2.2 */
	xeno_theme_use_gamma = TRUE;
	xeno_theme_gamma_exp = 2.2;
	
	env_str = getenv("XENO_THEME_GAMMA");
	if (env_str && !theme_parse_boolean (env_str, &xeno_theme_use_gamma)) {
		gchar *end_str;
		gboolean d;
		
		d = g_strtod(env_str, &end_str);
		if (env_str != end_str) {
			xeno_theme_gamma_exp = d;
			if (d == 1.0)
				xeno_theme_use_gamma = FALSE;
		}
	}
	
  #if INSTALL_PATCHES
	xeno_patches_install();
  #endif
}

void theme_exit (void)
{
  #if !XENO_GTK2
	if (xeno_style_data_chunk)
		g_mem_chunk_destroy (xeno_style_data_chunk);
	
	if (xeno_rc_data_chunk)
		g_mem_chunk_destroy (xeno_rc_data_chunk);
	
	xeno_theme_engine = NULL;
  #endif
  #if INSTALL_PATCHES
	xeno_patches_uninstall();
  #endif
}


/*
 * The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
 */
G_MODULE_EXPORT const gchar* g_module_check_init (GModule *module);

const gchar* g_module_check_init (GModule *module) {
	return gtk_check_version (GTK_MAJOR_VERSION,
							  GTK_MINOR_VERSION,
							  GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}


/* end */

