#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <xs.h>
#include <xenctrl.h>
#include <xen/grant_table.h>
#include <xen/io/xenbus.h>

#include "xenbackd.h"
#include "daemon.h"
#include "shared.h"

/* ------------------------------------------------------------- */

static LIST_HEAD(xendevs);
static int devcount;

/* ------------------------------------------------------------- */

static struct xendev *find_xendev(int dom, int dev)
{
    struct xendev *xendev;
    struct list_head *item;

    list_for_each(item, &xendevs) {
	xendev = list_entry(item, struct xendev, next);
	if (xendev->dom == dom  &&  xendev->dev == dev)
	    return xendev;
    }
    return NULL;
}

static struct xendev *get_xendev(int dom, int dev,
				 char *be_root,
				 struct xs_handle *xs,
				 struct devops *ops)
{
    struct xendev *xendev;

    xendev = find_xendev(dom, dev);
    if (!xendev) {
	xendev = malloc(ops->size);
	memset(xendev,0,ops->size);
	xendev->dom    = dom;
	xendev->dev    = dev;

	xendev->be_root = be_root;
	xendev->xs      = xs;
	xendev->ops     = ops;

	xendev->evtchnd = xc_evtchn_open();
	if (xendev->evtchnd < 0) {
	    fprintf(stderr, "can't connect to evtchnd\n");
	    free(xendev);
	    return NULL;
	}
	fcntl(xc_evtchn_fd(xendev->evtchnd),F_SETFD,FD_CLOEXEC);
	
	xendev->gnttabdev = xc_gnttab_open();
	if (xendev->gnttabdev < 0) {
	    fprintf(stderr, "can't open gnttab device\n");
	    free(xendev);
	    return NULL;
	}

	list_add_tail(&xendev->next, &xendevs);
	devcount++;
	d1printf("%s: %d/%d\n", __FUNCTION__,
		 xendev->dom, xendev->dev);

	if (xendev->ops->alloc)
	    xendev->ops->alloc(xendev);
    }
    return xendev;
}

static struct xendev *del_xendev(int dom, int dev)
{
    struct xendev *xendev;
    struct list_head *item, *tmp;

    list_for_each_safe(item, tmp, &xendevs) {
	xendev = list_entry(item, struct xendev, next);
	if (xendev->dom != dom)
	    continue;
	if (xendev->dev != dev && dev != -1)
	    continue;

	if (xendev->ops->free)
	    xendev->ops->free(xendev);

	if (xendev->fe) {
	    char token[BUFSIZE];
	    snprintf(token, sizeof(token), "fe:%p", xendev);
	    xs_unwatch(xendev->xs, xendev->fe, token);
	    free(xendev->fe);
	}

	if (xendev->evtchnd > 0)
	    xc_evtchn_close(xendev->evtchnd);
	if (xendev->gnttabdev > 0)
	    xc_gnttab_close(xendev->gnttabdev);
	
	d1printf("%s: %d/%d\n", __FUNCTION__,
		 xendev->dom, xendev->dev);
	list_del(&xendev->next);
	devcount--;
	free(xendev);
    }
    return NULL;
}

/* ------------------------------------------------------------- */

int write_be_str(struct xendev *xendev, char *node, char *val)
{
    char abspath[BUFSIZE];
    
    snprintf(abspath, sizeof(abspath), "%s/%d/%d/%s",
	     xendev->be_root, xendev->dom, xendev->dev, node);
    return xs_write(xendev->xs, 0, abspath, val, strlen(val));
}

int write_be_int(struct xendev *xendev, char *node, int ival)
{
    char val[32];

    snprintf(val, sizeof(val), "%d", ival);
    return write_be_str(xendev, node, val);
}

char *read_be_str(struct xendev *xendev, char *node)
{
    char abspath[BUFSIZE];
    unsigned int len;

    snprintf(abspath, sizeof(abspath), "%s/%d/%d/%s",
	     xendev->be_root, xendev->dom, xendev->dev, node);
    return xs_read(xendev->xs, 0, abspath, &len);
}

int read_be_int(struct xendev *xendev, char *node)
{
    char *val;
    int ival;

    val = read_be_str(xendev, node);
    ival = val ? atoi(val) : 0;
    free(val);
    return ival;
}

char *read_fe_str(struct xendev *xendev, char *node)
{
    char abspath[BUFSIZE];
    unsigned int len;

    snprintf(abspath, sizeof(abspath), "%s/%s",
	     xendev->fe, node);
    return xs_read(xendev->xs, 0, abspath, &len);
}

int read_fe_int(struct xendev *xendev, char *node)
{
    char *val;
    int ival;

    val = read_fe_str(xendev, node);
    ival = val ? atoi(val) : 0;
    free(val);
    return ival;
}

static int check_state_xendev(struct xendev *xendev, int be, int fe)
{
    enum xenbus_state state;

    if (be) {
	state = read_be_int(xendev, "state");
	if (xendev->be_state != state) {
	    d1printf("%s: be: %d -> %d\n", __FUNCTION__,
		     xendev->be_state, state);
	    xendev->be_state  = state;
	}
    }

    if (fe) {
	state = read_fe_int(xendev, "state");
	if (xendev->fe_state != state) {
	    d1printf("%s: fe: %d -> %d\n", __FUNCTION__,
		     xendev->fe_state, state);
	    xendev->fe_state  = state;
	}
    }
    return 0;
}

int change_state_xendev(struct xendev *xendev, enum xenbus_state state)
{
    write_be_int(xendev, "state", state);
    d1printf("%s: be: %d -> %d\n", __FUNCTION__,
	     xendev->be_state, state);
    xendev->be_state  = state;
    return 0;
}

static int setup_xendev(struct xendev *xendev)
{
    char token[BUFSIZE];

    if (NULL != xendev->fe)
	/* already done */
	return 0;

    xendev->fe = read_be_str(xendev, "frontend");
    if (NULL == xendev->fe)
	/* no fe ptr yet */
	return 0;
    xendev->online = read_be_int(xendev, "online");

    /* setup frontend watch */
    snprintf(token, sizeof(token), "fe:%p", xendev);
    xs_watch(xendev->xs, xendev->fe, token);

    check_state_xendev(xendev, 1, 1);

    if (xendev->ops->setup)
	xendev->ops->setup(xendev);
    return 0;
}

/* ------------------------------------------------------------- */

static void xenstore_scan(struct xs_handle *xs, char *be_root,
			  struct devops *ops)
{
    struct xendev *xendev;
    char path[BUFSIZE], token[BUFSIZE];
    char **dom = NULL;
    char **dev = NULL;
    unsigned int cdom = 0, cdev = 0, i, j;
        
    /* setup watch */
    snprintf(token, sizeof(token), "be:%p:%p", be_root, ops);
    xs_watch(xs, be_root, token);

    /* look for backends */
    d1printf("%s: %s\n", __FUNCTION__, be_root);
    dom = xs_directory(xs, 0, be_root, &cdom);
    for (i = 0; i < cdom; i++) {
	d1printf("%s:   %s\n", __FUNCTION__, dom[i]);
	snprintf(path, sizeof(path), "%s/%s", be_root, dom[i]);
	dev = xs_directory(xs, 0, path, &cdev);
	for (j = 0; j < cdev; j++) {
	    d1printf("%s:     %s\n", __FUNCTION__, dev[j]);
	    xendev = get_xendev(atoi(dom[i]), atoi(dev[j]), be_root, xs, ops);
	    if (NULL == xendev)
		continue;
	    setup_xendev(xendev);
	}
	if (dev)
	    free(dev);
    }
    if (dom)
	free(dom);
}

static void xenstore_update_be(struct xs_handle *xs, char *watch,
			       char *be_root, struct devops *ops)
{
    struct xendev *xendev;
    char path[BUFSIZE], *val;
    int dom, dev;
    unsigned int len;

    len = strlen(be_root);
    if (0 != strncmp(be_root, watch, strlen(be_root)))
	return;
    if (3 != sscanf(watch+len, "/%d/%d/%255s", &dom, &dev, path)) {
	strcpy(path, "");
	if (2 != sscanf(watch+len, "/%d/%d", &dom, &dev)) {
	    dev = -1;
	    if (1 != sscanf(watch+len, "/%d", &dom))
		return;
	}
    }
    val = xs_read(xs, 0, watch, &len);

    if (NULL == val) {
	del_xendev(dom, dev);
	return;
    }

    if (-1 != dev) {
	xendev = get_xendev(dom, dev, be_root, xs, ops);
	if (NULL != xendev) {
	    if (0 == strcmp(path, "online"))
		xendev->online = atoi(val);
	    setup_xendev(xendev);
	    if (xendev->fe && xendev->ops->xs_be)
		xendev->ops->xs_be(xendev, path, val);
	}
    }

    if (val)
	free(val);
}

static void xenstore_update_fe(struct xs_handle *xs, char *watch, struct xendev *xendev)
{
    char *val, *node;
    unsigned int len;

    val = xs_read(xs, 0, watch, &len);

    node = strrchr(watch, '/');
    if (node) {
	node++;
	if (0 == strcmp(node, "state"))
	    check_state_xendev(xendev, 0, 1);
    }
    if (xendev->ops->xs_fe)
	xendev->ops->xs_fe(xendev, node, val);

    if (val)
	free(val);
}

static void xenstore_update(struct xs_handle *xs)
{
    char **vec = NULL;
    intptr_t ptr, be;
    unsigned int count;
        
    vec = xs_read_watch(xs, &count);
    if (NULL == vec)
	goto cleanup;
    d2printf("%s: %s [%s]\n", __FUNCTION__,
	     vec[XS_WATCH_PATH], vec[XS_WATCH_TOKEN]);

    if (2 == sscanf(vec[XS_WATCH_TOKEN], "be:%" PRIxPTR ":%" PRIxPTR, &be, &ptr))
	xenstore_update_be(xs, vec[XS_WATCH_PATH], (void*)be, (void*)ptr);
    if (1 == sscanf(vec[XS_WATCH_TOKEN], "fe:%" PRIxPTR, &ptr))
	xenstore_update_fe(xs, vec[XS_WATCH_PATH], (void*)ptr);

cleanup:
    if (vec)
	free(vec);
}

int wait_for_data(int fd, int timeout)
{
    fd_set rd;
    struct timeval tv;

    if (timeout) {
	tv.tv_sec = timeout;
	tv.tv_usec = 0;
    }
    FD_ZERO(&rd);
    FD_SET(fd,&rd);
    return select(fd+1, &rd, NULL, NULL, timeout ? &tv : NULL);
}

int wait_for_event(struct xendev *xendev)
{
    int port;

    if (1 != wait_for_data(xc_evtchn_fd(xendev->evtchnd), 0))
	return -1;
    port = xc_evtchn_pending(xendev->evtchnd);
    xc_evtchn_unmask(xendev->evtchnd, port);
    return port;
}

static void shutdown_xendevs(void)
{
    struct xendev *xendev;
    struct list_head *item;

    list_for_each(item, &xendevs) {
	xendev = list_entry(item, struct xendev, next);
	if (xendev->be_state == XenbusStateConnected)
	    change_state_xendev(xendev, XenbusStateInitialising);
    }
}

static void write_stats(char *name)
{
    struct xendev *xendev;
    struct list_head *item;
    char filename[BUFSIZE], tmpfile[BUFSIZE];
    FILE *fp;
    int fd;

    snprintf(filename, sizeof(filename), "/var/run/%s.stats", name);
    snprintf(tmpfile,  sizeof(tmpfile),  "/var/run/%s.stats.XXXXXX", name);
    fd = mkstemp(tmpfile);
    if (-1 == fd)
	return;
    fp = fdopen(fd, "w");
    if (NULL == fp)
	return;
    fprintf(fp, "%s stats\n", name);
    list_for_each(item, &xendevs) {
	xendev = list_entry(item, struct xendev, next);
	fprintf(fp, "  dom %d dev %d -- fe state %d, be state %d, online %d\n",
		xendev->dom, xendev->dev,
		xendev->fe_state, xendev->be_state, xendev->online);
	if (xendev->ops->stats && xendev->be_state == XenbusStateConnected)
	    xendev->ops->stats(xendev, fp);
    }
    fclose(fp);
    rename(tmpfile, filename);
}

uint32_t timediff_msecs(struct timeval *n, struct timeval *o)
{
    uint32_t msecs;

    msecs  = (n->tv_sec  - o->tv_sec)  * 1000;
    msecs += (n->tv_usec - o->tv_usec) / 1000;
    return msecs;
}

/* ------------------------------------------------------------- */

static int termsig;

static void catchsig(int sig)
{
    if (sig != SIGUSR2)
	termsig = sig;
}

int mainloop(struct devops *ops, char *be_name, int stat_secs)
{
    char be_root[256];
    struct xs_handle *xs = NULL;
    struct sigaction act,old;
    time_t last = 0, now;

    /* setup signal handler */
    memset(&act,0,sizeof(act));
    sigemptyset(&act.sa_mask);
    act.sa_handler = catchsig;
    sigaction(SIGTERM,&act,&old);
    sigaction(SIGUSR2,&act,&old);
    sigaction(SIGINT,&act,&old);

    xs = xs_daemon_open();
    if (!xs) {
	fprintf(stderr, "can't connect to xenstored\n");
	exit(1);
    }
    fcntl(xs_fileno(xs),F_SETFD,FD_CLOEXEC);
    
    /* fork into background, handle pidfile */
    daemonize();

    snprintf(be_root, sizeof(be_root), "/local/domain/0/backend/%s", be_name);
    xenstore_scan(xs, be_root, ops);

    /* main loop */
    for (;;) {
	if (termsig)
	    break;

	now = time(NULL);
	if (now != last) {
	    last = now;
	    write_stats(be_name);
	};

	if (1 == wait_for_data(xs_fileno(xs), stat_secs))
	    xenstore_update(xs);
    }
    
    d1printf("quit on signal: %d\n", termsig);
    shutdown_xendevs();
    
    return 0;
}
