#define	ENTRY(name) \
	.globl name; \
	.align 16; \
	name:

	.macro PUSH_ERROR
	sub $8, %rsp		/* space for error code */
	.endm
	
	.macro PUSH_TRAP_RBP trapno
	sub $8, %rsp		/* space for trap number */
	push %rbp
	mov $\trapno, %rbp
	mov %rbp, 8(%rsp)	/* save trap number on stack */
	.endm

	.macro PUSH_REGS
	push %rdi
	push %rsi
	push %r15
	push %r14
	push %r13
	push %r12
	push %r11
	push %r10
	push %r9
	push %r8
	push %rdx
	push %rcx
	push %rbx
	push %rax
	mov  %rsp,%rdi          /* struct regs pointer */
	.endm

	.macro POP_REGS
	pop %rax
	pop %rbx
	pop %rcx
	pop %rdx
	pop %r8
	pop %r9
	pop %r10
	pop %r11
	pop %r12
	pop %r13
	pop %r14
	pop %r15
	pop %rsi
	pop %rdi
	pop %rbp
	.endm

	.macro RETURN
	add $16, %rsp		/* remove error code & trap number */
	iretq			/* jump back */
	.endm	

	.macro DO_TRAP trapno func
	PUSH_TRAP_RBP \trapno
	PUSH_REGS
	call \func
	POP_REGS
	RETURN
	.endm

/* ------------------------------------------------------------------ */

	.code64
	.text

/* --- boot entry point --- */

ENTRY(boot)
	cli
	lea boot_stack_high(%rip), %rsp	/* setup stack */
	sub $176, %rsp			/* sizeof(struct regs_64) */
	mov  %rsp,%rdi                  /* struct regs pointer */
	cmp $0, %rbx
	jne secondary
	call do_boot
	POP_REGS
	RETURN

secondary:
	mov  %rdi,%rsi
	mov  %rbx,%rdi
	call do_boot_secondary
	POP_REGS
	RETURN

/* --- traps/faults handled by emu --- */

ENTRY(debug_int1)
	PUSH_ERROR
	DO_TRAP 1 do_int1

ENTRY(debug_int3)
	PUSH_ERROR
	DO_TRAP 3 do_int3

ENTRY(illegal_instruction)
	PUSH_ERROR
	DO_TRAP 6 do_illegal_instruction

ENTRY(no_device)
	PUSH_ERROR
	DO_TRAP 7 do_lazy_fpu

ENTRY(double_fault)
	DO_TRAP 8 do_double_fault

ENTRY(general_protection)
	DO_TRAP 13 do_general_protection

ENTRY(page_fault)
	DO_TRAP 14 do_page_fault

/* --- traps/faults forwarded to guest --- */

ENTRY(division_by_zero)
	PUSH_ERROR
	PUSH_TRAP_RBP 0
	jmp guest_forward

ENTRY(nmi)
	PUSH_ERROR
	PUSH_TRAP_RBP 2
	jmp guest_forward

ENTRY(overflow)
	PUSH_ERROR
	PUSH_TRAP_RBP 4
	jmp guest_forward

ENTRY(bound_check)
	PUSH_ERROR
	PUSH_TRAP_RBP 5
	jmp guest_forward

ENTRY(coprocessor)
	PUSH_ERROR
	PUSH_TRAP_RBP 9
	jmp guest_forward

ENTRY(invalid_tss)
	PUSH_TRAP_RBP 10
	jmp guest_forward

ENTRY(segment_not_present)
	PUSH_TRAP_RBP 11
	jmp guest_forward

ENTRY(stack_fault)
	PUSH_TRAP_RBP 12
	jmp guest_forward

ENTRY(floating_point)
	PUSH_ERROR
	PUSH_TRAP_RBP 16
	jmp guest_forward

ENTRY(alignment)
	PUSH_TRAP_RBP 17
	jmp guest_forward

ENTRY(machine_check)
	PUSH_ERROR
	PUSH_TRAP_RBP 18
	jmp guest_forward

ENTRY(simd_floating_point)
	PUSH_ERROR
	PUSH_TRAP_RBP 19
	jmp guest_forward

guest_forward:
	PUSH_REGS
	call do_guest_forward
	POP_REGS
	RETURN

/* --- interrupts 32 ... 255 --- */

ENTRY(smp_flush_tlb)
	PUSH_ERROR
	DO_TRAP -1 do_smp_flush_tlb

ENTRY(int_80)
	PUSH_ERROR
	DO_TRAP -1 do_int_80

ENTRY(irq_entries)
vector=0
.rept 256
	.align 16
	PUSH_ERROR
	PUSH_TRAP_RBP vector
	jmp irq_common
vector=vector+1
.endr

ENTRY(irq_common)
	PUSH_REGS
	call do_irq
	POP_REGS
	RETURN

/* --- syscall --- */

ENTRY(trampoline_syscall)
	/* we arrive here via per-cpu trampoline
	 * which sets up the stack for us */
	PUSH_ERROR
	PUSH_TRAP_RBP -1
	PUSH_REGS
	call do_syscall
	POP_REGS

	add $16, %rsp			/* remove error code + trapno */
	cmp $-1, -8(%rsp)
	je syscall_vmexit
	cmp $-2, -8(%rsp)
	je syscall_iretq

syscall_default:
	/* default sysret path */
	popq %rcx                       /* rip    */
	popq %r11                       /* cs     */
	popq %r11                       /* rflags */
	popq %rsp                       /* rsp    */
	sysretq

syscall_vmexit:
	/* bounce hypercall to userspace */
	popq %rcx                       /* rip    */
	popq %r11                       /* cs     */
	popq %r11                       /* rflags */
	popq %rsp                       /* rsp    */
	out %al, $0xe0                  /* let userspace handle it */
	sysretq

syscall_iretq:
	/* return from syscall via iretq */
	iretq

/* helpers */

ENTRY(broken_memcpy_pf)
	mov %rdx,%rcx
	cld
1:	rep movsb
	xor %rax,%rax
8:	ret

	.section .exfix, "ax"
9:	mov $-1, %rax
	jmp 8b
	.previous

	.section .extab, "a"
	.align 8
	.quad 1b,9b
	.previous

/* some 16 bit code for smp boot */

	.code16
	.align 4096
ENTRY(sipi)
	mov $0x00060000, %eax  /* EMUDEV_CMD_INIT_SECONDARY_VCPU */
	outl %eax, $0xe8       /* EMUDEV_REG_COMMAND */
	hlt
	.code64

/* emu boot stack, including syscall trampoline template */

	.data
	.globl boot_stack_low, boot_stack_high
	.globl cpu_ptr
	.globl trampoline_start, trampoline_patch, trampoline_stop
	.align 4096
boot_stack_low:
cpu_ptr:
	.quad 0
trampoline_start:
	movq %rsp, boot_stack_high-16(%rip)
	leaq boot_stack_high-16(%rip), %rsp
	push %r11                           /* rflags         */
	mov  $0xdeadbeef,%r11               /* emu C code must fixup cs & ss */
	movq %r11, boot_stack_high-8(%rip)  /* ss             */
	push %r11                           /* cs             */
	push %rcx                           /* rip            */

	.byte 0x49,0xbb                     /* mov data, %r11 ...         */
trampoline_patch:
	.quad 0                             /* ... data, for jump to ...  */
	jmpq *%r11                          /* ... trampoline_syscall     */
trampoline_stop:
	.align 4096
boot_stack_high:

/* boot page tables */

#define pageflags 0x063 /* preset, rw, accessed, dirty */
#define largepage 0x080 /* pse */

	.section .pt.64, "a"
	.globl emu_pgd

	.align 4096
emu_pgd:
	.fill 262,8,0
	.quad emu_pud - 0xffff830000000000 + pageflags
	.fill 249,8,0

	.align 4096
emu_pud:
	.quad emu_pmd - 0xffff830000000000 + pageflags
	.fill 511,8,0

	.align 4096
emu_pmd:
	.quad pageflags + largepage
	.fill 511,8,0
