#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include "xenner.h"
#include "mm.h"

/* ------------------------------------------------------------------ */

struct symbol {
    struct symbol  *next;
    uint64_t       addr;
    char           name[];
};

static struct symbol *symbols[256];

int parse_asm(char *filename)
{
    struct symbol *sym;
    char line[256], symbol[128];
    uint64_t addr;
    int idx, count = 0;
    FILE *fp;

    fp = fopen(filename, "r");
    if (NULL == fp)
	return -1;

    while (NULL != fgets(line, sizeof(line), fp)) {
	if (2 != sscanf(line, "%" PRIx64 " <%128[^>]>:", &addr, symbol))
	    continue;
	sym = malloc(sizeof(*sym) + strlen(symbol) + 1);
	sym->addr = addr;
	strcpy(sym->name, symbol);
	idx = addr % sizeof(symbols)/sizeof(symbols[0]);
	sym->next = symbols[idx];
	symbols[idx] = sym;
	count++;
    }
    fprintf(stderr, "%s: %s: %d symbols\n", __FUNCTION__, filename, count);
    return 0;
}

static char *find_symbol(uint64_t addr)
{
    struct symbol *sym;
    int idx;

    idx = addr % sizeof(symbols)/sizeof(symbols[0]);
    for (sym = symbols[idx]; NULL != sym; sym = sym->next)
	if (sym->addr == addr)
	    return sym->name;
    return NULL;
}

int do_debug_trap(struct xenvcpu *vcpu)
{
    struct xenvm *xen = vcpu->vm;
    uint64_t *emu = NULL, *stack = NULL;
    uint64_t rip, rsp;
    char *sym;
    
    if (xen->mode != XENMODE_64)
	vm_kill(vcpu, "debug: no 32bit support", 0);
    
    need_regs(vcpu);
    need_sregs(vcpu);
    emu = emu_vaddr_to_ptr(xen, vcpu->regs.rsp);
    rip = emu[0];
    rsp = emu[3];
    stack = guest_vaddr_to_ptr(vcpu, rsp);

    sym = find_symbol(rip);
    d0printf("%s: rip %" PRIx64 " rsp %" PRIx64 
	     " [ code %04" PRIx64 ":%016" PRIx64 " flags %08" PRIx64
	     " stack %04" PRIx64 ":%016" PRIx64
	     " ] (%s)\n", __FUNCTION__,
	     rip, rsp,
	     stack[1], stack[0], stack[2], stack[4], stack[3],
	     sym);
    
    return 0;
}
