/* Copyright (c) 2005, Google Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ---
 * Author: Markus Gutschke
 */

#include <assert.h>
#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "google/coredumper.h"


/* This is a really silly CPU hog, but we want to avoid creating a
 * core dump while we are executing code in libc. Depending on the
 * system environment, gdb might or might not make stack traces
 * available within libc, and that would make this unittest
 * non-deterministic.
 */
static volatile enum State { IDLE, RUNNING, DEAD } state1, state2;
static volatile unsigned int counter;
static void *Busy(void *arg) {
  volatile enum State *state = (volatile enum State *)arg;
  *state = RUNNING;
  while (*state == RUNNING) {
    counter++;
  }
  return 0;
}


/* Do not declare this function static, so that the compiler does not get
 * tempted to inline it. We want to be able to see some stack traces.
 */
void TestCoreDump() {
  static const char *msg1[] = { " ELF", "little endian", "UNIX - System V",
                                "Core file", "no sections in this file",
                                "NOTE", "no dynamic se",
                                "no relocations in this file",
                                "no unwind sections in this file",
                                "No version information found in this file",
                                "NT_PRPSINFO", "NT_TASKSTRUCT",
                                "NT_PRSTATUS", "NT_FPREGSET",
                                "NT_PRSTATUS", "NT_FPREGSET",
                                "NT_PRSTATUS", "NT_FPREGSET",
                                "DONE", 0 };
  static const char *msg2[] = { "Core was generated by",
                                " in TestCoreDump",
                                "$1 = ",
                                " 3 process ", " 2 process ", " 1 process ",
                                "[Switching to thread 2",
                                "in Busy",
                                "DONE", 0 };
  int         in[2], out[2], dummy, cmp, rc;
  pid_t       pid;
  FILE        *input, *output;
  char        buffer[4096];
  const char  **ptr;
  pthread_t   thread;
  struct stat statBuf;

  /* Make stdout unbuffered. We absolutely want to see all output, even
   * if the application aborted with an assertion failure.
   */
  setvbuf(stdout, NULL, _IONBF, 0);

  /* It is rather tricky to properly call fork() from within a multi-threaded
   * application. To simplify this problem, we fork and exec /bin/bash before
   * creating the first thread.
   */
  puts("Forking /bin/bash process");
  rc = pipe(in);  assert(!rc);
  rc = pipe(out); assert(!rc);

  if ((pid = fork()) == 0) {
    int i, openmax;
    dup2(in[0],  0);
    dup2(out[1], 1);
    dup2(out[1], 2);
    openmax = sysconf(_SC_OPEN_MAX);
    for (i = 3; i < openmax; i++)
      close(i);
    fcntl(0, F_SETFD, 0);
    fcntl(1, F_SETFD, 0);
    fcntl(2, F_SETFD, 0);
    execl("/bin/bash", "bash", "-ex", NULL);
    _exit(1);
  }
  assert(pid >= 0);
  assert(!close(in[0]));
  assert(!close(out[1]));
  input  = fdopen(in[1], "w");
  output = fdopen(out[0], "r");
  setvbuf(input, NULL, _IONBF, 0);
  setvbuf(output, NULL, _IONBF, 0);

  /* Create a random value in one of our auto variables; we will later look
   * for this value by inspecting the core file with gdb.
   */
  srand(time(0));
  dummy = random();
  cmp   = ~dummy;

  /* Start some threads that should show up in our core dump; this is
   * complicated by the fact that we do not want our threads to perform any
   * system calls. So, they are busy looping and checking a volatile
   * state variable, instead.
   */
  puts("Starting threads");
  pthread_create(&thread, 0, Busy, (void *)&state1);
  pthread_create(&thread, 0, Busy, (void *)&state2);
  while (state1 != RUNNING || state2 != RUNNING) {
    usleep(100*1000);
  }

  /* Prepare to create a core dump for the current process                   */
  puts("Writing core file to \"core-test\"");
  unlink("core-test");

  /* Check whether limits work correctly                                     */
  rc = WriteCoreDumpLimited("core-test", 0);
  assert(!rc);
  assert(stat("core-test", &statBuf) < 0);
  rc = WriteCoreDumpLimited("core-test", 256);
  assert(!rc);
  assert(!stat("core-test", &statBuf));
  assert(statBuf.st_size == 256);

  /* Create a full-size core file                                            */
  rc = WriteCoreDump("core-test");
  assert(!rc);

  /* Stop our threads                                                        */
  puts("Stopping threads");
  state1 = DEAD;
  state2 = DEAD;

  /* Open the core file with "readelf", and check that all the expected
   * entries can be found. We are not checking exact numeric values, as these
   * might differ between runs, and it seems overkill recomputing them here.
   */
  rc = fprintf(input,
               "cat /proc/%d/maps &&"
               "readelf -a \"core-test\"; (set +x; echo DONE)\n",
               getpid());
  assert(rc > 0);

  *buffer = '\000';
  for (ptr = msg1; *ptr; ptr++) {
    do {
      char *line;
      assert(strncmp(buffer, "DONE", 4));
      line = fgets(buffer, sizeof(buffer), output);
      assert(line);
      fputs(buffer, stdout);
    } while (!strstr(buffer, *ptr));
    printf("Found: %s\n", *ptr);
  }

  /* Open the core dump with gdb, and check that the stack traces look
   * correct. Again, we are not checking for exact numeric values.
   *
   * We also extract the value of the "dummy" environment variable, and check
   * that it is correct.
   */
  rc = fprintf(input, "gdb /proc/%d/exe \"core-test\"; (set +x; echo DONE)\n"
               "bt\n"
               "up 3\n"
               "print *(unsigned int *)0x%lx\n"
               "print %dU\n"
               "print %dU\n"
               "info threads\n"
               "thread 2\n"
               "thread apply all bt\n"
               "quit\n",
               getpid(), (unsigned long)&dummy, cmp, dummy);
  assert(rc > 0);

  *buffer = '\000';
  for (ptr = msg2; *ptr; ptr++) {
    do {
      char *line, *arg;
      assert(strncmp(buffer, "DONE", 4));
      line = fgets(buffer, sizeof(buffer), output);
      assert(line);
      fputs(buffer, stdout);
      if ((arg = strstr(buffer, "$1 = ")) != NULL) {
        cmp = atoi(arg + 5);
      }
    } while (!strstr(buffer, *ptr));
    printf("Found: %s\n", *ptr);
  }

  assert(dummy == cmp);
  printf("Magic marker matches %d\n", dummy);

  /* Kill bash process                                                       */
  kill(SIGTERM, pid);
  fclose(input);
  fclose(output);

  /* Get rid of our temporary test file                                      */
  unlink("core-test");

  return;
}

int main(int argc, char *argv[]) {
  TestCoreDump();
  puts("PASS");
  return 0;
}
