/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "application.h"

#include <QSettings>
#include <QSessionManager>
#include <QMessageBox>
#include <QDir>
#include <QUrl>
#include <QDesktopServices>
#include <QLocale>
#include <QTranslator>
#include <QLibraryInfo>

#include "connectionmanager.h"
#include "viewmanager.h"
#include "mainwindow.h"
#include "configdata.h"
#include "iconloader.h"

#include <cstdlib>

#if defined( Q_WS_WIN )
#define _WIN32_IE 0x0400
#include <shlobj.h>
#endif

Application::Application( int& argc, char** argv ) : QApplication( argc, argv )
{
    Q_INIT_RESOURCE( icons );
    Q_INIT_RESOURCE( resources );

    initializeDefaultPaths();
    processArguments();

    if ( !m_configPath.isEmpty() )
        configData = new ConfigData( new QSettings( m_configPath, QSettings::IniFormat ), true );
    else
        configData = new ConfigData( new QSettings( "WebIssues", "WebIssues" ), false );

    QString language = configData->language();
    if ( language.isEmpty() )
        language = QLocale::system().name();
    QLocale::setDefault( QLocale( language ) );

    loadTranslation( "qt", true );
    loadTranslation( "webissues", false );

#if defined( Q_WS_WIN )
    setStyle( "windowsmodernstyle" );
#endif

    setWindowIcon( IconLoader::icon( "webissues" ) );
    setQuitOnLastWindowClosed( false );

    connectionManager = new ConnectionManager();

    m_mainWindow = new MainWindow();

    QSettings* settings = configData->settings();
    settings->beginGroup( "ShutdownState" );
    bool visible = settings->value( "Visible", true ).toBool();
    bool connected = settings->value( "Connected", false ).toBool();
    settings->endGroup();

    if ( !configData->dockWindow() || configData->showAtStartup() == RestoreAlways ||
        ( configData->showAtStartup() == RestoreAuto && visible ) )
        m_mainWindow->show();

    if ( configData->reconnectAtStartup() == RestoreAlways ||
        ( configData->reconnectAtStartup() == RestoreAuto && connected ) )
        m_mainWindow->reconnect();
}

Application::~Application()
{
    QSettings* settings = configData->settings();
    settings->beginGroup( "ShutdownState" );
    settings->setValue( "Visible", m_mainWindow->isVisible() );
    settings->setValue( "Connected", connectionManager->isConnected() );
    settings->endGroup();

    delete m_mainWindow;
    m_mainWindow = NULL;

    delete connectionManager;
    connectionManager = NULL;

    delete configData;
    configData = NULL;
}

void Application::commitData( QSessionManager& manager )
{
#if !defined( QT_NO_SESSIONMANAGER )
    if ( manager.allowsInteraction() ) {
        if ( viewManager && !viewManager->queryCloseViews() )
            manager.cancel();
    }
#endif
}

void Application::about()
{
    QString link = "<a href=\"http://webissues.mimec.org\">webissues.mimec.org</a>";

    QString message;
    message += "<h3>" + tr( "WebIssues %1" ).arg( version() ) + "</h3>";
    message += "<p>" + tr( "Desktop client for WebIssues, an issue tracking and team collaboration system." ) + "</p>";
    message += "<p>" + trUtf8( "Copyright (C) 2006 Michał Męciński" ) + "<br>" + tr( "Copyright (C) 2007-2009 WebIssues Team" ) + "</p>";
    message += "<p>" + tr( "This program is licensed under the GNU General Public License." ) + "</p>";
    message += "<p>" + tr( "Visit %1 for more information." ).arg( link ) + "</p>";

    QMessageBox::about( activeWindow(), tr( "About WebIssues" ), message );
}

QString Application::version() const
{
    return QString( "0.9.6" );
}

QUrl Application::manualIndex() const
{
    return QUrl::fromLocalFile( m_manualPath + "/index.html" );
}

void Application::manual()
{
    QDesktopServices::openUrl( manualIndex() );
}

bool Application::loadTranslation( const QString& name, bool tryQtDir )
{
    QString fullName = name + "_" + QLocale().name();

    QTranslator* translator = new QTranslator( this );

    if ( translator->load( fullName, m_translationsPath ) ) {
        installTranslator( translator );
        return true;
    }

    if ( tryQtDir && translator->load( fullName, QLibraryInfo::location( QLibraryInfo::TranslationsPath ) ) ) {
        installTranslator( translator );
        return true;
    }

    delete translator;
    return false;
}

void Application::initializeDefaultPaths()
{
    // NOTE: update these paths after changing them in the project file

    QString appPath = applicationDirPath();

#if defined( Q_WS_WIN )
    m_manualPath = QDir::cleanPath( appPath + "/../doc" );
    m_translationsPath = QDir::cleanPath( appPath + "/../translations" );
#elif defined( Q_WS_MAC )
    m_manualPath = QDir::cleanPath( appPath + "/../Resources/doc" );
    m_translationsPath = QDir::cleanPath( appPath + "/../Resources/translations" );
#else
    m_manualPath = QDir::cleanPath( appPath + "/../share/doc/webissues/doc" );
    m_translationsPath = QDir::cleanPath( appPath + "/../share/webissues/translations" );
#endif

    QString homePath = QDir::homePath();

#if defined( Q_WS_WIN )
    char appDataPath[ MAX_PATH ];
    if ( SHGetSpecialFolderPathA( 0, appDataPath, CSIDL_APPDATA, FALSE ) ) {
        m_dataPath = QDir::fromNativeSeparators( QString::fromLocal8Bit( appDataPath ) );
        m_dataPath += QLatin1String( "/WebIssues" );
    } else {
        m_dataPath = homePath + "/WebIssues";
    }

    char localAppDataPath[ MAX_PATH ];
    if ( SHGetSpecialFolderPathA( 0, localAppDataPath, CSIDL_LOCAL_APPDATA, FALSE ) ) {
        m_cachePath = QDir::fromNativeSeparators( QString::fromLocal8Bit( localAppDataPath ) );
        m_cachePath += QLatin1String( "/WebIssues" );
    } else {
        m_cachePath = m_dataPath + "/.cache";
    }
#else
    char* dataHome = getenv( "XDG_DATA_HOME" );
    if ( dataHome ) {
        if ( dataHome[ 0 ] == QLatin1Char( '/' ) )
            m_dataPath = dataHome;
        else
            m_dataPath = homePath + '/' + dataHome;
    } else {
        m_dataPath = homePath + "/.local/share";
    }

    m_dataPath += QLatin1String( "/webissues" );

    char* cacheHome = getenv( "XDG_CACHE_HOME" );
    if ( cacheHome ) {
        if ( cacheHome[ 0 ] == QLatin1Char( '/' ) )
            m_cachePath = cacheHome;
        else
            m_cachePath = homePath + '/' + cacheHome;
    } else {
        m_cachePath = homePath + "/.cache";
    }

    m_cachePath += QLatin1String( "/webissues" );
#endif

    m_storagePath = m_cachePath;
}

void Application::processArguments()
{
    QStringList args = arguments();

    for ( int i = 1; i < args.count(); i++ ) {
        QString arg = args.at( i );
        if ( arg == QLatin1String( "-config" ) ) {
            if ( i + 1 < args.count() && !args.at( i + 1 ).startsWith( '-' ) )
                m_configPath = QDir::fromNativeSeparators( args.at( ++i ) );
        } else if ( arg == QLatin1String( "-data" ) ) {
            if ( i + 1 < args.count() && !args.at( i + 1 ).startsWith( '-' ) )
                m_dataPath = QDir::fromNativeSeparators( args.at( ++i ) );
        } else if ( arg == QLatin1String( "-cache" ) ) {
            if ( i + 1 < args.count() && !args.at( i + 1 ).startsWith( '-' ) )
                m_cachePath = QDir::fromNativeSeparators( args.at( ++i ) );
        } else if ( arg == QLatin1String( "-storage" ) ) {
            if ( i + 1 < args.count() && !args.at( i + 1 ).startsWith( '-' ) )
                m_storagePath = QDir::fromNativeSeparators( args.at( ++i ) );
        }
    }
}
