#!/usr/bin/env perl

#-------------------------------------------------------------------
# WebGUI is Copyright 2001-2009 Plain Black Corporation.
#-------------------------------------------------------------------
# Please read the legal notices (docs/legal.txt) and the license
# (docs/license.txt) that came with this distribution before using
# this software.
#-------------------------------------------------------------------
# http://www.plainblack.com                     info@plainblack.com
#-------------------------------------------------------------------

our ($webguiRoot);

BEGIN {
    $webguiRoot = "../..";
    unshift (@INC, $webguiRoot."/lib");
}

use strict;
use Getopt::Long;
use WebGUI::Session;
use WebGUI::Storage;
use WebGUI::Asset;
use WebGUI::ProfileField;


my $toVersion = '7.9.9';
my $quiet; # this line required


my $session = start(); # this line required

# upgrade functions go here
addIndexToUserSessionLog($session);
addHeightToCarousel($session);
synchronizeUserProfileTables($session);

finish($session); # this line required


#----------------------------------------------------------------------------
# Describe what our function does
#sub exampleFunction {
#    my $session = shift;
#    print "\tWe're doing some stuff here that you should know about... " unless $quiet;
#    # and here's our code
#    print "DONE!\n" unless $quiet;
#}

#----------------------------------------------------------------------------
# Describe what our function does
sub addIndexToUserSessionLog {
    my $session = shift;
    print "\tAdd index to UserSessionLogTable... " unless $quiet;
    $session->db->write(q|alter table userLoginLog add index sessionId (sessionId)|);
    print "DONE!\n" unless $quiet;
}

#----------------------------------------------------------------------------
# Describe what our function does
sub addHeightToCarousel {
    my $session = shift;
    print "\tAdd slide height to Carousel... " unless $quiet;
    $session->db->write(q|alter table Carousel add column slideHeight int(11)|);
    print "DONE!\n" unless $quiet;
}


#----------------------------------------------------------------------------
# Describe what our function does
sub synchronizeUserProfileTables {
    my $session = shift;
    print "\tMake sure that userProfileField, and userProfileData tables are aligned correctly... " unless $quiet;
    my $dbh         = $session->db->dbh;
    my $fields = WebGUI::ProfileField->getFields($session);
    foreach my $field ( @{ $fields } ) {
        my $columnInfo = $dbh->column_info(undef, undef, 'userProfileData', $field->getId)->fetchrow_hashref();
        if (! $columnInfo) {
            printf "\n\t\tDeleting broken field: %s", $field->getId;
            $session->db->deleteRow('userProfileField', 'fieldName', $field->getId);
        }
    }

    print "  ...DONE!\n" unless $quiet;
}

#----------------------------------------------------------------------------
# Move Template attachments to JSON collateral
sub migrateAttachmentsToJson {
    my $session = shift;
    print "\tMoving template attachments to JSON... " unless $quiet;
    # and here's our code
    $session->db->write(
        "ALTER TABLE template ADD attachmentsJson LONGTEXT"
    );

    my $attach;     # hashref (template) of hashrefs (revisionDate)
                    # of arrayrefs (attachments) of hashrefs (attachment)
    my $sth = $session->db->read( "SELECT * FROM template_attachments" );
    while ( my $row = $sth->hashRef ) {
        push @{ $attach->{ $row->{templateId} }{ $row->{revisionDate} } }, {
            url         => $row->{url},
            type        => $row->{type},
        };
    }

    for my $templateId ( keys %{ $attach } ) {
        for my $revisionDate ( keys %{ $attach->{$templateId} } ) {
            my $data    = $attach->{$templateId}{$revisionDate};
            my $asset   = WebGUI::Asset->newByDynamicClass( $session, $templateId, $revisionDate );
            $asset->update({ attachmentsJson => JSON->new->encode( $data ) });
        }
    }

    $session->db->write(
        "DROP TABLE template_attachments"
    );

    print "DONE!\n" unless $quiet;
}

# -------------- DO NOT EDIT BELOW THIS LINE --------------------------------

#----------------------------------------------------------------------------
# Add a package to the import node
sub addPackage {
    my $session     = shift;
    my $file        = shift;

    print "\tUpgrading package $file\n" unless $quiet;
    # Make a storage location for the package
    my $storage     = WebGUI::Storage->createTemp( $session );
    $storage->addFileFromFilesystem( $file );

    # Import the package into the import node
    my $package = eval {
        my $node = WebGUI::Asset->getImportNode($session);
        $node->importPackage( $storage, {
            overwriteLatest    => 1,
            clearPackageFlag   => 1,
            setDefaultTemplate => 1,
        } );
    };

    if ($package eq 'corrupt') {
        die "Corrupt package found in $file.  Stopping upgrade.\n";
    }
    if ($@ || !defined $package) {
        die "Error during package import on $file: $@\nStopping upgrade\n.";
    }

    return;
}

#-------------------------------------------------
sub start {
    my $configFile;
    $|=1; #disable output buffering
    GetOptions(
        'configFile=s'=>\$configFile,
        'quiet'=>\$quiet
    );
    my $session = WebGUI::Session->open($webguiRoot,$configFile);
    $session->user({userId=>3});
    my $versionTag = WebGUI::VersionTag->getWorking($session);
    $versionTag->set({name=>"Upgrade to ".$toVersion});
    return $session;
}

#-------------------------------------------------
sub finish {
    my $session = shift;
    updateTemplates($session);
    migrateAttachmentsToJson( $session );
    my $versionTag = WebGUI::VersionTag->getWorking($session);
    $versionTag->commit;
    $session->db->write("insert into webguiVersion values (".$session->db->quote($toVersion).",'upgrade',".time().")");
    $session->close();
}

#-------------------------------------------------
sub updateTemplates {
    my $session = shift;
    return undef unless (-d "packages-".$toVersion);
    print "\tUpdating packages.\n" unless ($quiet);
    opendir(DIR,"packages-".$toVersion);
    my @files = readdir(DIR);
    closedir(DIR);
    my $newFolder = undef;
    foreach my $file (@files) {
        next unless ($file =~ /\.wgpkg$/);
        # Fix the filename to include a path
        $file       = "packages-" . $toVersion . "/" . $file;
        addPackage( $session, $file );
    }
}

#vim:ft=perl
