#!/bin/bash
VERSION='0.4.0'
#
# Copyright (C) 2003-2007  Ola Lundqvist <opal@debian.org>
# Copyright (C) 2007       Jochen Voss <voss@seehuhn.de>
# Copyright (C) 2006       Miroslav Kure <kurem@upcase.inf.upol.cz>
# Copyright (C) 2002       Mark Lawrence <nomad@null.net>
# Copyright (C) 2002       Paul Sladen   <vserver@paul.sladen.org>
#
# This is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2, or (at your option) any later
# version.
# 
# This is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
# 
# You should have received a copy of the GNU General Public License with
# the source package as the file COPYING.  If not, write to the Free
# Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
# MA 02110-1301, USA.
#
# Install a virtual debian server (vserver) from a debian HTTP/FTP archive
#
# ----------------------------------------------------------------
# Configurable items:

shopt -s extglob

# No context id set by default
CONTEXT=

# Root directory of your virtual servers (probably shouldn't change this)
# Support for 0.30.203 and above of util-vserver
VROOTDIR="/etc/vservers/.defaults/vdirbase"

# Packages to install in addition to the base defaults
# MUST INCLUDE ALL DEPENDENCIES (seperated by "," commas)
# Devfsd is very important.
INSTALL_PACKAGES="nfsbooted"

if [ -n "$LANG" ] && [ "$LANG" != "C" ] ; then
    INSTALL_PACKAGES="$INSTALL_PACKAGES locales"
fi

# Packages installed from within the vserver after it has
# been setup (seperated by "," commas)
EXTRA_PACKAGES="nfsbooted"

# Packages to remove from the base defaults (seperated by "," commas)
REMOVE_PACKAGES="lilo,makedev,nfs-kernel-server,lpr"

# sysvinit services relating to hardware access to remove
#REMOVE_LINKS="klogd hwclock.sh setserial urandom networking umountfs halt reboot"

# Architecture: overide on non-Debian host such as Redhat
# otherwise dpkg will detect whether we are i386/powerpc/sparc/etc
ARCH=""

# Which debian distribution
# (warning: this has only been tested with woody, sarge and etch)
DIST="etch"

# Local or nearest location of a debian mirror (must include the `/debian')
MIRROR="http://ftp.uk.debian.org/debian"

# debian-non-US mirror (must include the '/debian-non-US', only used for woody)
MIRROR_NON_US="http://non-us.debian.org/debian-non-US"

# mirror for security updates
MIRROR_SECURITY="http://security.debian.org"

# Default network interface for vservers:
INTERFACE="eth0"

if [ -r /etc/vservers/util-vserver-vars ] ; then
    . /etc/vservers/util-vserver-vars
fi

# The name of a debconf database to be read by debconf's file driver.
# This can be used to store the answers for debconf questions during
# install and thus allows for unattended installation.  See the
# debconf.conf(5) manual page for the file format.
DEBCONF_FILE_DB=""

if [ -r /etc/vservers/newvserver-vars ] ; then
    . /etc/vservers/newvserver-vars    
fi

# Support for 0.30.203 and above of util-vserver
if [ -n "$DEFAULT_VSERVERDIR" ] ; then
    VROOTDIR="$DEFAULT_VSERVERDIR"
fi

# ----------------------------------------------------------------
# Nothing from here on should need changing.
# ----------------------------------------------------------------

# NOTE: debootstrap handles multiple MIRRORs, so there is no reason why
# we shouldn't too--that way we could just let it build /etc/apt/sources.list

usage ()
{
    cat << EOF 1>&2
usage:	${0##*/} [OPTIONS] --hostname x --domain y.z --ip 1.2.3.4
	(see --help for more information)
EOF
}
full_usage ()
{
	cat << EOF
Usage:	${0##*/} [OPTIONS] --hostname x --domain y.z --ip 1.2.3.4 --nfs-server 2.3.4.5
Creates a new Debian vserver by downloading packages via HTTP/FTP

Options:
        -h, --help		this help
        -V, --version		copyright and version information
	--arch			set target architecture (eg. --arch "i386")
				(autodetected on Debian host if dpkg available)
	--dist			defaults to "etch", passed to debootstrap.
	--context n             Set the context id to be used.
	--fakeinit		use "/sbin/init" to boot vserver
	--conffile		extra configuration file to load.
	--interface		interface for IP addresses (if not "eth0")
	--mirror		Debian HTTP/FTP mirror (including the /debian)
	--sshkeys		copy pub-keys to "/root/.ssh/authorized_keys"
	-v, --verbose		show extra output during setup
	--vsroot		location of "/vserver/" directory
Required:
        --nfs-server            The ip of the nfs server to use.
	--hostname		hostname for new vserver (eg. "alpha")
	--domain		dns domain for new vserver (eg. "example.com")
	--ip			IPv4 address for new vserver
				(syntax: --ip <ip>[/<netmask|prefixlength>])

	Either --ip or --domain/--hostname may be omitted if the
	corresponding information can be found via a name server
	query. For this to work the dnsutils package must be installed.

You can also set variables in /etc/vservers/newvserver-vars.

EOF
}
full_version ()
{
    cat << EOF
${0##*/} version $VERSION
Copyright (C) 2003-2007  Ola Lundqvist <opal@debian.org>
Copyright (C) 2007       Jochen Voss <voss@seehuhn.de>
Copyright (C) 2006       Miroslav Kure <kurem@upcase.inf.upol.cz>
Copyright (C) 2002       Mark Lawrence <nomad@null.net>
Copyright (C) 2002       Paul Sladen   <vserver@paul.sladen.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

EOF
}
parse_args ()
{
	while [ $# -gt 0 ] ; do
		#echo "parse_args() doing :$#: :$1: :$*:"
		case "$1" in
		    --help|-h)
			full_usage
			shift
			exit 0
			;;
		    --version|-V)
			full_version
			shift
			exit 0
			;;
		    --arch)
			case "$2" in
			    [a-z]*)
				ARCH="$2"
				;;
			    *)
				echo "${0##*/} error: $1 overrides dpkg autodetect on non-Debian host-servers" 1>&2
				echo 'e.g. "i386"' 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    --context)
		        CONTEXT=$2
			shift 2
			;;
		    --conffile)
		        if [ -r "$2" ] ; then
			    . "$2"
			else
			    echo "Error, can not load config file $2."
			    exit 1
			fi
			shift 2
			;;
		    --nfs-server)
		        NFS_SERVER=$2
			shift 2
			;;
		    --dist)
			case "$2" in
			    [a-z]*)
				DIST="$2"
				if [ "sarge" != "$2" ] && [ "woody" != "$2" ] && [ "etch" != "$2" ] ; then
				    echo "${0##*/} warning: I only know how to do \"woody\", \"sarge\" and \"etch\", be careful!" 1>&2
				fi
				;;
			    *)
				echo "${0##*/} error: $1 requires a Debian distribution" 1>&2
				echo 'e.g. "woody", "sarge" or "etch"' 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    --domain)
			case "$2" in
			    [a-z]*[a-z])
				VDOMAIN="$2"
				;;
			    *)
				echo "${0##*/} error: $1 requires a dns domain-name" 1>&2
				echo 'e.g. "example.com"' 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    --fakeinit)
		        # Note space at beginning--this gets tagged straight on
			FAKEINIT=" fakeinit"
			shift
			;;
		    --hostname)
			case "$2" in
			    [a-z0-9]*([a-z0-9_-])[a-z0-9])
				VHOST="$2"
				;;
			    *)
				echo "${0##*/} error: $1 must be a hostname for the vserver" 1>&2
				echo 'e.g. "alpha"' 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    --interface)
			case "$2" in
			    [a-z]*)
				INTERFACE="$2"
				;;
			    *)
				echo "${0##*/} error: $1 must be followed by a network interface" 1>&2
				echo 'e.g. "eth1"' 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    --ip)
			# This does for an octet: ([0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]) ;-)
			case "$2" in
			    [0-9]*.[0-9]*.[0-9]*.[0-9]*)
				IP="$2"
				;;
			    *)
				echo "${0##*/} error: $1 requires a single IPv4  e.g. \"192.168.100.1\"" 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    --mirror)
			case "$2" in
			    [hf]*:/*)
				MIRROR="$2"
				;;
			    *)
				echo "${0##*/} error: $1 requires a debian mirror"  1>&2
				echo 'e.g. "http://ftp.uk.debian.org/debian"' 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    --verbose|-v)
			export verbose="true"
			shift
			;;
		    --sshkeys)
			if [ -f "$2" ]; then
			    SSH_KEYS="$2"
			else
			    echo "${0##*/} error: $1 must be followed by a valid public-key-file!" 1>&2
			    echo 'e.g. "/root/.ssh/id_dsa.pub"' 1>&2
			    exit 1
			fi
			shift 2
			;;
		    --vsroot)
			case "$2" in
			    /*)
				if [ -d "$2" ]; then
				    VROOTDIR="$2"
				else
				    echo "${0##*/} error: $1 needs a valid absolute directory"  1>&2
				    echo 'e.g. "/vservers or /etc/vservers/.defaults/vdirbase"' 1>&2
				    exit 1
				fi
				;;
			    *)
				echo "${0##*/} error: $1 needs a valid absolute directory"  1>&2
				echo 'e.g. "/vservers or /etc/vservers/.defaults/vdirbase"' 1>&2
				exit 1
				;;
			esac
			shift 2
			;;
		    -*)
			usage
			exit 1
			;;
		    ?*)
			usage
			exit 1
			;;
		esac
	done
}

parse_args $@

if [ -x "$(which dig)" ] ; then
    if [ -n "$IP" -a \( -z "$VHOST" -o -z "$VDOMAIN" \) ]; then
	tmp=$(dig +short -x $(echo $IP | sed 's;/[0-9]*$;;'))
	if [ -n "$tmp" ]; then
	    if [ -z "$VHOST" ]; then
		VHOST=$(echo $tmp | sed 's/\..*//')
	    fi
	    if [ -z "$VDOMAIN" ]; then
		VDOMAIN=$(echo $tmp | sed 's/[^.]*\.//')
	    fi
	fi
    fi
    
    if [ -n "$VHOST" -a -n "$VDOMAIN" -a -n "$INTERFACE" -a -z "$IP" ]; then
	prefix=$(ip -o addr show dev "$INTERFACE" primary scope global | \
	    sed -n 's;.*inet [0-9.]*/\([0-9]*\).*;\1;p')
	IP="$(dig +short "$VHOST.$VDOMAIN")/$prefix"
    fi
    
    if ! [ -n "$VHOST" -a -n "$VDOMAIN" -a -n "$IP" ]; then
	echo "${0##*/} error: --hostname, --domain and --ip are required" 1>&2
	usage
	exit 1
    fi
fi


if ! [ -n "$VHOST" -a -n "$VDOMAIN" -a -n "$IP" -a -n "$NFS_SERVER" ]; then
    echo "${0##*/} error: --hostname, --domain, --ip and --nfs-server are required" 1>&2
    usage
    exit 1
fi

IP_ADDR=${IP%%/*}

# Strip final slashes off a couple of things
MIRROR="${MIRROR%/}"
MIRROR_NON_US="${MIRROR_NON_US%/}"
MIRROR_SECURITY="${MIRROR_SECURITY%/}"
VROOTDIR="${VROOTDIR%/}"

# Check we've got debootstrap available
if [ ! -x /usr/sbin/debootstrap ]; then
  cat << EOF 1>&2
${0##*/}: Requires the debootstrap package to bootstrap Debian
  Debian Host:	apt-get install debootstrap
  RPM Host:	rpm -i http://people.debian.org/~blade/install/debootstrap/debootstrap-0.1.17.3-2.i386.rpm
EOF
  exit 1
fi

if ! cat /proc/self/status | grep '^\(VxID:[^0-9]0\|s_context:[^0-9]0\)$'; then
    echo "${0##*/} error:"
    echo "	Must be run from the host server (security context 0)" 1>&2
    echo '	on a "vserver/ctx-patch" enabled kernel' 1>&2
    echo '	See: http://www.solucorp.qc.ca/miscprj/s_context.hc' 1>&2
    exit 1
fi

if [ -x /usr/bin/id ] && [ `id -u` -ne 0 ]; then
  echo "${0##*/} error: Must be run as root!" 1>&2
  exit 1
fi

# check for /vserver/$VHOST/etc/ incase we are on pre-mounted LVM partition
# (used to just check for "/vserver/$VHOST/" existing
if [ -d "$VROOTDIR/$VHOST/etc/" -o -f "/etc/vservers/$VHOST.conf" ] ; then
  cat << EOF 1>&2
${0##*/} error: Virtual Server "$VHOST" appears to already exist
  check "/etc/vservers/$VHOST.conf" or "/vservers/$VHOST/etc/";
EOF
  exit 1
fi

# This is used to keep a cache of the downloaded .deb packges for next install
if [ -d "$VROOTDIR/ARCHIVES/$DIST" ]; then
    mkdir -p "$VROOTDIR/$VHOST/var/cache/apt/archives"
    cp -a "$VROOTDIR/ARCHIVES/$DIST/"*.deb "$VROOTDIR/$VHOST/var/cache/apt/archives"
fi

# We only want to pass the Architecture if we need to (autodectected otherwise)
if [ -n "$ARCH" ]; then
    ARCH_ARGUMENT="--arch $ARCH"
fi

# Function to optionally set a context id.
CONTEXT_ARGUMENT=
if [ -n "$CONTEXT" ] ; then
    CONTEXT_ARGUMENT="--context $CONTEXT"
fi

## use "vserver ... build" to build the new vserver
if ! /usr/sbin/vserver "$VHOST" build -m debootstrap \
       --rootdir "$VROOTDIR" --hostname "$VHOST" --interface "$INTERFACE:$IP" \
       $CONTEXT_ARGUMENT \
       -- -d "$DIST" -m "$MIRROR" \
       -- $ARCH_ARGUMENT \
       --include="$INSTALL_PACKAGES" --exclude="$REMOVE_PACKAGES"
  then
    echo "${0##*/}: error: vserver-build failure. Cannot continue."
    exit 1
fi

# Create a locale.gen if needed.
if [ -n "$LANG" ] && [ "$LANG" != "C" ] ; then
    if [ ! -e "$VROOTDIR/$VHOST/etc/locale.gen" ] ; then
	echo $LANG $(locale charmap) > "$VROOTDIR/$VHOST/etc/locale.gen"
    elif ! grep $LANG "$VROOTDIR/$VHOST/etc/locale.gen" > /dev/null ; then
	echo $LANG $(locale charmap) >> "$VROOTDIR/$VHOST/etc/locale.gen"
    fi
fi

# Make it so that apt and friends work
if [ "woody" == "$DIST" ] ; then
    cat << EOF > "$VROOTDIR/$VHOST/etc/apt/sources.list"
deb $MIRROR/ $DIST main
deb-src $MIRROR/ $DIST main

deb $MIRROR_NON_US $DIST/non-US main
deb-src $MIRROR_NON_US $DIST/non-US main

deb $MIRROR_SECURITY $DIST/updates main

EOF
else
       cat << EOF > "$VROOTDIR/$VHOST/etc/apt/sources.list"
deb $MIRROR/ $DIST main
deb-src $MIRROR/ $DIST main

deb $MIRROR_SECURITY $DIST/updates main

EOF
fi

# Fix up the available device nodes, for security
if cd "$VROOTDIR/$VHOST/dev"; then
    tar cfp /var/cache/vserver/dev.tar.$$ full null ptmx random tty urandom zero
    rm -rf "$VROOTDIR/$VHOST/dev/"*
    tar xfp /var/cache/vserver/dev.tar.$$
    rm /var/cache/vserver/dev.tar.$$
    mkdir pts
    mkdir shm
fi

# Give the new host a hostname
echo "$VHOST" > "$VROOTDIR/$VHOST/etc/hostname"

# Set up the /etc/hosts file (needed for some parts of the base-config)
cat << EOF > "$VROOTDIR/$VHOST/etc/hosts"
# /etc/hosts

127.0.0.1	localhost
$IP_ADDR	$VHOST.$VDOMAIN $VHOST

# The following lines are desirable for IPv6 capable hosts
# (added automatically by netbase upgrade)

::1	ip6-localhost ip6-loopback
fe00::0 ip6-localnet
ff00::0 ip6-mcastprefix
ff02::1 ip6-allnodes
ff02::2 ip6-allrouters
ff02::3 ip6-allhosts

EOF

# Set up network interface
cat <<EOF > "$VROOTDIR/$VHOST/etc/hosts"
# /etc/network/interfaces -- configuration file for ifup(8), ifdown(8)

# The loopback interface
auto lo
iface lo inet loopback

# Do NEVER add the interface that you boot from here!

EOF

# grab DNS servers from the host-server for `resolv.conf'
HOST_IP=$(ip -o addr show dev $INTERFACE primary scope global | \
          sed -n 's;.*inet \([^/]*\)/.*;\1;p')
(echo search $VDOMAIN;
 grep '^nameserver' /etc/resolv.conf | sed "s/127.0.0.1/$HOST_IP/") \
    > "$VROOTDIR/$VHOST/etc/resolv.conf"

# If there is a proxy server statement in-use in the Host server, copy it across
if [ -f /etc/apt/apt.conf ]; then
    cp /etc/apt/apt.conf $VROOTDIR/$VHOST/etc/apt/apt.conf
fi

# support the apt.conf.d directories
if [ -d /etc/apt/apt.conf.d ] ; then
    cp -a /etc/apt/apt.conf.d/* $VROOTDIR/$VHOST/etc/apt/apt.conf.d/ > /dev/null 2>&1
fi

mkdir -p $VROOTDIR/$VHOST/nfsroot

# Create a dummy fstab
cat << EOF > "$VROOTDIR/$VHOST/etc/fstab"
# /etc/fstab: static file system information.
#
# <file system> <mount point>	<type>	<options>			<dump>	<pass>
$NFS_SERVER:$VROOTDIR/$VHOST /   nfs defaults,rw,nfsvers=3		0       0
$NFS_SERVER:$VROOTDIR/$VHOST /nfsroot nfs  defaults,ro,nfsvers=3	0       0
$NFS_SERVER:$VROOTDIR/usr    /usr   nfs defaults,ro,nfsvers=3	0	0
#$NFS_SERVER:$VROOTDIR/etc    /etc   nfs defaults,rw,nfsvers=3	0	0
$NFS_SERVER:$VROOTDIR/lib    /lib   nfs defaults,ro,nfsvers=3	0	0
$NFS_SERVER:$VROOTDIR/boot   /boot   nfs defaults,ro,nfsvers=3	0	0
$NFS_SERVER:$VROOTDIR/bin    /bin   nfs defaults,ro,nfsvers=3	0	0
$NFS_SERVER:$VROOTDIR/sbin   /sbin   nfs defaults,ro,nfsvers=3	0	0
$NFS_SERVER:$VROOTDIR/opt   /opt   nfs defaults,ro,nfsvers=3	0	0
proc		/proc		proc	defaults			0	0
EOF

# The new vserver patch now automatically mounts /proc
# Debian needs /dev/pts mounting "gid=5,mode=620" by vserver
cat << EOF > "$VROOTDIR/$VHOST/etc/mtab"
/dev/hdv1 / vfs none 0 0
proc /proc proc rw 0 0
devpts /dev/pts devpts rw,gid=5,mode=620 0 0
EOF

# Create default /etc/vservers entry
cat << EOF > /etc/vservers/$VHOST.conf
S_HOSTNAME="$VHOST"
IPROOT="$IP"
IPROOTDEV="$INTERFACE"
ONBOOT="yes"
S_NICE=""
S_FLAGS="lock nproc$FAKEINIT"
ULIMIT="-H -u 256 -n 1024"
S_CAPS="CAP_NET_RAW"

# *NOT* DNS domain name, for NIS only
S_DOMAINNAME=""

EOF

if [ -n "$EXTRA_PACKAGES" ]; then
    EXTRA_PACKAGES_INSTALL="apt-get --assume-yes install ${EXTRA_PACKAGES//,/ }"
fi

if [ -f /etc/timezone ]; then
    tz=$(cat /etc/timezone)
    zonefile="$VROOTDIR/$VHOST/usr/share/zoneinfo/$tz"
    if [ -f "$zonefile" ]; then
	echo "$tz" >$VROOTDIR/$VHOST/etc/timezone
	cp "$zonefile" "$VROOTDIR/$VHOST/etc/localtime"
    fi
fi

if [ -n "$DEBCONF_FILE_DB" ]; then
    cp "$DEBCONF_FILE_DB" "$VROOTDIR/$VHOST/config.dat"
    SET_DEBCONF_OVERRIDE='export DEBCONF_DB_OVERRIDE="File{/config.dat}"'
fi

# ------------------------------------------------------------
# From here on we do things live in the server
# Generate the script that runs the rest of the setup from within the
# virtual server.
cat << EOF > $VROOTDIR/$VHOST/vserver-config.sh
#!/bin/sh

$SET_DEBCONF_OVERRIDE

[ -x /usr/sbin/locale-gen ] && /usr/sbin/locale-gen

dselect update

if [ ! -f /etc/timezone ]; then
    # tzsetup was part of base-config which is gone since etch
    # tzconfig is part of libc, so it should be ubiquitious
    if [ -x /usr/sbin/tzsetup ]; then
	/usr/sbin/tzsetup -y
    elif [ -x /usr/sbin/tzconfig ]; then
	/usr/sbin/tzconfig
    fi
fi

dpkg-reconfigure -u passwd

tasksel

rm -f /etc/exim/exim.conf
eximconfig

# because the --exclude flag doesn\'t seem to work on debootstrap
apt-get --assume-yes --purge remove `echo $REMOVE_PACKAGES | sed -e 's/,/ /g'`

for link in $REMOVE_LINKS
do
    update-rc.d -f \$link remove
done

$EXTRA_PACKAGES_INSTALL

EOF

# Run the above commands from within the server
chmod 755 $VROOTDIR/$VHOST/vserver-config.sh
vserver $VHOST exec /vserver-config.sh
rm -f $VROOTDIR/$VHOST/vserver-config.sh
rm -f "$VROOTDIR/$VHOST/config.dat"
rm -f "$VROOTDIR/$VHOST/config.dat-old"

# If you need to install your SSH management keys into the vserver
if [ -f "$SSH_KEYS" ]; then
    mkdir -p "$VROOTDIR/$VHOST/root/.ssh"
    chmod 700 "$VROOTDIR/$VHOST/root/.ssh/"
    cat "$SSH_KEYS" >> "$VROOTDIR/$VHOST/root/.ssh/authorized_keys"
    chmod 600 "$VROOTDIR/$VHOST/root/.ssh/authorized_keys"
fi

# Stop all the processes that were started inside the server
export PREVLEVEL=2
vserver $VHOST exec /etc/init.d/rc 0
vserver $VHOST stop

cat $VROOTDIR/$VHOST/etc/nfsbooted/fstab >> $VROOTDIR/$VHOST/etc/fstab

# Populate the archive for future virtual servers
if [ ! -d $VROOTDIR/ARCHIVES/$DIST ]; then
    mkdir -p $VROOTDIR/ARCHIVES/$DIST
fi
cp $VROOTDIR/$VHOST/var/cache/apt/archives/*.deb $VROOTDIR/ARCHIVES/$DIST

echo "Restarting rebootmgr."
invoke-rc.d rebootmgr restart
echo
echo "You should now adjust /etc/vservers/$VHOST.conf to suit your needs,"
echo "or else just go ahead and type \`vserver $VHOST start' to start"
echo "your new virtual server.  debian/rules!"
echo
echo "Also make sure that /var/lib/vservers/ is exported from /etc/exports."
echo
