// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        clipping.h
//
// Purpose:     Interface to class Clipping containing clipping
//              algorithms and tools.
//
// Created:      8 Feb 1996  Georg Meszaros
//
// Modified:    15 Jun 1996  Georg Meszaros 
//
//
// Description:
//
// Declaration of class Clipping.
// 
//
//</file>


#ifndef harmony_scene_clipping_h
#define harmony_scene_clipping_h


//<class>
//
// Name:    Clipping     
//
// Purpose: clip a polygon against a plane
//
// Public Interface:
/*
   - static void print3D(const point3D& p)
     > Just a tool to print a point3D or a vertex3D to cerr.

   - static int inFront(const point3D& vertex, 
                        const vector3D& normal, 
                        double plane)
     Calculates if the vertex lies in front of the plane
     described by the face normal vector and d of a plane equation
     ax + by + cz = d = plane.
 
   - static int inBack(const point3D& vertex, 
                       const vector3D& normal, 
                       double plane)
     Calculates if the vertex lies behind the plane
     described by the face normal vector and d of a plane equation
     ax + by + cz = d = plane.
  
   - static int onPlane(const point3D& vertex, 
                        const vector3D& normal, 
                        double plane)
     Calculates if the vertex lies on the plane
     described by the face normal vector and d of a plane equation
     ax + by + cz = d = plane. A margin is used to reference points
     very close to the plane as "on the plane" to deal with errors in
     the mathematical precision.

   - static void clip(Face& face, const vector3D& normal, float plane, 
                       Face*& front_face, Face*& back_face, SplitType& split_type);
  
     Calculates the parts of the face that lie in the front or in the
     back of the plane described by the face normal vector and 
     d of a plane equation ax + by + cz = d = plane. 
     If the entire face is infront or behind the plane then no new
     face is created. Otherwise 2 new faces are created.     


   - const double margin_ = 0.01;
     This margin is used to overcome errors of mathematical
     calculation to decide if a point lies on a plane.

*/
// Description:
// Everything needed to implement the Sutherland-Hodgeman Polygon-Clipping Algorithm.
// Foley, van Dam !
//
//</class>

enum PolygonSide { FRONT, BACK };
enum SplitType { SAME_PLANE, ONLY_FRONT, ONLY_BACK, SPLIT };

#include <ge3d/vectors.h>
#include <ge3d/color.h>

class Face;

class Clipping
{
  public:

    static void printRGB(const colorRGB& c);

    static void print3D(const point3D& p);

    static int inFront(const point3D& vertex, const vector3D& normal, float plane);

    static int inBack(const point3D& vertex, const vector3D& normal, float plane);

    static int onPlane(const point3D& vertex, const vector3D& normal, float plane);

    static void clip(Face& face, const vector3D& normal, float plane, 
                     Face*& front_face, Face*& back_face, SplitType& split_type);

  private:

    static point3D* intersection(const point3D& vertex_1, const point3D& vertex_2,
                                 const vector3D& normal, float plane,
                                 const vector3D& normal_1, const vector3D& normal_2,
                                 vector3D& new_normal);

    static point3D* intersection(const point3D& vertex_1, const point3D& vertex_2,
                                 const vector3D& normal, float plane);

    static Face* split(Face& face, 
                       const vector3D& normal, float plane, 
                       PolygonSide whichside); 
   
};

#endif
