/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file Session.hpp
 */

#ifndef SESSION_HPP
#define SESSION_HPP

class Palette;
class Color;
class QByteArray;
class QColor;
class QString;
class QStringList;

/*!
 * \def ORG_NAME
 * \brief The "organization" name
 */
#define ORG_NAME "ViTE"
/*!
 * \def ORG_DOMAIN
 * \brief The "organization" domain
 */
#define ORG_DOMAIN "vite.gforge.inria.com"
/*!
 * \def APP_NAME
 * \brief The application name
 */
#define APP_NAME "ViTE"

/*!
 * \def RECENT_FILES
 * \brief The recent file key for Settings
 */
#define RECENT_FILES "recentFiles"
/*!
 * \def PLUGIN_DIR
 * \brief The plugin directory key for Settings
 */
#define PLUGIN_DIR "plugin_dir"

/*!
 * \def CUR_PALETTE
 * \brief The current palette key for Settings
 */
#define CUR_PALETTE "palette/current_name"
/*!
 * \def PALETTE_NAMES
 * \brief The current palette names key for Settings
 */
#define PALETTE_NAMES "palette/palettes_name"

/*!
 * \brief Class used to store information between two software launch.
 *
 */
class Session{

private:

    static Palette *_palette;
    /*!
     * \brief The class constructor.
     */
    Session();

public:

    /*!
     * \brief The class destructor.
     */
    virtual ~Session();
  
    /*!
     * \brief Sub structure to store export information.
     */
    struct Export{

        /*!
         * \brief Store the export file dialog state.
         */
        static QByteArray file_dialog_state;

        /*!
         * \brief Default value of the export dialog state.
         */
        const static QByteArray DEFAULT_STATE;
    };

    /*!
     * \brief Sub structure to store render information.
     */
    struct Render{
        
    };

    /*!
     * \brief Sub structure to store the Minimap information.
     */
    struct Minimap{
        static int x, y;
        static int width, height;
        static int pen_size;
        static QColor pen_color, brush_color;
        static bool is_closed;
    };
    
    static void save_minimap(int x, int y, int w, int h, bool hide_on_start);
    
    /*!
     * \brief maximum number of file shown in the recent files menu.
     */ 
    static const int _MAX_NB_RECENT_FILES = 10;


    static void save_config_file();

    static void load_config_file();

    /*!
     * \fn get_recent_files()
     * \brief Get the list of the recent files
     */
    static const QStringList get_recent_files();

    /*!
     * \fn add_recent_file(const QString &filename)
     * \brief Add to the recent files opened the new file.
     * \param filename : the name of the file read
     */
    static void add_recent_file(const QString &filename);

    /*!
     * \fn clear_recent_files()
     * \brief Clear all the recent files opened. Only in the QSettings files, not in the graphical interface.
     */
    static void clear_recent_files();

    /*!
     * \fn load_plugin_directories(QStringList &list)
     * \brief Fill the QStringList passed in parameter with the user plugin directories
     */
    static void load_plugin_directories(QStringList &list);
    /*!
     * \fn save_plugin_directories(const QStringList &list)
     * \brief Save the QStringList passed in parameter as the user plugin directories
     */
    static void save_plugin_directories(const QStringList &list);

    /*!
     * \fn set_use_palette(bool use)
     * \brief Set if we use the current palette when we load a trace.
     * \param use : true if we use the current palette.
     */
    static void set_use_palette(bool use);

    /*!
     * \fn get_use_palette()
     * \brief Get if we use the current palette when we load a trace.
     */
    static bool get_use_palette();
    /*!
     * \fn get_palettes_name(QStringList &list)
     * \brief Fill the list with all the existing palettes name.
     * \param list : the list filled
     */
    static void get_palettes_name(QStringList &list);

    /*!
     * \fn get_palette(const std::string &name)
     * \brief Create and return the palette with the name passed in parameter.
     * \param name : the name of the required palette
     */
    static Palette *get_palette(const std::string &name = "default");
    /*!
     * \fn get_current_palette()
     * \brief return the name of the current palette (last used)
     */
    static std::string get_current_palette();
    /*!
     * \fn set_current_palette(const std::string &name)
     * \brief set the name of the current palette
     * \param name : the name of the palette
     */
    static void set_current_palette(const std::string &name);

    /*!
     * \fn create_palette(const std::string &name)
     * \brief Create an empty palette named name.
     * \param name : the name of the palette
     */
    static void create_palette(const std::string &name);
    /*!
     * \fn remove_palette(const std::string &name)
     * \brief Remove the palette named name.
     * \param name : the name of the palette
     */
    static void remove_palette(const std::string &name);

    /*!
     * \fn copy_palette(const std::string &src, const std::string &dest)
     * \brief Copy the src palette to dest
     * \param src : the name of the source palette
     * \param dest : the name of the destination palette
     */
    static void copy_palette(const std::string &src, const std::string &dest);
    /*!
     * \fn add_state_to_palette(const std::string &palette_name, const std::string &state_name, const Color &c)
     * \brief Add the state with its color of the palette.
     * \param palette_name : the palette name
     * \param state_name : the state name
     * \param c : the state color
     */
    static void add_state_to_palette(const std::string &palette_name, const std::string &state_name, const Color &c);
    /*!
     * \fn remove_state_to_palette(const std::string &palette_name, const std::string &state_name)
     * \brief Remove the state of the palette.
     * \param palette_name : the palette name
     * \param state_name : the state name
     */
    static void remove_state_to_palette(const std::string &palette_name, const std::string &state_name);
};

#endif
