/*
   Copyright (c) 2010 Sebastian Trueg <trueg@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "converterdialog.h"
#include "converterjob.h"
#include "nepomukstuff.h"
#include "virtuosocontroller.h"

#include <KPushButton>
#include <KUrlRequester>
#include <KLocale>
#include <KDebug>

#include <QtGui/QProgressBar>
#include <QtGui/QLabel>
#include <QtGui/QVBoxLayout>
#include <QtGui/QApplication>


ConverterDialog::ConverterDialog( bool autoMode,
                                  bool backup,
                                  bool keepDump,
                                  QWidget* parent )
    : KDialog( parent ),
      m_autoMode( autoMode ),
      m_backup( backup ),
      m_keepDump( keepDump ),
      m_progressLabel(0),
      m_progressBar(0)
{
    setButtons( Close );
    QWidget* w = mainWidget();
    m_progressLabel = new QLabel( w );
    QVBoxLayout* layout = new QVBoxLayout( w );
    layout->addWidget( m_progressLabel );
    bool needConversion = false;
    int version = Nepomuk::usedVirtuosoVersion();
    kDebug() << "Detected Virtuoso version:" << version;

    switch( version ) {
    case 0: {
        m_progressLabel->setText( i18n( "No Virtuoso Nepomuk database found. Nothing to convert." ) );
        break;
    }

    case 5: {
        QString installedVirtuoso = Soprano::VirtuosoController::locateVirtuosoBinary();
        int installedVirtuosoVersion = Nepomuk::virtuosoBinVersion( installedVirtuoso );
        kDebug() << "Installed Virtuoso" << installedVirtuoso << installedVirtuosoVersion;

        switch( installedVirtuosoVersion ) {
        case 5:
            m_progressLabel->setText( i18n( "Installed Virtuoso version is 5. No need to convert." ) );
            break;

        case 6:
            m_progressLabel->setText( i18n( "Ready to convert Nepomuk database to Virtuoso 6 format." ) );
            needConversion = true;
            break;

        default:
            m_progressLabel->setText( i18n( "Could not find a valid Virtuoso installation." ) );
            break;
        }
        break;
    }

    case 6: {
        m_progressLabel->setText( i18n( "Nepomuk Database already converted to version 6." ) );
        break;
    }

    default:
        m_progressLabel->setText( i18n( "Failed to determine the correct Virtuoso version number. Cannot convert." ) );
        break;
    }

    if ( needConversion ) {
        if ( m_autoMode ) {
            QMetaObject::invokeMethod( this, "prepareConversion", Qt::QueuedConnection );
        }
        else {
            setButtons( Close|Apply );
            setButtonText( Apply, i18n( "Start Conversion" ) );
            connect( this, SIGNAL( applyClicked() ), this, SLOT( prepareConversion() ) );
        }
    }
    else {
        if ( m_autoMode )
            QMetaObject::invokeMethod( qApp, "quit", Qt::QueuedConnection );
    }
}


ConverterDialog::~ConverterDialog()
{
}


void ConverterDialog::prepareConversion()
{
    QWidget* w = mainWidget();

    m_progressLabel->setText( i18n( "Converting Virtuoso database to version 6..." ) );
    m_progressBar = new QProgressBar( w );
    m_progressBar->setRange( 0, 0 );

    w->layout()->addWidget( m_progressBar );

    enableButton( Close, false );
    enableButton( Apply, false );

    ConverterJob* job = new ConverterJob( this );
    job->setBackup( m_backup );
    job->setKeepDump( m_keepDump );
    connect( job, SIGNAL( infoMessage( KJob*, QString, QString ) ),
             this, SLOT( slotInfoMessage( KJob*, QString, QString ) ) );
    connect( job, SIGNAL( result( KJob* ) ),
             this, SLOT( slotConversionDone( KJob* ) ) );
    job->start();
}


void ConverterDialog::slotInfoMessage( KJob*, const QString& message, const QString& )
{
    kDebug() << message;
    m_progressLabel->setText( message );
}


void ConverterDialog::slotConversionDone( KJob* job )
{
    enableButton( Close, true );
    m_progressBar->hide();

    if ( !job->errorText().isEmpty() ) {
        m_progressLabel->setText( i18n( "Nepomuk Virtuoso database conversion failed: %1", job->errorString() ) );
    }
    else {
        m_progressLabel->setText( i18n( "Nepomuk Virtuoso database successfully converted to version 6." ) );
        if ( m_autoMode )
            qApp->quit();
    }
}

#include "converterdialog.moc"
