/*
**  This file is part of Vidalia, and is subject to the license terms in the
**  LICENSE file, found in the top level directory of this distribution. If you
**  did not receive the LICENSE file with this file, you may obtain it from the
**  Vidalia source package distributed by the Vidalia Project at
**  http://www.vidalia-project.net/. No part of Vidalia, including this file,
**  may be copied, modified, propagated, or distributed except according to the
**  terms described in the LICENSE file.
*/

/*
** \file CrashReportDialog.h
** \brief Dialog that asks the user whether they would like to
** submit the crash report, along with optional additional details
** about what they were doing at the time of the crash.
*/

#include "ui_CrashReportDialog.h"

#include <QHash>
#include <QByteArray>

class QString;


class CrashReportDialog : public QDialog
{
  Q_OBJECT

public:
  /** Default constructor.
   */
  CrashReportDialog(QWidget *parent = 0);

  /** Sets the crash <b>annotations</b> key-value pairs associated with
   * the generated minidump.
   */
  void setCrashAnnotations(const QHash<QString,QString> &annotations);

  /** Sets the <b>minidump</b> contents generated by the crashed
   * applications exception handler.
   */
  void setMinidump(const QString &id, const QByteArray &minidump);

  /** Uploads the generated minidump, user comments, and any additional
   * crash annotations generated by the exception handler to the crash
   * reporting server.
   * \sa setMinidump()
   * \sa setCrashAnnotations()
   */
  void submitCrashReport();

public slots:
  /** Called when the user clicks the "Restart Vidalia" button on the
   * dialog. If the "Submit my crash report..." checkbox is checked, it
   * will first attempt to submit the crash report. After that is complete,
   * it will try to restart the Vidalia process with any arguments specified
   * in the crash annotations file.
   * \sa setCrashAnnotations()
   */
  virtual void accept();

  /** Called when the user clicks the "Don't Restart" button on the
   * dialog. If the "Submit my crash report.." checkbox is checked, it
   * will attempt to submit the crash report and then exit without
   * restarting Vidalia.
   */
  virtual void reject();

private:
  /** Each minidump is given a randomly-generated GUID when it is created,
   * which is used to form the minidump filename. This ID is also used by
   * the crash reporting server when accepting and processing uploaded
   * minidumps.
   */
  QString _minidumpId;

  /** Contents of the generated minidump.
   */
  QByteArray _minidump;

  /** Set of parsed key-value pairs generated by the crashed application's
   * exception handler and written alongside the minidump.
   */
  QHash<QString,QString> _annotations;

  /** Qt Designer created object.
   */
  Ui::CrashReportDialog ui;
};

