#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <gtk/gtk.h>

#include "v3dmp.h"
#include "v3dmodel.h"

#include "vmaundo.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


gpointer VMAUndoNew(gint type, const gchar *name);
void VMAUndoDelete(gpointer u);
void VMAUndoSetRepeats(gpointer u, gint repeats);

gpointer VMAUndoListGetPtr(gpointer *list, gint total, gint i);
gpointer VMAUndoListInsert(
	gpointer **list, gint *total, gint i,
	gint type, const gchar *name
);
void VMAUndoListDelete(gpointer **list, gint *total, gint i);
void VMAUndoListDeleteAll(gpointer **list, gint *total);



/*
 *	Dynamically allocates a new undo structure of the specified
 *	type and name.
 */
gpointer VMAUndoNew(gint type, const gchar *name)
{
	size_t size = 0;
	gpointer u = NULL;


	switch(type)
	{
	  case VMA_UNDO_TYPE_SET_NORMAL:
	    size = sizeof(vma_undo_set_normal_struct);
	    break;

	  case VMA_UNDO_TYPE_SET_TEXCOORD:
	    size = sizeof(vma_undo_set_texcoord_struct);
	    break;

	  case VMA_UNDO_TYPE_SET_VERTEX:
	    size = sizeof(vma_undo_set_vertex_struct);
	    break;

	  case VMA_UNDO_TYPE_SET_TEXORIENT:
	    size = sizeof(vma_undo_set_texorient_struct);
	    break;

	  case VMA_UNDO_TYPE_SET_HEIGHTFIELD:
	    size = sizeof(vma_undo_set_heightfield_struct);
	    break;

	  case VMA_UNDO_TYPE_FLIP_WINDING:
	    size = sizeof(vma_undo_flip_winding_struct);
	    break;

	  case VMA_UNDO_TYPE_CREATE_PRIMITIVE:
	    size = sizeof(vma_undo_create_primitive_struct);
	    break;

	  case VMA_UNDO_TYPE_DELETE_PRIMITIVE:
	    size = sizeof(vma_undo_delete_primitive_struct);
	    break;

	  case VMA_UNDO_TYPE_TRANSLATE_PRIMITIVE:
	    size = sizeof(vma_undo_translate_primitive_struct);
	    break;

	  case VMA_UNDO_TYPE_ADD_VERTEX:
	    size = sizeof(vma_undo_add_vertex_struct);
	    break;

	  case VMA_UNDO_TYPE_REMOVE_VERTEX:
	    size = sizeof(vma_undo_remove_vertex_struct);
	    break;


	  default:
	    fprintf(
		stderr,
		"MAUndoNew(): Unsupported undo structure type %i.\n",
		type
	    );
	    break;
	}

	if(size > 0)
	{
	    vma_undo_common_struct *uc;

	    /* Allocate the undo structure. */
	    u = calloc(1, size);
	    if(u == NULL)
		return(u);

	    /* Get undo structure as common and set type and name. */
	    uc = u;
	    uc->type = type;
	    uc->name = ((name == NULL) ? NULL : g_strdup(name));
	    uc->repeats = 0;
	}

	return(u);
}

/*
 *	Deallocates the undo structure u and all of its allocated
 *	resources.
 */
void VMAUndoDelete(gpointer u)
{
	vma_undo_common_struct *uc;

	vma_undo_create_primitive_struct *u_create_primitive;
	vma_undo_add_vertex_struct *u_add_vertex;


	if(u == NULL)
	    return;

	/* Deallocate common members. */
	uc = u;
	g_free(uc->name);
	uc->name = NULL;

	/* Deallocate structure type specific members. */
	switch(uc->type)
	{
	  case VMA_UNDO_TYPE_CREATE_PRIMITIVE:
	    u_create_primitive = (vma_undo_create_primitive_struct *)u;
	    V3DMPDestroy(u_create_primitive->p);
	    u_create_primitive->p = NULL;
	    break;

	  case VMA_UNDO_TYPE_ADD_VERTEX:
	    u_add_vertex = (vma_undo_add_vertex_struct *)u;
	    free(u_add_vertex->v);
	    u_add_vertex->v = NULL;
	    free(u_add_vertex->n);
	    u_add_vertex->n = NULL;
	    free(u_add_vertex->tc);
	    u_add_vertex->tc = NULL;
	    break;

/* Add other types that have dynamically allocated substructures
 * that need to be free()'ed here.
 */
	}

	/* Deallocate structure itself. */
	g_free(u);

	return;
}

/*
 *	Sets the repeats value on the given undo structure to the
 *	value specified by repeats.
 */
void VMAUndoSetRepeats(gpointer u, gint repeats)
{
	vma_undo_common_struct *uc = (vma_undo_common_struct *)u;
	if(uc == NULL)
	    return;

	uc->repeats = repeats;

	return;
}




/*
 *      Checks if item i exists in the list and returns its pointer
 *      if it exists or NULL if it does not.
 */
gpointer VMAUndoListGetPtr(gpointer *list, gint total, gint i)
{
	if((list == NULL) || (i < 0) || (i >= total))
	    return(NULL);
	else
	    return(list[i]);
}

/*
 *      Inserts a new undo structure item at list position i and
 *      returns the pointer to the undo structure or NULL on failure.
 *
 *      If i is negative then the item will be appended to the list.
 *
 *      If i is greater or equal to (*total) + 1 then i will
 *      be set to the value of (*total).
 *
 *      Both list and total need to be non NULL.
 */
gpointer VMAUndoListInsert(
	gpointer **list, gint *total, gint i,
	gint type, const gchar *name
)
{
	if((list == NULL) || (total == NULL))
	    return(NULL);

	if((*total) < 0)
	    (*total) = 0;

	/* Allocate more pointers for the list. */
	(*total) = (*total) + 1;
	(*list) = (gpointer *)g_realloc(
	    *list,
	    (*total) * sizeof(gpointer)
	);  
	if((*list) == NULL) 
	{
	    (*total) = 0;
	    return(NULL);
	}

	/* Append or insert? */
	if(i < 0)
	{
	    /* Append. */

	    i = (*total) - 1;
 
	    (*list)[i] = VMAUndoNew(type, name);
	}
	else
	{
	    /* Insert. */
	    gint n;

	    if(i >= (*total))
		i = (*total) - 1;

	    /* Shift pointers. */
	    for(n = (*total) - 1; n > i; n--)
		(*list)[n] = (*list)[n - 1];

	    (*list)[i] = VMAUndoNew(type, name);
	}

	return((*list)[i]);
}

/*
 *      Deletes the item i from the list.  Item pointer i on the list
 *      will then be set to NULL.
 */
void VMAUndoListDelete(gpointer **list, gint *total, gint i)
{
	gpointer u;

	if((list == NULL) || (total == NULL))
	    return;

	/* Item exists and is allocated? */
	u = VMAUndoListGetPtr(*list, *total, i);
	if(u != NULL)
	{
	    VMAUndoDelete(u);
	    (*list)[i] = NULL;  /* Reset item pointer to NULL. */
	}

	return;
}

/*
 *      Deletes the entire list and resets the pointers list
 *      and total to NULL.
 */
void VMAUndoListDeleteAll(gpointer **list, gint *total)
{
	gint i;

	if((list == NULL) || (total == NULL))
	    return;

	for(i = 0; i < (*total); i++)
	    VMAUndoListDelete(list, total, i);

	g_free(*list);
	*list = NULL;

	*total = 0;

	return;
}
