/*  $Id: conjugation.h,v 1.4 2009/10/31 02:02:05 sarrazip Exp $
    conjugation.h - Generic conjugation interface

    verbiste - French conjugation system
    Copyright (C) 2003-2006 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_conjugation
#define _H_conjugation

#include <verbiste/FrenchVerbDictionary.h>

#include <vector>
#include <string>

typedef std::vector<std::string> VS;
typedef std::vector<VS> VVS;
typedef std::vector<VVS> VVVS;


/** Create the French verb dictionary, which can conjugate and
    deconjugate French verbs.
    Uses the LIBDATADIR macro and environment variable.
    @returns			an object of class FrenchVerbDictionary,
				allocated by new
    @throws	logic_error	throw by the FrenchVerbDictionary constructor
*/
verbiste::FrenchVerbDictionary *initVerbDict() throw(std::logic_error);


/** Obtains the conjugation of the given infinitive.
    @param	fvd		verb dictionary from which to obtain
				the conjugation
    @param	infinitive	UTF-8 string containing the infinitive form
				of the verb to conjugate (e.g., "manger")
    @param	dest		structure into which the conjugated is written
    @param	includePronouns put pronouns before conjugated verbs in the
				modes where pronouns are used
*/
void getConjugation(const verbiste::FrenchVerbDictionary &fvd,
			const std::string &infinitive,
			VVVS &dest,
			bool includePronouns = false);


/** Get the tense name for a certain cell of the conjugation table.
    The conjugation table is a 4x4 grid and 11 of the 16 cells are
    used by the tenses to be displayed.
    @param	row	row number (0..3)
    @param	col	column number (0..3)
    @returns		an internationalized (and possibly abbreviated) name
			(in UTF-8) for the tense that goes in the cell
			designated by row and col;
			the returned string is empty if no tense goes in
			the designated cell
*/
std::string getTenseNameForTableCell(int row, int col);


/** Composes the text of the conjugation in a certain tense.
    The words of this conjugation that match a certain user text
    are marked specially.

    @param	tense			structure that represents a certain
					tense of a certain verb
    @param	lowerCaseLatin1UserText	conjugated verb as entered by the
					user, but converted to lower-case,
					and assumed to be in Latin-1
    @param	openMark		ASCII string to be used before verbs
					that match lowerCaseLatin1UserText
					(e.g., "<b>")
    @param	closeMark		ASCII string to be used after verbs
					that match lowerCaseLatin1UserText
					(e.g., "</b>")
    @returns				a Latin-1 string containing several
					lines of text, each line being
					separated by a newline ('\n')
					character; the string does not end
					the last line with a newline however;
					e.g., "aie\nayons\nayez"
*/
std::string createTableCellText(const VVS &tense,
				const std::string &lowerCaseULatin1serText,
				const std::string &openMark,
				const std::string &closeMark);


#endif  /* _H_conjugation */
