

#include <openbabel/obconversion.h>
#include <openbabel/obmolecformat.h>
#include <openbabel/mol.h>
#include <openbabel/math/matrix3x3.h>
#include <openbabel/math/vector3.h>

#include <visu_tools.h>
#include <visu_basic.h>
#include <renderingMethods/renderingAtomic.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolFileFormat.h>
#include <coreTools/toolMatrix.h>
#include <visu_elements.h>
#include <visu_data.h>
#include <visu_pairs.h>

using namespace OpenBabel;

extern "C"
{
  gboolean obloaderInit();
  const char* obloaderGet_description();
  const char* obloaderGet_authors();
  const char* obloaderGet_icon();
}

#define OPENBABEL_DESCRIPTION _("<span size=\"smaller\">" \
				"This plug-in wraps the <b>OpenBabel</b>\n" \
				"library (visit the home page at URL\n" \
				"<span color=\"blue\"><u>http://www.openbabel.org</u></span>)</span>.")
#define OPENBABEL_AUTHORS     _("Caliste Damien:\n   wrapper.")

/* Local variables. */
static gchar *iconPath;

/* Local methods. */
static gboolean loadOpenBabelFile(VisuData *data, const gchar* filename,
				  FileFormat *format, int nSet, GError **error);

/* Required methods for a loadable module. */
gboolean obloaderInit()
{
  RenderingFormatLoad* meth;
  char *type[] = {(char*)0};

  DBG_fprintf(stderr, "Openbabel : loading plug-in 'openbabel'...\n");

  meth = (RenderingFormatLoad*)g_malloc(sizeof(RenderingFormatLoad));
  meth->name = g_strdup("OpenBabel wrapper");
  meth->fmt = fileFormatNew(_("OpenBabel known formats"), type);
  if (!meth->fmt)
    {
      g_error("Can't initialize the OpenBabel loading method, aborting...\n");
    }
  meth->priority = 75;
  meth->load = loadOpenBabelFile;
  renderingAtomicAdd_loadMethod(meth);

  iconPath = g_build_filename(V_SIM_PIXMAPS_DIR, "openbabel.png", NULL);

  return TRUE;
}

const char* obloaderGet_description()
{
  return OPENBABEL_DESCRIPTION;
}

const char* obloaderGet_authors()
{
  return OPENBABEL_AUTHORS;
}

const char* obloaderGet_icon()
{
  return (char*)iconPath;
}

static char* getSymbol(OpenBabel::OBAtom *atom)
{
  return (char*)etab.GetSymbol(atom->GetAtomicNum());
}

static void addNode(GList **lst, OpenBabel::OBAtom *atom)
{
  char *ele;
  GList *tmplst;
  GList *lstatom;
  gboolean found;

  ele = getSymbol(atom);

  /* Look for type ele in lst. */
  found = FALSE;
  tmplst = *lst;
  while (tmplst && !found)
    {
      lstatom = (GList*)tmplst->data;
      if ( !strcmp(getSymbol((OpenBabel::OBAtom*)lstatom->data), ele) )
	{
	  // 	  fprintf(stderr, "one node for element '%s'\n", ele);
	  lstatom = g_list_prepend(lstatom, (gpointer)atom);
	  tmplst->data = lstatom;
	  found = TRUE;
	}
      tmplst = g_list_next(tmplst);
    }
  if (!found)
    {
      //      fprintf(stderr, "one element '%s'\n", ele);
      lstatom = (GList*)0;
      lstatom = g_list_prepend(lstatom, (gpointer)atom);
      *lst = g_list_prepend(*lst, lstatom);
    }
}

static gboolean loadOpenBabelFile(VisuData *data, const gchar* filename,
				  FileFormat *format, int nSet _U_, GError **error)
{
  OpenBabel::OBMol *mol;
  OpenBabel::OBFormat *pFormat, *xyzFormat;
  bool res;
  gboolean res2;
  std::ifstream fin(filename);
  std::istream* pIn = &fin;
  OpenBabel::OBConversion conv(pIn, NULL);
  GList *allNodes, *lstatom, *tmplst;
  int ok, i, j;
  VisuElement **types, *visuele, *ele1, *ele2;
  unsigned int *nattyp;
  unsigned int ntype;
  char *ele;
  OpenBabel::OBUnitCell *uc;
  double rprimdFull[9], rprimd[3][3];
  std::vector<double> eleRGB;
  double boxGeometry[6];
  float xyz[3], length, lengthMin, lengthMax;
  GList *pairs;
  gchar *infoUTF8;
  double vect[3], radius;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  /* Create a new OpenBabel object. */
  mol = new OpenBabel::OBMol;

  /* Try to guess the file format. */
  DBG_fprintf(stderr, "Open Babel : try to guess the file format of '%s'.\n", filename);
  pFormat = conv.FormatFromExt(filename);
  if (!pFormat)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("'%s' doesn't match any file format."), filename);
      fin.close();
      return FALSE;
    }
  if ( pFormat->Flags() & NOTREADABLE )
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("format of '%s' is not a readable one."), filename);
      fin.close();
      return FALSE;
    }
  /* Exclude the xyz file format since V_Sim handles it natively. */
  xyzFormat = conv.FindFormat("xyz");
  if (xyzFormat == pFormat)
    {
      DBG_fprintf(stderr, "OpenBabel : skip XYZ format.\n");
      fin.close();
      return FALSE;
    }

  DBG_fprintf(stderr, " | set format %p.\n", (gpointer)pFormat);
  DBG_fprintf(stderr, " | format description\n%s\n", pFormat->Description());
  conv.SetInFormat(pFormat);

  /* Read the file. */
  DBG_fprintf(stderr, "Open Babel : read the file.\n");
  res = conv.Read(mol);
  fin.close();
  if (!res)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("The given file doesn't match the format '%s'."),
			   format->description);
      return FALSE;
    }

  /* Store if the file is periodic or not. */
  uc = (OBUnitCell*)mol->GetData(OBGenericDataType::UnitCell);
  if (uc)
    {
      DBG_fprintf(stderr, "OpenBabel : file has periodic conditions.\n");
      uc->GetCellMatrix().GetArray(rprimdFull);
      for (i = 0; i < 3; i++)
	for (j = 0; j < 3; j++)
	  rprimd[j][i] = rprimdFull[i * 3 + j];
      res2 = matrix_reducePrimitiveVectors(boxGeometry, rprimd);
      if (!res2)
	{
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("The cell is not well formed (basis is not 3D)."));
	  return FALSE;
	}
      visuDataSet_boxGeometry(data, boxGeometry, BOX_PERIODIC);
      uc->GetOffset().Get(vect);
    }
  else
    {
      DBG_fprintf(stderr, "OpenBabel : file has no periodic conditions.\n");
      for (i = 0; i < 3; i++)
	vect[i] = 0.;
    }

  DBG_fprintf(stderr, "OpenBabel : first pass to find Elements.\n");
  allNodes = (GList*)0;
  FOR_ATOMS_OF_MOL(a,mol)
    {
      addNode(&allNodes, &(*a));
    }
  DBG_fprintf(stderr, "OpenBabel : count nodes.\n");
  /* Allocate the space for the nodes. */
  ntype = g_list_length(allNodes);
  types = (VisuElement**)g_malloc(sizeof(VisuElement*) * ntype);
  nattyp = (unsigned int*)g_malloc(sizeof(unsigned int) * ntype);
  /* Get the visuelements. */
  i = 0;
  tmplst = allNodes;
  while (tmplst)
    {
      lstatom = (GList*)tmplst->data;
      ele = getSymbol((OpenBabel::OBAtom*)lstatom->data);
      types[i] = visuElementGet_fromName(ele);
      if (!types[i])
	{
	  types[i] = visuElementNew_withName(ele);
	  ok = visuElementAdd(types[i]);
	  // We use the standard OB colours.
	  eleRGB = etab.GetRGB(((OpenBabel::OBAtom*)lstatom->data)->GetAtomicNum());
	  types[i]->rgb[0] = eleRGB[0];
	  types[i]->rgb[1] = eleRGB[1];
	  types[i]->rgb[2] = eleRGB[2];
	  colorAdd_floatRGBA(types[i]->rgb, NULL);
	  // We use standard radius for element.
	  radius =
	    etab.GetCovalentRad(((OpenBabel::OBAtom*)lstatom->data)->GetAtomicNum());
	  renderingAtomicSet_radius(types[i], (float)radius);
	}      
      nattyp[i] = g_list_length(lstatom);
      g_list_free(lstatom);
      DBG_fprintf(stderr, " | '%s' : %d.\n", types[i]->name, nattyp[i]);
      tmplst = g_list_next(tmplst);
      i += 1;
    }
  g_list_free(allNodes);

  /* Allocate space in the given visudata. */
  ok = visuDataSet_population(data, ntype, nattyp, types);
  if (!ok)
    {
      g_error("Can't store the nodes in the VisuData object.");
    }
  g_free(nattyp);
  g_free(types);

  /* Stores coordinates. */
  i = 0;
  FOR_ATOMS_OF_MOL(a,mol)
    {
      visuele = visuElementGet_fromName(getSymbol(&(*a)));
      xyz[0] = (float)a->GetX() + vect[0];
      xyz[1] = (float)a->GetY() + vect[1];
      xyz[2] = (float)a->GetZ() + vect[2];
      visuDataAdd_nodeFromElement(data, visuele, xyz, FALSE);
      i += 1;
    }

  if (!uc)
    visuDataSet_tightBox(data);

  visuDataApply_boxGeometry(data, 0.f);

  /* Set the bonds, if any. */
  FOR_BONDS_OF_MOL(b, mol)
    {
      ele1 = visuElementGet_fromName(getSymbol(b->GetBeginAtom()));
      ele2 = visuElementGet_fromName(getSymbol(b->GetEndAtom()));
      pairs = visuPairGet_links(ele1, ele2);
      lengthMin = visuPairGet_distance((VisuPairData*)pairs->data, PAIRS_MIN);
      lengthMax = visuPairGet_distance((VisuPairData*)pairs->data, PAIRS_MAX);
      length = (float)b->GetLength();
      if (lengthMax < length)
	visuPairSet_distance((VisuPairData*)pairs->data, length, PAIRS_MAX);
      if (lengthMin > length || lengthMin == 0.)
	visuPairSet_distance((VisuPairData*)pairs->data, length, PAIRS_MIN);
    }

  /* Set the commentary. */
  if (mol->GetTitle())
    {
      infoUTF8 = g_locale_to_utf8(mol->GetTitle(), -1, NULL, NULL, NULL);
      if (infoUTF8)
	{
	  visuDataSet_fileCommentary(data, infoUTF8, 0);
	  g_free(infoUTF8);
	}
      else
	{
	  g_warning("Can't convert '%s' to UTF8.\n", mol->GetTitle());
	}
    }
  
  return TRUE;
}
