/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "axes.h"
#include "../renderingMethods/renderingAtomic.h"
#include "../renderingMethods/renderingSpin.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <opengl.h>
#include <openGLFunctions/text.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_configFile.h>
#include <renderingBackend/visu_windowInterface.h>
#include <openGLFunctions/objectList.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolConfigFile.h>

#include <math.h>

/**
 * SECTION:axes
 * @short_description: Defines methods to draw axes.
 *
 * <para>The axes are the X, Y and Z lines drawn on the bottom right of the
 * screen defining a given orthogonal basis set in which the box is
 * projected.</para>
 * <para>The axis may be different, depending on the rendering method
 * currently used. For instance, when the spin is used, a projection
 * of the colour scheme is added to the simple lines of the basis
 * set. Besides that, axes are defined by their width (see
 * axesSet_lineWidth()) and their colour (see
 * axesSet_RGBValues()).</para>
 */

/* Parameters & resources*/
/* This is a boolean to control is the axes is render or not. */
#define FLAG_RESOURCE_AXES_USED   "axes_are_on"
#define DESC_RESOURCE_AXES_USED   "Control if the axes are drawn ; boolean (0 or 1)"
#define RESOURCE_AXES_USED_DEFAULT 0
static gboolean readAxesAreOn(gchar **lines, int nbLines,
			      int position, VisuData *dataObj, GError **error);
/* A resource to control the color used to render the lines of the axes. */
#define FLAG_RESOURCE_AXES_COLOR   "axes_color"
#define DESC_RESOURCE_AXES_COLOR   "Define the color of the axes ; three floating point values (0. <= v <= 1.)"
static float axesRGB[3];
static float axesRGBDefault[3] = {1.0, 0.5, 0.1};
static gboolean readAxesColor(gchar **lines, int nbLines,
			      int position, VisuData *dataObj, GError **error);
/* A resource to control the width to render the lines of the axes. */
#define FLAG_RESOURCE_AXES_LINE   "axes_line_width"
#define DESC_RESOURCE_AXES_LINE   "Define the width of the lines of the axes ; one floating point values (1. <= v <= 10.)"
#define RESOURCE_AXES_LINE_DEFAULT 1.
static float axesLineWidth;
static gboolean readAxesLineWidth(gchar **lines, int nbLines,
				  int position, VisuData *dataObj, GError **error);
/* A resource to control the stipple to render the lines of the box. */
#define FLAG_RESOURCE_AXES_STIPPLE   "axes_line_stipple"
#define DESC_RESOURCE_AXES_STIPPLE   "Dot scheme detail for the lines of the axes ; 0 < integer < 2^16"
#define RESOURCE_AXES_STIPPLE_DEFAULT 65535
static guint16 axesLineStipple;
static gboolean readAxesLineStipple(gchar **lines, int nbLines,
				    int position, VisuData *dataObj, GError **error);

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static void exportResourcesAxes(GString *data, VisuData *dataObj);


static OpenGLExtension* extensionAxes;


static void rebuildAxes(VisuData *dataObj);

/* This flag can be set internally to specify that axes should be recreated. */
static gboolean axesHaveBeenBuilt;

/* Local callbacks */
static void onAxesParametersChange(VisuData *dataObj, OpenGLView *view, gpointer data);
static void onAxesConfigChange(GObject *visu, VisuData *dataObj, gpointer data);
static void rebuildAxesOnResourcesLoaded(GObject *visu, VisuData *dataObj,
					 gpointer data);
static void onDataNewSignal(GObject *visu, VisuData *dataObj, gpointer data);

OpenGLExtension* initExtensionAxes()
{
  char *name = EXT_AXES_ID;
  char *description = _("Draw the axes at the bottom right of the rendering area.");
  int listAxes;
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr,"Initialising the axes OpenGL extension...\n");
  listAxes = openGLObjectList_new(1);
  extensionAxes = OpenGLExtension_new(name, _(name), description,
				      listAxes, rebuildAxes);

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_AXES_USED,
					  DESC_RESOURCE_AXES_USED,
					  1, readAxesAreOn);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_AXES_COLOR,
					  DESC_RESOURCE_AXES_COLOR,
					  1, readAxesColor);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_AXES_LINE,
					  DESC_RESOURCE_AXES_LINE,
					  1, readAxesLineWidth);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_AXES_STIPPLE,
					  DESC_RESOURCE_AXES_STIPPLE,
					  1, readAxesLineStipple);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesAxes);

  /* Initialisation des valeurs par dfaut. */
  axesSet_RGBValues(axesRGBDefault, MASK_RGB_ALL);
  extensionAxes->used = RESOURCE_AXES_USED_DEFAULT;
  OpenGLExtensionSet_priority(extensionAxes, OPENGL_EXTENSION_PRIORITY_LAST);
  OpenGLExtensionSet_saveOpenGLState(extensionAxes, TRUE);
  axesLineWidth   = RESOURCE_AXES_LINE_DEFAULT;
  axesLineStipple = RESOURCE_AXES_STIPPLE_DEFAULT;

  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onDataNewSignal), (gpointer)0);

  g_signal_connect(VISU_INSTANCE, "SpinAxes",
 		   G_CALLBACK(onAxesConfigChange), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
 		   G_CALLBACK(rebuildAxesOnResourcesLoaded), (gpointer)0);


  axesHaveBeenBuilt = FALSE;
  
  return extensionAxes;
}

/* Method used to change the value of the parameter axes_color. */
gboolean axesSet_RGBValues(float rgb[3], int mask)
{
  int diff = 0;
  
  if (mask & MASK_RGB_R && axesRGB[0] != rgb[0])
    {
      axesRGB[0] = rgb[0];
      diff = 1;
    }
  if (mask & MASK_RGB_G && axesRGB[1] != rgb[1])
    {
      axesRGB[1] = rgb[1];
      diff = 1;
    }
  if (mask & MASK_RGB_B && axesRGB[2] != rgb[2])
    {
      axesRGB[2] = rgb[2];
      diff = 1;
    }
  if (!diff)
    return FALSE;

  axesHaveBeenBuilt = FALSE;
  return (gboolean)extensionAxes->used;
}

/* Method used to change the value of the parameter axes_line_width. */
gboolean axesSet_lineWidth(float width)
{
  if (width < 1. || width > 10. || width == axesLineWidth)
    return FALSE;

  axesLineWidth = width;
  axesHaveBeenBuilt = FALSE;
  return (gboolean)extensionAxes->used;
}
gboolean axesSet_lineStipple(guint16 stipple)
{
  if (stipple == axesLineStipple)
    return FALSE;

  axesLineStipple = stipple;
  axesHaveBeenBuilt = FALSE;
  return (gboolean)extensionAxes->used;
}

/* Method used to change the value of the parameter axes_is_on. */
gboolean axesSet_areOn(gboolean value)
{
  if (value == extensionAxes->used)
    return FALSE;

  extensionAxes->used = value;
  return (value && !axesHaveBeenBuilt);
}

/* Get methods. */
float* axesGet_RGBvalues()
{
  return axesRGB;
}

int axesGet_areOn()
{
  return extensionAxes->used;
}

float axesGet_lineWidth()
{
  return axesLineWidth;
}
guint16 axesGet_lineStipple()
{
  return axesLineStipple;
}


/****************/
/* Private part */
/****************/

static void rebuildAxes(VisuData *dataObj)
{
  openGLText_rebuildFontList();
  /* Set flag to false to force recreation of axes. */
  axesHaveBeenBuilt = FALSE;
  axesDraw(dataObj);
}
static void rebuildAxesOnResourcesLoaded(GObject *visu _U_, VisuData *dataObj,
					 gpointer data _U_)
{
  if (!dataObj)
    return;
  axesDraw(dataObj);
}
static void onDataNewSignal(GObject *visu _U_, VisuData *dataObj,
			    gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "OpenGLNearFar",
		   G_CALLBACK(onAxesParametersChange), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLWidthHeight",
		   G_CALLBACK(onAxesParametersChange), (gpointer)0);
}
static void onAxesConfigChange(GObject *visu _U_, VisuData *dataObj, gpointer data)
{
  g_return_if_fail(dataObj);

  onAxesParametersChange(dataObj, visuDataGet_openGLView(dataObj), data);
}

static void onAxesParametersChange(VisuData *dataObj, OpenGLView *view _U_,
				   gpointer data _U_)
{
  axesHaveBeenBuilt = FALSE;
  axesDraw(dataObj);
}

/**
 * drawAxes
 * @length: the length of the axes
 * @width: the width of the line used to draw the axes
 * @long_axes:
 *
 * Draw three axes following the current x,y,z directions. Their @length and
 * @width can be specified.
 */
void drawAxes(float length, GLsizei w, GLsizei h, float width,
	      char *legend, gboolean long_axes) 
{
  glLineWidth(width);
  glColor3fv(axesRGB);
  glBegin(GL_LINES);
  glVertex3f(long_axes ? -length : 0, 0.0f, 0.0f); glVertex3f(length, 0.0f, 0.0f);
  glVertex3f(0.0f, long_axes ? -length : 0, 0.0f); glVertex3f(0.0f, length, 0.0f);
  glVertex3f(0.0f, 0.0f, long_axes ? -length : 0); glVertex3f(0.0f, 0.0f, length);
  glEnd();

  glRasterPos3f(length, 0.0f, 0.0f); openGLText_drawChars("x", TEXT_NORMAL); 
  glRasterPos3f(0.0f, length, 0.0f); openGLText_drawChars("y", TEXT_NORMAL); 
  glRasterPos3f(0.0f, 0.0f, length); openGLText_drawChars("z", TEXT_NORMAL);

  if(legend != NULL)
    {
      glMatrixMode (GL_PROJECTION);
      glPushMatrix();
      glLoadIdentity ();  
      gluOrtho2D (0, MIN(w,h), 0, MIN(h,w));
      glMatrixMode (GL_MODELVIEW);
      glPushMatrix();
      glLoadIdentity ();  
      glRasterPos3f(20., 5., 0.9); openGLText_drawChars(legend, TEXT_NORMAL);
      glPopMatrix();
      glMatrixMode(GL_PROJECTION);
      glPopMatrix();
      glMatrixMode(GL_MODELVIEW);
    }
}

/**
 * draw_coloured_cone:
 * @r: the radius of the cone
 * @h: the semi-height of the cone
 * @n: the precision used to draw the sphere
 *
 * Draws a coloured double cone at the given position.
 */ 
void draw_coloured_cone(double r, double h, int n, float phi_prime_zero)
{
  float hsv[3], rgb[3];
  int i,j;
  double theta1,theta2,theta3;
  float e_x, e_y, e_z, p_x, p_y, p_z;

  g_return_if_fail(r >= 0 && n >= 0);
  
  if (n < 4 || r <= 0) 
    {
      glBegin(GL_POINTS);
      glVertex3f(0,0,0);
      glEnd();
      return;
    }

  glFrontFace(GL_CW);

  glPushMatrix();
  glRotatef(phi_prime_zero, 0, 0, 1);

  glRotatef(-90, 1, 0, 0);

  hsv[1] = 0;
  hsv[2] = 1;

  for (j=0;j<n/2;j++) 
    {
      theta1 = j * 2 * G_PI / n - G_PI_2;
      theta2 = (j + 1) * 2 * G_PI / n - G_PI_2;
      
      glBegin(GL_QUAD_STRIP);
      for (i=0;i<=n;i++) 
	{
	  theta3 = i * 2 * G_PI / n;
	  
	  hsv[0] = /*1-*/(float)i/(float)n;

	  hsv[1] = 2*(float)(1+j)/(float)(n/2);
	  if(hsv[1] > 1) hsv[1] = 1;
      
	  hsv[2] = 2-2*(float)(1+j)/(float)(n/2); 
	  if(hsv[2] > 1) hsv[2] = 1; 

	  e_x = /*cos(theta2) **/hsv[1]*hsv[2]* cos(theta3);
	  e_y = sin(theta2);
	  e_z = /*cos(theta2) **/hsv[1]*hsv[2]* sin(theta3);
	  p_x = r * e_x;
	  p_y = /*r * e_y*/h*(hsv[1] - hsv[2]);
	  p_z = r * e_z;
	  
	  color_HSVtoRGB(rgb, hsv);
	  glColor3f(rgb[0], rgb[1], rgb[2]); 
	  glNormal3f(e_x,e_y,e_z);
	  glVertex3f(p_x,p_y,p_z);
	  
	  hsv[0] = /*1-*/(float)i/(float)n;

	  hsv[1] = 2*(float)j/(float)(n/2);
	  if(hsv[1] > 1) hsv[1] = 1;
      
	  hsv[2] = 2-2*(float)j/(float)(n/2); 
	  if(hsv[2] > 1) hsv[2] = 1; 

	  e_x = /*cos(theta1) **/hsv[1]*hsv[2]* cos(theta3);
       	  e_y = sin(theta1);
	  e_z = /*cos(theta1) **/hsv[1]*hsv[2]* sin(theta3);
	  p_x = r * e_x;
	  p_y = /*r * e_y*/h*(hsv[1] - hsv[2]);
	  p_z = r * e_z;
	
	  color_HSVtoRGB(rgb, hsv);
	  glColor3f(rgb[0], rgb[1], rgb[2]); 
	  glNormal3f(e_x,e_y,e_z);
	  glVertex3f(p_x,p_y,p_z);

	}
      glEnd();
    }
  glPopMatrix();
  glFrontFace(GL_CCW);
  
}

/* void drawConeCircle(/\*XYZ c,*\/ double r, int n)  */
/* { */
/*   /\*  XYZ e,p;*\/ */
/*   GLboolean antialiasing_was_on; */
/*   GLUquadric* trash = gluNewQuadric(); */
/*   int i; */
/*   double theta, height = 1; */

/* /\*   glEnable (GL_POLYGON_SMOOTH); *\/ */
/* /\*   glDisable (GL_DEPTH_TEST); *\/ */

/*   antialiasing_was_on = enableGlFeature(GL_LINE_SMOOTH); */
/*   glPushMatrix(); */
/*   glRotatef(90, 1, 0, 0); */
/*   glTranslatef(0, 0, -height/2); */
/*   glLineWidth(axesLineWidth); */
/*   glColor3f(0, 0, 0);  */
/*   gluQuadricOrientation(trash, GLU_INSIDE); */
/*   gluCylinder(trash, 0.95*r, 0.95*r, height, n, 1); */
/* /\*   glBegin(GL_LINE_LOOP); *\/ */
/* /\*   for (i=0;i<=n;i++)  *\/ */
/* /\*     { *\/ */
/* /\*       theta = i * TWOPI / n; *\/ */
/* /\*       e.x = cos(theta); *\/ */
/* /\*       e.z = sin(theta); *\/ */
/* /\*       p.x = 0.8*(c.x +  r * e.x); *\/ */
/* /\*       p.z = 0.8*(c.z +  r * e.z); *\/ */
/* /\*       glVertex3f(p.x, -0.6, p.z); *\/ */
/* /\*       glVertex3f(p.x, 0.6, p.z); *\/ */
/* /\*     } *\/ */
/* /\*   glEnd(); *\/ */
/*   glPopMatrix(); */
/*   restoreGlFeature(GL_LINE_SMOOTH, antialiasing_was_on); */
/* /\*   glEnable (GL_DEPTH_TEST); *\/ */
/* /\*   glDisable(GL_POLYGON_SMOOTH); *\/ */

/* } */

void axesDraw(VisuData *dataObj)
{
  float length;
  GLsizei w, h;
  GLint xx, yy;
  double mini, maxi, near, far;
  OpenGLView *view;

  /* Nothing to draw; */
  if(!axesGet_areOn() || axesHaveBeenBuilt) return;

  /* Nothing to draw if no data is associated to
     the current rendering window. */
  if (!dataObj)
    return;

  DBG_fprintf(stderr, "Extension axes: creating axes.\n");

  view = visuDataGet_openGLView(dataObj);
  length = 0.26178f * view->camera->length0;
  w = 0.2f * MIN(view->window->width, view->window->height);
  h = w;
  xx = view->window->width - w;
  yy = 0;
  mini = -0.5f * view->camera->length0 *
    (view->camera->d_red - 1.f) / view->camera->d_red;
  maxi = -mini;
  near = far = view->camera->d_red * view->camera->length0;
  near -= view->camera->length0;
  far  += view->camera->length0;

  axesHaveBeenBuilt = TRUE;

  openGLText_initFontList();
  
  glDeleteLists(extensionAxes->objectListId, 1);

  glNewList(extensionAxes->objectListId, GL_COMPILE);

  glEnable(GL_CULL_FACE);

  /* Dsactivation de la lumire et du brouillard et activation du culling. */
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);

  /* Resetting the depth buffer. */
  glDisable(GL_DEPTH_TEST);

  if (axesLineStipple != 65535)
    {
      glEnable(GL_LINE_STIPPLE);
      glLineStipple(1, axesLineStipple);
    }

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glFrustum(mini, maxi, mini, maxi, near, far);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glTranslated(+view->box->dxxs2+view->camera->centre[0],
	       +view->box->dyys2+view->camera->centre[1],
	       +view->box->dzzs2+view->camera->centre[2]);
  if(visuRenderingClassIs_currentByName(VISU_RENDERING_SPIN))
    {
      float phi_prime_zero, phi_a, theta_a;

      phi_prime_zero = rspin_getGlobalResource_float(spin_globalColorWheel);
      phi_a          = rspin_getGlobalResource_float(spin_globalConePhi);
      theta_a        = rspin_getGlobalResource_float(spin_globalConeTheta);

      glViewport(xx, yy, w, h);

      /* Draw the first color cone */
      drawAxes(1.5*length, w, h, axesLineWidth, _("front"), TRUE);

      glPushMatrix();
      glRotatef(phi_a, 0, 0, 1);
      glRotatef(theta_a, 0, 1, 0);
      draw_coloured_cone(length, 1.2*length, 16, phi_prime_zero);
      glPopMatrix();

      glViewport(xx, yy+h, w, h);

      /* Enabling front culling and drawing the second color cone */
      glPushMatrix();
      glRotatef(phi_a, 0, 0, 1);
      glRotatef(theta_a, 0, 1, 0);
      glCullFace(GL_FRONT);
      draw_coloured_cone(length, 1.2*length, 16, phi_prime_zero);    
      glCullFace(GL_BACK);
      glPopMatrix();

      drawAxes(1.5*length, w, h, axesLineWidth, _("back"), TRUE);
    }
  else if(visuRenderingClassIs_currentByName(VISU_RENDERING_ATOMIC))
    {
      glViewport(xx, yy, w, h);

      drawAxes(length, w, h, axesLineWidth, NULL, FALSE);
    }
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  /* Back to the main viewport. */
  glViewport(0, 0, view->window->width, view->window->height);

  glEndList();
}

/* Parameters & resources*/

/* This is a boolean to control is the axes is render or not. */
static gboolean readAxesAreOn(gchar **lines, int nbLines,
			      int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  axesSet_areOn(val);
  return TRUE;
}
/* A resource to control the color used to render the lines of the axes. */
static gboolean readAxesColor(gchar **lines, int nbLines,
			      int position, VisuData *dataObj _U_, GError **error)
{
  float rgb[3];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, rgb, 3, error))
    return FALSE;

  if (configFileClamp_float(&rgb[0], rgb[0], 0., 1.) ||
      configFileClamp_float(&rgb[1], rgb[1], 0., 1.) ||
      configFileClamp_float(&rgb[2], rgb[2], 0., 1.))
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 3 floating points"
			     "(0 <= v <= 1) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_AXES_COLOR);
      axesSet_RGBValues(axesRGBDefault, MASK_RGB_ALL);
      return FALSE;
    }
  axesSet_RGBValues(rgb, MASK_RGB_ALL);

  return TRUE;
}
/* A resource to control the width to render the lines of the axes. */
static gboolean readAxesLineWidth(gchar **lines, int nbLines,
				  int position, VisuData *dataObj _U_, GError **error)
{
  float width;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &width, 1, error))
    return FALSE;
  if (configFileClamp_float(&width, width, 1., 10.))
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 1 floating point"
			     "(1 <= v <= 10) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_AXES_LINE);
      axesSet_lineWidth(RESOURCE_AXES_LINE_DEFAULT);
      return FALSE;
    }
  axesSet_lineWidth(width);

  return TRUE;
}
/* A resource to control the width to render the lines of the axes. */
static gboolean readAxesLineStipple(gchar **lines, int nbLines,
				    int position, VisuData *dataObj _U_, GError **error)
{
  gint stipple;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_integer(lines[0], position, &stipple, 1, error))
    return FALSE;
  axesSet_lineStipple((guint16)stipple);

  return TRUE;
}

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static void exportResourcesAxes(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_AXES_USED);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_AXES_USED);
  g_string_append_printf(data, "    %d\n", extensionAxes->used);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_AXES_COLOR);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_AXES_COLOR);
  g_string_append_printf(data, "    %4.3f %4.3f %4.3f\n",
	  axesRGB[0], axesRGB[1], axesRGB[2]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_AXES_LINE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_AXES_LINE);
  g_string_append_printf(data, "    %4.0f\n", axesLineWidth);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_AXES_STIPPLE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_AXES_STIPPLE);
  g_string_append_printf(data, "    %d\n", axesLineStipple);
  g_string_append_printf(data, "\n");
}
