/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SiteconWorkers.h"
#include "SiteconIOWorkers.h"
#include "SiteconSearchTask.h"

#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>

#include <U2Core/DNASequence.h>
#include <U2Core/DNATranslation.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/AppContext.h>
#include <U2Core/L10n.h>
#include <U2Core/Log.h>
#include <U2Core/MultiTask.h>
#include <U2Core/FailTask.h>
#include <U2Core/TaskSignalMapper.h>

#include <QtGui/QApplication>
/* TRANSLATOR U2::LocalWorkflow::SiteconSearchWorker */

namespace U2 {
namespace LocalWorkflow {

static const QString MODEL_PORT("in-sitecon");

static const QString NAME_ATTR("result-name");
static const QString STRAND_ATTR("strand");
static const QString SCORE_ATTR("min-score");
static const QString E1_ATTR("err1");
static const QString E2_ATTR("err2");

const QString SiteconSearchWorker::ACTOR_ID("sitecon-search");

void SiteconSearchWorker::registerProto() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    {
        Descriptor md(MODEL_PORT, SiteconSearchWorker::tr("Sitecon Model"), SiteconSearchWorker::tr("Profile data to search with."));
        Descriptor sd(CoreLibConstants::IN_SEQ_PORT_ID, SiteconSearchWorker::tr("Sequence"), SiteconSearchWorker::tr("Input nucleotide sequence to search in."));
        Descriptor od(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID, SiteconSearchWorker::tr("SITECON annotations"), 
            SiteconSearchWorker::tr("Annotations marking found TFBS sites."));
        
        QMap<Descriptor, DataTypePtr> modelM;
        modelM[SiteconWorkerFactory::SITECON_SLOT] = SiteconWorkerFactory::SITECON_MODEL_TYPE();
        p << new PortDescriptor(md, DataTypePtr(new MapDataType("sitecon.search.model", modelM)), true /*input*/, false, IntegralBusPort::BLIND_INPUT);
        QMap<Descriptor, DataTypePtr> seqM;
        seqM[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        p << new PortDescriptor(sd, DataTypePtr(new MapDataType("sitecon.search.sequence", seqM)), true /*input*/);
        QMap<Descriptor, DataTypePtr> outM;
        outM[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
        p << new PortDescriptor(od, DataTypePtr(new MapDataType("sitecon.search.out", outM)), false /*input*/, true /*multi*/);
    }
    {
        Descriptor nd(NAME_ATTR, SiteconSearchWorker::tr("Result annotation"), 
            SiteconSearchWorker::tr("Annotation name for marking found regions"));
        Descriptor snd(STRAND_ATTR, L10N::searchIn(),
            SiteconSearchWorker::tr("Which strands should be searched: direct(1), complement(-1) or both(0)."));
        Descriptor scd(SCORE_ATTR, SiteconSearchWorker::tr("Min score"), 
            QApplication::translate("SiteconSearchDialog", "min_err_tip", 0, QApplication::UnicodeUTF8));
        Descriptor e1d(E1_ATTR, SiteconSearchWorker::tr("Min Err1"), 
            SiteconSearchWorker::tr("Alternative setting for filtering results, minimal value of Error type I."
            "<br>Note that all thresholds (by score, by err1 and by err2) are applied when filtering results."));
        Descriptor e2d(E2_ATTR, SiteconSearchWorker::tr("Max Err2"), 
            SiteconSearchWorker::tr("Alternative setting for filtering results, max value of Error type II."
            "<br>Note that all thresholds (by score, by err1 and by err2) are applied when filtering results."));
        a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, "misc_feature");
        a << new Attribute(snd, CoreDataTypes::NUM_TYPE(), false, 0);
        a << new Attribute(scd, CoreDataTypes::NUM_TYPE(), false, 85);
        a << new Attribute(e1d, CoreDataTypes::NUM_TYPE(), false, 0.);
        a << new Attribute(e2d, CoreDataTypes::NUM_TYPE(), false, 0.001);
    }

    Descriptor desc(ACTOR_ID, tr("Search for TFBS with SITECON"), 
        tr("Searches each input sequence for transcription factor binding sites significantly similar to specified SITECON profiles."
        " In case several profiles were supplied, searches with all profiles one by one and outputs merged set of annotations for each sequence.")
        );
    ActorPrototype* proto = new IntegralBusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;

    {
        QVariantMap m; m["minimum"] = 60; m["maximum"] = 100; m["suffix"] = "%";
        delegates[SCORE_ATTR] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = 0.; m["maximum"] = 1.; m["singleStep"] = 0.1;
        delegates[E1_ATTR] = new DoubleSpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = 0.000001; m["maximum"] = 1.; m["singleStep"] = 0.001; m["decimals"] = 6;
        delegates[E2_ATTR] = new DoubleSpinBoxDelegate(m);
    }

    QVariantMap strandMap; 
    strandMap[tr("both strands")] = 0;
    strandMap[tr("direct strand")] = 1;
    strandMap[tr("complement strand")] = -1;
    delegates[STRAND_ATTR] = new ComboBoxDelegate(strandMap);
    
    proto->setPrompter(new SiteconSearchPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":sitecon/images/sitecon.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_TRANSCRIPTION(), proto);
}

QString SiteconSearchPrompter::composeRichDoc() {
    Actor* modelProducer = qobject_cast<IntegralBusPort*>(target->getPort(MODEL_PORT))->getProducer(MODEL_PORT);
    Actor* seqProducer = qobject_cast<IntegralBusPort*>(target->getPort(CoreLibConstants::IN_SEQ_PORT_ID))->getProducer(CoreLibConstants::IN_SEQ_PORT_ID);

    QString seqName = seqProducer ? tr("For each sequence from <u>%1</u>,").arg(seqProducer->getLabel()) : "";
    QString modelName = modelProducer ? tr("with all profiles provided by <u>%1</u>,").arg(modelProducer->getLabel()) : "";

    QString resultName = getRequiredParam(NAME_ATTR);

    QString strandName;
    switch (getParameter(STRAND_ATTR).toInt()) {
    case 0: strandName = SiteconSearchWorker::tr("both strands"); break;
    case 1: strandName = SiteconSearchWorker::tr("direct strand"); break;
    case -1: strandName = SiteconSearchWorker::tr("complement strand"); break;
    }

    QString doc = tr("%1 search transcription factor binding sites (TFBS) %2."
        "<br>Recognize sites with <u>similarity %3%</u>, process <u>%4</u>."
        "<br>Output the list of found regions annotated as <u>%5</u>.")
        .arg(seqName)
        .arg(modelName)
        .arg(getParameter(SCORE_ATTR).toInt())
        .arg(strandName)
        .arg(resultName);

    return doc;
}

void SiteconSearchWorker::init() {
    modelPort = ports.value(MODEL_PORT);
    dataPort = ports.value(CoreLibConstants::IN_SEQ_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID);
    dataPort->addComplement(output);
    output->addComplement(dataPort);

    strand = actor->getParameter(STRAND_ATTR)->getAttributeValue<int>();
    cfg.minPSUM = actor->getParameter(SCORE_ATTR)->getAttributeValue<int>();
    cfg.minE1 = actor->getParameter(E1_ATTR)->getAttributeValue<double>();
    cfg.maxE2 = actor->getParameter(E2_ATTR)->getAttributeValue<double>();
    resultName = actor->getParameter(NAME_ATTR)->getAttributeValue<QString>();
}

bool SiteconSearchWorker::isReady() {
    return ((!models.isEmpty() && modelPort->isEnded()) && dataPort->hasMessage()) || modelPort->hasMessage();
}

Task* SiteconSearchWorker::tick() {
    if(cfg.minPSUM > 100 || cfg.minPSUM < 60){
        return new FailTask(tr("Min score can not be less 60% or more 100%"));
    }
    if(cfg.minE1 > 1 || cfg.minE1 < 0){
        return new FailTask(tr("Min Err1 can not be less 0 or more 1"));
    }
    if(cfg.maxE2 > 1 || cfg.maxE2 < 0){
        return new FailTask(tr("Max Err2 can not be less 0 or more 1"));
    }
    if(!(strand == 1 || strand == 0 || strand == -1)){
        return new FailTask(tr("Search in strand can only be 0(both) or 1(direct) or -1(complement)"));
    }
    while (modelPort->hasMessage()) {
        models << modelPort->get().getData().toMap().value(SiteconWorkerFactory::SITECON_SLOT.getId()).value<SiteconModel>();
    }
    if (models.isEmpty() || !modelPort->isEnded() || !dataPort->hasMessage()) {
        return NULL;
    }
    DNASequence seq = dataPort->get().getData().toMap().value(BioActorLibrary::SEQ_SLOT().getId()).value<DNASequence>();
    
    if (!seq.isNull() && seq.alphabet->getType() == DNAAlphabet_NUCL) {
        SiteconSearchCfg config(cfg);
        config.complOnly = (strand < 0);
        if (strand <= 0) {
            QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation(seq.alphabet, DNATranslationType_NUCL_2_COMPLNUCL);
            if (!compTTs.isEmpty()) {
                config.complTT = compTTs.first();
            }
        }
        QList<Task*> subtasks;
        foreach(SiteconModel model, models) {
            subtasks << new SiteconSearchTask(model, seq.constData(), seq.length(), config, 0);
        }
        Task* t = new MultiTask(tr("Search TFBS in %1").arg(seq.getName()), subtasks);
        connect(new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_taskFinished(Task*)));
        return t;
    }
    QString err = tr("Bad sequence supplied to SiteconSearch: %1").arg(seq.getName());
    //if (failFast) {
        return new FailTask(err);
    /*} else {
        algoLog.error(err);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), QVariant()));
        if (dataPort->isEnded()) {
            output->setEnded();
        }
        return NULL;
    }*/
}

void SiteconSearchWorker::sl_taskFinished(Task* t) {
    QList<SharedAnnotationData> res;
    foreach(Task* sub, t->getSubtasks()) {
        SiteconSearchTask* sst = qobject_cast<SiteconSearchTask*>(sub);
        res += SiteconSearchResult::toTable(sst->takeResults(), resultName);
    }
    QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(res);
    output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
    if (dataPort->isEnded()) {
        output->setEnded();
    }
    algoLog.info(tr("Found %1 TFBS").arg(res.size())); //TODO set task description for report
}

bool SiteconSearchWorker::isDone() {
    return dataPort->isEnded();
}

} //namespace LocalWorkflow
} //namespace U2
