/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "KalignTask.h"
#include "KalignAdapter.h"
#include "KalignConstants.h"

extern "C" {
#include "kalign2/kalign2_context.h"
}

#include <U2Core/AppContext.h>
#include <U2Core/AppSettings.h>
#include <U2Core/AppResources.h>
#include <U2Core/StateLockableDataModel.h>
#include <U2Core/DocumentModel.h>
#include <U2Core/IOAdapter.h>
#include <U2Core/Counter.h>
#include <U2Core/DNASequenceObject.h>
#include <U2Core/ProjectModel.h>
#include <U2Core/AddDocumentTask.h>
#include <U2Core/LoadDocumentTask.h>
#include <U2Gui/OpenViewTask.h>

extern "C" kalign_context *getKalignContext() {
    U2::KalignContext* ctx = static_cast<U2::KalignContext*>(U2::TLSUtils::current(KALIGN_CONTEXT_ID));
    assert(ctx->d != NULL);
    return ctx->d;
}

namespace U2 {

const QString KalignMainTask::taskName(tr("Kalign"));

const QString KalignMainTask::OPTION_GAP_OPEN_PENALTY = "open-pen";
const QString KalignMainTask::OPTION_GAP_EXTENSION_PENALTY = "ext-pen";
const QString KalignMainTask::OPTION_TERMINAL_GAP_PENALTY = "term-pen";
const QString KalignMainTask::OPTION_BONUS_SCORE = "bonus";

void KalignTaskSettings::reset() {
    gapExtenstionPenalty = -1;
    gapOpenPenalty = -1;
    termGapPenalty = -1;
    secret = -1;
        inputFilePath="";
}

KalignTask::KalignTask(const MAlignment& ma, const KalignTaskSettings& _config) 
:TLSTask(tr("KALIGN alignment"), TaskFlags_FOSCOE), config(_config), inputMA(ma)
{
    GCOUNTER( cvar, tvar, "KalignTask" );
    inputSubMA = inputMA;
    resultSubMA.setAlphabet(inputSubMA.getAlphabet());
    tpm = Task::Progress_Manual;
    //TODO: add task resource usage
}

void KalignTask::_run() {
    assert(!hasErrors());
    doAlign(); 
    if (!hasErrors() && !isCanceled()) {
        assert(resultMA.getAlphabet()!=NULL);
    }
}

void KalignTask::doAlign() {
    assert(resultSubMA.isEmpty());
    KalignAdapter::align(inputSubMA, resultSubMA, stateInfo);
    if (hasErrors()) {
        return;
    }
    resultMA = resultSubMA;
}

Task::ReportResult KalignTask::report() {
    KalignContext* ctx = static_cast<KalignContext*>(taskContext);
    delete ctx->d;
    return ReportResult_Finished;
}

TLSContext* KalignTask::createContextInstance()
{
    kalign_context* ctx = new kalign_context;
    init_context(ctx, &stateInfo);
    if(config.gapOpenPenalty != -1) {
        ctx->gpo = config.gapOpenPenalty;
    }
    if(config.gapExtenstionPenalty != -1) {
        ctx->gpe = config.gapExtenstionPenalty;
    }
    if(config.termGapPenalty != -1) {
        ctx->tgpe = config.termGapPenalty;
    }
    if(config.secret != -1) {
        ctx->secret = config.secret;
    }
    return new KalignContext(ctx);
}
//////////////////////////////////////////////////////////////////////////
// KalignGObjectTask

KalignGObjectTask::KalignGObjectTask(MAlignmentObject* _obj, const KalignTaskSettings& _config) 
: Task("", TaskFlags_NR_FOSCOE), obj(_obj), lock(NULL), kalignTask(NULL), config(_config)
{
    QString aliName = obj->getDocument()->getName();
    QString tn;
    tn = tr("KALIGN align '%1'").arg(aliName);
    setTaskName(tn);
    setUseDescriptionFromSubtask(true);
    setVerboseLogMode(true);
}

KalignGObjectTask::~KalignGObjectTask() {
    assert(lock == NULL);
}

void KalignGObjectTask::prepare() {
    if (obj.isNull()) {
        stateInfo.setError(tr("object_removed"));
        return;
    }
    if (obj->isStateLocked()) {
        stateInfo.setError(tr("object_is_state_locked"));
        return;
    }

    lock = new StateLock("kalign_lock");
    obj->lockState(lock);
    kalignTask = new KalignTask(obj->getMAlignment(), config);

    addSubTask(kalignTask);
}

Task::ReportResult KalignGObjectTask::report() {
    if (lock!=NULL) {
        obj->unlockState(lock);
        delete lock;
        lock = NULL;
    }
    propagateSubtaskError();
    if (hasErrors() || isCanceled()) {
        return ReportResult_Finished;
    }
    assert(!obj.isNull());
    if (obj->isStateLocked()) {
        stateInfo.setError(tr("object_is_state_locked"));
        return ReportResult_Finished;
    }
    assert(kalignTask->inputMA.getNumRows() == kalignTask->resultMA.getNumRows());
    obj->setMAlignment(kalignTask->resultMA);    

    return ReportResult_Finished;
}

////////////////////////////////////////
//KalignWithExtFileSpecifySupportTask
KalignWithExtFileSpecifySupportTask::KalignWithExtFileSpecifySupportTask(const KalignTaskSettings& _config) :
        Task("Run Kalign alignment task", TaskFlags_NR_FOSCOE),
        config(_config)
{
    mAObject = NULL;
    currentDocument = NULL;
    saveDocumentTask = NULL;
    loadDocumentTask = NULL;
    kalignGObjectTask = NULL;
}

void KalignWithExtFileSpecifySupportTask::prepare(){
    DocumentFormatConstraints c;
    c.checkRawData = true;
    c.supportedObjectTypes += GObjectTypes::MULTIPLE_ALIGNMENT;
    c.rawData = BaseIOAdapters::readFileHeader(config.inputFilePath);
    QList<DocumentFormatId> formats = AppContext::getDocumentFormatRegistry()->selectFormats(c);
    if (formats.isEmpty()) {
        stateInfo.setError(  tr("input_format_error") );
        return;
    }

    DocumentFormatId alnFormat = formats.first();
    loadDocumentTask=
            new LoadDocumentTask(alnFormat,
                                 config.inputFilePath,
                                 AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(config.inputFilePath)));
    addSubTask(loadDocumentTask);
}
QList<Task*> KalignWithExtFileSpecifySupportTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    if(subTask->hasErrors()) {
        stateInfo.setError(subTask->getError());
        return res;
    }
    if(hasErrors() || isCanceled()) {
        return res;
    }
    if(subTask==loadDocumentTask){
        currentDocument=loadDocumentTask->takeDocument();
        assert(currentDocument!=NULL);
        assert(currentDocument->getObjects().length()==1);
        mAObject=qobject_cast<MAlignmentObject*>(currentDocument->getObjects().first());
        assert(mAObject!=NULL);
        kalignGObjectTask=new KalignGObjectTask(mAObject,config);
        res.append(kalignGObjectTask);
    }else if(subTask == kalignGObjectTask){
        saveDocumentTask = new SaveDocumentTask(currentDocument,AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(config.inputFilePath)),config.inputFilePath);
        res.append(saveDocumentTask);
    }else if(subTask==saveDocumentTask){
        Project* proj = AppContext::getProject();
        if (proj == NULL) {
            res.append(AppContext::getProjectLoader()->openProjectTask(currentDocument->getURLString(), false));
        } else {
            bool docAlreadyInProject=false;
            foreach(Document* doc, proj->getDocuments()){
                if(doc->getURL() == currentDocument->getURL()){
                    docAlreadyInProject=true;
                }
            }
            if (docAlreadyInProject) {
                res.append(new LoadUnloadedDocumentAndOpenViewTask(currentDocument));
            } else {
                // Add document to project
                res.append(new AddDocumentTask(currentDocument));
                res.append(new LoadUnloadedDocumentAndOpenViewTask(currentDocument));
            }
        }
    }
    return res;
}
Task::ReportResult KalignWithExtFileSpecifySupportTask::report(){
    return ReportResult_Finished;
}

KalignMainTask::KalignMainTask(MAlignmentObject* _obj, const MSAAlignTaskSettings & _config)
: MSAAlignTask(_obj, _config, TaskFlags_NR_FOSCOE)
{
    GCOUNTER( cvar, tvar, "KalignMainTask" );
    KalignTaskSettings config;
    config.gapOpenPenalty = _config.getCustomValue(OPTION_GAP_OPEN_PENALTY, -1).toDouble();
    config.gapExtenstionPenalty = _config.getCustomValue(OPTION_GAP_EXTENSION_PENALTY, -1).toDouble();
    config.termGapPenalty = _config.getCustomValue(OPTION_TERMINAL_GAP_PENALTY, -1).toDouble();
    config.secret = _config.getCustomValue(OPTION_BONUS_SCORE, -1).toDouble();
    addSubTask(new KalignGObjectTask(_obj, config));
}

} //namespace
