/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DocActors.h"
#include "CoreLib.h"

#include <U2Lang/CoreDataTypes.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Core/DocumentModel.h>
#include <U2Core/SaveDocumentTask.h>
#include <U2Misc/DialogUtils.h>

namespace U2 {
namespace Workflow {

/****************************
 * DocActorProto
 *****************************/
DocActorProto::DocActorProto(const DocumentFormatId& _fid, const Descriptor& _desc, const QList<PortDescriptor*>& _ports, 
                             const QList<Attribute*>& _attrs ) : IntegralBusActorPrototype(_desc, _ports, _attrs), fid(_fid) {
}

DocActorProto::DocActorProto(const Descriptor& _desc, const GObjectType& t, const QList<PortDescriptor*>& _ports,
                             const QList<Attribute*>& _attrs ) : IntegralBusActorPrototype(_desc, _ports, _attrs), type(t) {
}

bool DocActorProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params, const QString & urlAttrId ) const {
    QList<DocumentFormat*> fs;
    QString url = WorkflowUtils::getDropUrl(fs, md);
    foreach(DocumentFormat* df, fs) {
        if (fid == df->getFormatId()) {
            if (params) {
                params->insert( urlAttrId, url );
            }
            return true;
        }
    }
    return false;
}

QString DocActorProto::prepareDocumentFilter() {
    if( !fid.isEmpty() ) {
        return DialogUtils::prepareDocumentsFileFilter( fid, true );
    } else {
        assert( !type.isEmpty() );
        return DialogUtils::prepareDocumentsFileFilterByObjType( type, true );
    }
}

/****************************
 * ReadDocActorProto
 *****************************/
ReadDocActorProto::ReadDocActorProto(const DocumentFormatId& _fid, const Descriptor& _desc, const QList<PortDescriptor*>& _ports, 
                                     const QList<Attribute*>& _attrs ) : DocActorProto( _fid, _desc, _ports, _attrs ) {
    attrs << new Attribute( CoreLibConstants::URL_IN_ATTR(), CoreDataTypes::STRING_TYPE(), true );
    QMap< QString, PropertyDelegate* > delegateMap;
    delegateMap[CoreLibConstants::URL_IN_ATTR().getId()] = new URLDelegate( prepareDocumentFilter(), QString(), true );
    setEditor( new DelegateEditor( delegateMap ) );
}

bool ReadDocActorProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return DocActorProto::isAcceptableDrop( md, params, CoreLibConstants::URL_IN_ATTR().getId() );
}

/****************************
 * WriteDocActorProto
 *****************************/
WriteDocActorProto::WriteDocActorProto(const DocumentFormatId& _fid, const Descriptor& _desc, const QList<PortDescriptor*>& _ports, 
                                       const QString & portId, const QList<Attribute*>& _attrs )
: DocActorProto( _fid, _desc, _ports, _attrs ), outPortId(portId) {
    construct();
}

WriteDocActorProto::WriteDocActorProto(const Descriptor& _desc, const GObjectType & t, const QList<PortDescriptor*>& _ports, 
                                       const QString & portId, const QList<Attribute*>& _attrs )
: DocActorProto(_desc, t, _ports, _attrs), outPortId(portId) {
    construct();
}

bool WriteDocActorProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return DocActorProto::isAcceptableDrop( md, params, CoreLibConstants::URL_OUT_ATTR().getId() );
}

void WriteDocActorProto::construct() {
    attrs << new Attribute(CoreLibConstants::URL_OUT_ATTR(), CoreDataTypes::STRING_TYPE(), false );
    attrs << new Attribute(BioActorLibrary::FILE_MODE_ATTR(), CoreDataTypes::NUM_TYPE(), false, SaveDoc_Roll);

    QMap< QString, PropertyDelegate* > delegateMap;
    delegateMap[CoreLibConstants::URL_OUT_ATTR().getId()] = new URLDelegate(prepareDocumentFilter(), QString(), false );
    delegateMap[BioActorLibrary::FILE_MODE_ATTR_ID] = new FileModeDelegate(attrs.size() > 2);

    setEditor(new DelegateEditor(delegateMap));
    setValidator(new ScreenedParamValidator(CoreLibConstants::URL_OUT_ATTR().getId(), ports.first()->getId(), CoreLibConstants::URL_SLOT().getId()));
    setPortValidator(outPortId, new ScreenedSlotValidator(CoreLibConstants::URL_SLOT().getId()));
}

/****************************
 * WriteGenbankPrompter
 *****************************/
QString WriteGenbankPrompter::composeRichDoc() {
    QString outPortId = target->getInputPorts().first()->getId();
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(outPortId));
    Actor* seqProducer = input->getProducer(BioActorLibrary::SEQ_SLOT().getId());
    QString seqName = seqProducer ? tr(" sequence from <u>%1</u>").arg(seqProducer->getLabel()) : "";
    QString annName = getProducers(outPortId, BioActorLibrary::FEATURE_TABLE_SLOT().getId());
    if (!annName.isEmpty()) {
        annName = tr(" set of annotations from <u>%1</u>").arg(annName);
    }
    
    QString url = getScreenedURL( qobject_cast<IntegralBusPort*>(target->getPort(outPortId)), 
        CoreLibConstants::URL_OUT_ATTR().getId(), CoreLibConstants::URL_SLOT().getId() );
    
    QString data;
    if (seqName.isEmpty() && annName.isEmpty()) {
        QString doc = tr("Write sequence(s) in Genbank format to <u>%1</u>.").arg(url);
        return doc;
    } else if (!seqName.isEmpty() && !annName.isEmpty()) {
        data = tr("each %1 and %2").arg(seqName).arg(annName);
    } else {
        data = tr("each ") + seqName + annName;
    }
    
    return tr("Write %1 in Genbank format, to <u>%2</u>.")
        .arg(data)
        .arg(url);
}

/****************************
 * WriteFastaPrompter
 *****************************/
QString WriteFastaPrompter::composeRichDoc() {
    QString outPortId = target->getInputPorts().first()->getId();
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(outPortId));
    QString url = getScreenedURL( qobject_cast<IntegralBusPort*>(target->getPort(outPortId)),
        CoreLibConstants::URL_OUT_ATTR().getId(), CoreLibConstants::URL_SLOT().getId() );

    Actor* seqProducer = input->getProducer(BioActorLibrary::SEQ_SLOT().getId());
    if (!seqProducer) {
        QString doc = tr("Writes sequence(s) in %1 format to <u>%2</u>.").arg(format).arg(url);
        return doc;
    }

    QString doc = tr("Writes sequence(s) from <u>%1</u> in %2 format to  <u>%3</u>.")
        .arg(seqProducer->getLabel())
        .arg(format)
        .arg(url);

    return doc;
}

ActorDocument* WriteFastaPrompter::createDescription(Actor* a) {
    WriteFastaPrompter* doc = new WriteFastaPrompter(format, a);
    doc->connect(a, SIGNAL(si_labelChanged()), SLOT(sl_actorModified()));
    doc->connect(a, SIGNAL(si_modified()), SLOT(sl_actorModified()));
    foreach(Workflow::Port* input, a->getInputPorts()) {
        doc->connect(input, SIGNAL(bindingChanged()), SLOT(sl_actorModified()));
    }
    return doc;
}

/****************************
 * WriteDocPrompter
 *****************************/
QString WriteDocPrompter::composeRichDoc() {
    QString outPortId = target->getInputPorts().first()->getId();
    QString url = getScreenedURL( qobject_cast<IntegralBusPort*>(target->getPort(outPortId)), 
        CoreLibConstants::URL_OUT_ATTR().getId(), CoreLibConstants::URL_SLOT().getId() );
    
    QString producers = getProducers(outPortId, slot);
    if (producers.isEmpty()) {
        QString unsetStr = "<font color='red'>"+tr("unset")+"</font>";
        return spec.arg(unsetStr).arg(url);
    }
    return spec.arg(producers).arg(url);
}

/****************************
 * ReadDocPrompter
 *****************************/
QString ReadDocPrompter::composeRichDoc() {
    return spec.arg(getURL(CoreLibConstants::URL_IN_ATTR().getId()));
}

}//namespace Workflow
}//namespace U2
