/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RemoteBLASTWorker.h"

#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>
#include <U2Core/Log.h>
#include <U2Core/DNASequence.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/FailTask.h>

namespace U2 {
namespace LocalWorkflow {


const QString RemoteBLASTWorkerFactory::ACTOR_ID("blast-ncbi");

const QString DATABASE("db");
const QString EXPECT("e-val");
const QString MAX_HITS("max-hits");
const QString SHORT_SEQ("short-sequence");
const QString ANNOTATION_NAME("result-name");
const QString ORIGINAL_OUT("blast-output");

void RemoteBLASTWorkerFactory::init() {
    QList<PortDescriptor*> p; 
    QList<Attribute*> a;
    Descriptor ind(CoreLibConstants::IN_SEQ_PORT_ID, RemoteBLASTWorker::tr("Input sequence"), 
        RemoteBLASTWorker::tr("The sequence to search the annotations for"));
    Descriptor outd(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID, RemoteBLASTWorker::tr("Annotations"), 
        RemoteBLASTWorker::tr("Found annotations"));

    QMap<Descriptor, DataTypePtr> inM;
    inM[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
    p << new PortDescriptor(ind, DataTypePtr(new MapDataType("blast.ncbi.sequence", inM)), true);
    QMap<Descriptor, DataTypePtr> outM;
    outM[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
    p << new PortDescriptor(outd, DataTypePtr(new MapDataType("blast.ncbi.annotations", outM)), false, true);
    
    Descriptor db(DATABASE,RemoteBLASTWorker::tr("Database"),
        RemoteBLASTWorker::tr("Select the database to search through. Available databases are blastn, blastp and cdd"));
    Descriptor evalue(EXPECT,RemoteBLASTWorker::tr("Expected value"),
        RemoteBLASTWorker::tr("This parameter specifies the statistical significance threshold of reporting matches against the database sequences."));
    Descriptor hits(MAX_HITS,RemoteBLASTWorker::tr("Max hits"),
        RemoteBLASTWorker::tr("Maximum number of hits."));
    Descriptor short_seq(SHORT_SEQ,RemoteBLASTWorker::tr("Short sequence"),
        RemoteBLASTWorker::tr("Optimize search for short sequences."));
    Descriptor annotateAs(ANNOTATION_NAME,RemoteBLASTWorker::tr("Annotate as"),
        RemoteBLASTWorker::tr("Name for annotations"));
    Descriptor output(ORIGINAL_OUT, RemoteBLASTWorker::tr("BLAST output"),
        RemoteBLASTWorker::tr("Location of BLAST output file. This parameter insignificant for cdd search."));

    a << new Attribute(db,CoreDataTypes::STRING_TYPE(),true,"ncbi-blastn");
    a << new Attribute(evalue,CoreDataTypes::STRING_TYPE(),false,10);
    a << new Attribute(hits,CoreDataTypes::NUM_TYPE(),false,10);
    a << new Attribute(short_seq,CoreDataTypes::BOOL_TYPE(),false,false);
    a << new Attribute(annotateAs,CoreDataTypes::STRING_TYPE(),false);
    a << new Attribute(output, CoreDataTypes::STRING_TYPE(),false);

    Descriptor desc(ACTOR_ID, RemoteBLASTWorker::tr("Remote BLAST"), 
        RemoteBLASTWorker::tr("Finds annotations for DNA sequence in remote database")
        );
    ActorPrototype* proto = new IntegralBusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates; 

    {
        QVariantMap m;
        m["minimum"] = 1;
        m["maximum"] = 500;
        delegates[MAX_HITS] = new SpinBoxDelegate(m);
    }

    {
        QVariantMap m;
        m["ncbi-blastn"] = "ncbi-blastn";
        m["ncbi-blastp"] = "ncbi-blastp";
        m["ncbi-cdd"] = "ncbi-cdd";
        delegates[DATABASE] = new ComboBoxDelegate(m);
    }

    {
        QVariantMap m;
        m["1e-100"] = 1e-100;
        m["1e-10"] = 1e-10;
        m["1"] = 1;
        m["10"] = 10;
        m["100"] = 100;
        m["1000"] = 1000;
        delegates[EXPECT] = new ComboBoxDelegate(m);
    }

    delegates[ORIGINAL_OUT] = new URLDelegate("(*.xml)","xml file");

    proto->setPrompter(new RemoteBLASTPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":remote_blast/images/remote_db_request.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_BASIC(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new RemoteBLASTWorkerFactory());
}

QString RemoteBLASTPrompter::composeRichDoc() {
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(CoreLibConstants::IN_SEQ_PORT_ID));
    Actor* producer = input->getProducer(BioActorLibrary::SEQ_SLOT().getId());
    QString unsetStr = "<font color='red'>"+tr("unset")+"</font>";
    QString producerName = tr(" from <u>%1</u>").arg(producer ? producer->getLabel() : unsetStr);

    QString doc = tr("For sequence %1 find annotations in database <u>%2</u>")
        .arg(producerName).arg(getParameter(DATABASE).toString());
    return doc;
}

void RemoteBLASTWorker::init() {
    input = ports.value(CoreLibConstants::IN_SEQ_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID);
}

bool RemoteBLASTWorker::isReady() {
    return (input && input->hasMessage());
}

Task* RemoteBLASTWorker::tick() {
    if((actor->getParameter(ANNOTATION_NAME)->getAttributeValue<QString>()).isEmpty()){
        algoLog.details(tr("Annotations name is empty, default name used"));
    }
    
    Message inputMessage = getMessageAndSetupScriptValues(input);
    //cfg.minrl = 0;
    //cfg.maxrl = 3000;
    cfg.dbChoosen = actor->getParameter(DATABASE)->getAttributeValue<QString>().split("-").last();
    cfg.aminoT = NULL;

    int evalue = actor->getParameter(EXPECT)->getAttributeValue<int>();
    int maxHits = actor->getParameter(MAX_HITS)->getAttributeValue<int>();
    bool shortSeq = actor->getParameter(SHORT_SEQ)->getAttributeValue<bool>();

    if(evalue <= 0 ){
        algoLog.error(tr("Incorrect value for 'e-value' parameter, default value passed to schema"));
        evalue = 10;
    }

    if(cfg.dbChoosen == "cdd") {
        cfg.params = "db=cdd";
        addParametr(cfg.params,ReqParams::cdd_hits,maxHits);
        addParametr(cfg.params,ReqParams::cdd_eValue,evalue);
    }
    else {
        cfg.params = "CMD=Put";
        addParametr(cfg.params,ReqParams::database,"nr");
        addParametr(cfg.params, ReqParams::program, cfg.dbChoosen);
        QString filter;
        QString wordSize;
        if(shortSeq) {
            evalue = 1000;
            filter = "";
            if(cfg.dbChoosen == "blastn") {
                addParametr(cfg.params, ReqParams::wordSize, 7);
            }
        }
        else {
            addParametr(cfg.params, ReqParams::filter, "L");
        }
        addParametr(cfg.params, ReqParams::expect, evalue);
        
        addParametr(cfg.params, ReqParams::hits, maxHits);
    }
    DNASequence seq = inputMessage.getData().toMap().value(BioActorLibrary::SEQ_SLOT().getId()).value<DNASequence>();
    
    seq.info.clear();
    DNAAlphabet *alp = AppContext::getDNAAlphabetRegistry()->findAlphabet(seq.seq);
    /*if(seq.length()>MAX_BLAST_SEQ_LEN) {
        log.error(tr("The sequence is too long"));
        return NULL;
    }*/
    if(alp == AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::AMINO_DEFAULT())) {
        if(cfg.dbChoosen == "blastn") {
            algoLog.details(tr("Selected nucleotide database"));
            return NULL;
        }
    }
    else {
        if(cfg.dbChoosen != "blastn") {
            algoLog.details(tr("Selected amino acid database"));
            return NULL;
        }
    }
    cfg.query = seq.seq;
    cfg.retries = 60;
    cfg.filterResult = 0;
    Task* t = new RemoteBLASTTask(cfg);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

bool RemoteBLASTWorker::isDone() {
    return !input || input->isEnded();
}

void RemoteBLASTWorker::sl_taskFinished() {
    RemoteBLASTTask * t = qobject_cast<RemoteBLASTTask*>(sender());
    if (t->getState() != Task::State_Finished || t->hasErrors()) {
        return;
    }

    if(output) {
        if(actor->getParameter(DATABASE)->getAttributeValue<QString>() != "ncbi-cdd") {
            QString url = actor->getParameter(ORIGINAL_OUT)->getAttributeValue<QString>();
            if(!url.isEmpty()) {
                IOAdapterFactory * iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( BaseIOAdapters::LOCAL_FILE );
                IOAdapter * io = iof->createIOAdapter();
                if(io->open( url, IOAdapterMode_Write )) {
                    QByteArray output = t->getOutputFile();
                    io->writeBlock(output);
                    io->close();
                }
            }
        }

        QList<SharedAnnotationData> res = t->getResultedAnnotations();
        QString annName = actor->getParameter(ANNOTATION_NAME)->getAttributeValue<QString>();
        if(!annName.isEmpty()) {
            for(int i = 0; i<res.count();i++) {
                res[i]->name = annName;
            }
        }
        QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(res);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
        if (input->isEnded()) {
            output->setEnded();
        }
    }
}

}
}
