/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ORFWorker.h"

#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>

#include <U2Core/DNASequence.h>
#include <U2Core/DNATranslation.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/AppContext.h>
#include <U2Core/L10n.h>
#include <U2Core/Log.h>
#include <U2Algorithm/ORFAlgorithmTask.h>
#include <U2Core/FailTask.h>
#include <QtGui/QApplication>

/* TRANSLATOR U2::LocalWorkflow::ORFWorker */

namespace U2 {
namespace LocalWorkflow {

/*****************************
 * ORFWorkerFactory
 *****************************/
static const QString NAME_ATTR("result-name");
static const QString STRAND_ATTR("strand");
static const QString ID_ATTR("genetic-code");
static const QString LEN_ATTR("min-length");
static const QString FIT_ATTR("require-stop-codon");
static const QString INIT_ATTR("require-init-codon");
static const QString ALT_ATTR("allow-alternative-codons");

const QString ORFWorkerFactory::ACTOR_ID("orf-search");

void ORFWorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;

    {
        Descriptor id(CoreLibConstants::IN_SEQ_PORT_ID, ORFWorker::tr("Input sequences"), 
            ORFWorker::tr("A nucleotide sequence to search ORFs in. Protein sequences are skipped."));
        Descriptor od(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID, ORFWorker::tr("ORF annotations"), 
            ORFWorker::tr("A set of annotations marking ORFs found in the sequence."));
        
        QMap<Descriptor, DataTypePtr> inM;
        inM[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        p << new PortDescriptor(id, DataTypePtr(new MapDataType("orf.sequence", inM)), true /*input*/);
        QMap<Descriptor, DataTypePtr> outM;
        outM[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
        p << new PortDescriptor(od, DataTypePtr(new MapDataType("orf.annotations", outM)), false /*input*/, true /*multi*/);
    }
    {
        Descriptor nd(NAME_ATTR, ORFWorker::tr("Annotate as"), ORFWorker::tr("Name of the result annotations marking found ORFs"));
        Descriptor sd(STRAND_ATTR, L10N::searchIn(), ORFWorker::tr("Which strands should be searched: direct, complement or both."));
        Descriptor ttd(ID_ATTR, ORFWorker::tr("Genetic code"), ORFWorker::tr("Which genetic code should be used for translating the input nucleotide sequence."));
        Descriptor ld(LEN_ATTR, ORFWorker::tr("Min length"), QApplication::translate("ORFDialogBase", "min_len_tip", 0, QApplication::UnicodeUTF8));
        Descriptor fd(FIT_ATTR, ORFWorker::tr("Require stop codon"), QApplication::translate("ORFDialogBase", "must_fit_tip", 0, QApplication::UnicodeUTF8));
        Descriptor ind(INIT_ATTR, ORFWorker::tr("Require init codon"), QApplication::translate("ORFDialogBase", "must_init_tip", 0, QApplication::UnicodeUTF8));
        Descriptor ad(ALT_ATTR, ORFWorker::tr("Allow alternative codons"), QApplication::translate("ORFDialogBase", "allow_alt_tip", 0, QApplication::UnicodeUTF8));

        a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, QVariant("ORF"));
        a << new Attribute(ttd, CoreDataTypes::STRING_TYPE(), false, QVariant(DNATranslationID(1)));
        a << new Attribute(ld, CoreDataTypes::NUM_TYPE(), false, QVariant(100));
        a << new Attribute(sd, CoreDataTypes::NUM_TYPE(), false, QVariant(0));
        a << new Attribute(fd, CoreDataTypes::BOOL_TYPE(), false, QVariant(false));
        a << new Attribute(ind, CoreDataTypes::BOOL_TYPE(), false, QVariant(true));
        a << new Attribute(ad, CoreDataTypes::BOOL_TYPE(), false, QVariant(false));
    }

    Descriptor desc(ACTOR_ID, ORFWorker::tr("ORF marker"), 
        ORFWorker::tr("Finds Open Reading Frames (ORFs) in each supplied nucleotide sequence, stores found regions as annotations." 
        "<p>Protein sequences are skipped if any supplied to input."
        "<p><dfn>ORFs are DNA sequence regions that could potentially encode a protein,"
        " and usually give a good indication of the presence of a gene in the surrounding sequence.</dfn></p>"
        "<p>In the sequence, ORFs are located between a start-code sequence (initiation codon) and a stop-code sequence (termination codon),"
        " defined by the selected genetic code.</p>")
        );
    ActorPrototype* proto = new IntegralBusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    
    
    QVariantMap lenMap; lenMap["minimum"] = QVariant(0); lenMap["maximum"] = QVariant(INT_MAX); 
    lenMap["suffix"] = L10N::suffixBp();
    delegates[LEN_ATTR] = new SpinBoxDelegate(lenMap);
    
    QVariantMap strandMap; 
    strandMap[ORFWorker::tr("both strands")] = QVariant(ORFAlgorithmStrand_Both);
    strandMap[ORFWorker::tr("direct strand")] = QVariant(ORFAlgorithmStrand_Direct);
    strandMap[ORFWorker::tr("complement strand")] = QVariant(ORFAlgorithmStrand_Complement);
    delegates[STRAND_ATTR] = new ComboBoxDelegate(strandMap);

    QVariantMap idMap;
    QList<DNATranslation*> TTs = AppContext::getDNATranslationRegistry()->
        lookupTranslation(AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::NUCL_DNA_DEFAULT()), 
        DNATranslationType_NUCL_2_AMINO);
    foreach(DNATranslation* tt, TTs) {
        idMap[tt->getTranslationName()] = tt->getTranslationId();
    }
    delegates[ID_ATTR] = new ComboBoxDelegate(idMap);
    
    proto->setPrompter(new ORFPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":orf_marker/images/orf_marker.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_BASIC(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new ORFWorkerFactory());
}

/*****************************
 * ORFPrompter
 *****************************/
QString ORFPrompter::composeRichDoc() {
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(CoreLibConstants::IN_SEQ_PORT_ID));
    Actor* producer = input->getProducer(BioActorLibrary::SEQ_SLOT().getId());
    QString unsetStr = "<font color='red'>"+tr("unset")+"</font>";
    QString producerName = tr(" from <u>%1</u>").arg(producer ? producer->getLabel() : unsetStr);

    ORFAlgorithmSettings cfg;
    cfg.strand = ORFAlgorithmStrand(getParameter(STRAND_ATTR).toInt());
    cfg.minLen = getParameter(LEN_ATTR).toInt();
    cfg.mustFit = getParameter(FIT_ATTR).toBool();
    cfg.mustInit = getParameter(INIT_ATTR).toBool();
    cfg.allowAltStart = getParameter(ALT_ATTR).toBool();

    QString extra;
    if (!cfg.mustInit) {
        extra += tr(", allow ORFs <u>starting with any codon</u> other than terminator");
    } else if (cfg.allowAltStart) {
        extra += tr(", take into account <u>alternative start codons</u>");
    }
    if (cfg.mustFit) {
        extra += tr(", <u>ignore non-terminated</u> ORFs");
    }

    QString strandName;
    switch (cfg.strand) {
    case ORFAlgorithmStrand_Both: strandName = ORFWorker::tr("both strands"); break;
    case ORFAlgorithmStrand_Direct: strandName = ORFWorker::tr("direct strand"); break;
    case ORFAlgorithmStrand_Complement: strandName = ORFWorker::tr("complement strand"); break;
    }

    QString resultName = getRequiredParam(NAME_ATTR);

    QString transId = getParameter(ID_ATTR).toString();
    QString ttName = AppContext::getDNATranslationRegistry()->
        lookupTranslation(AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::NUCL_DNA_DEFAULT()), DNATranslationType_NUCL_2_AMINO, transId)->getTranslationName();


    QString doc = tr("For each nucleotide sequence%1, find ORFs in <u>%2</u> using the <u>%3</u>."
        "<br>Detect only ORFs <u>not shorter than %4 bps</u>%5."
        "<br>Output the list of found regions annotated as <u>%6</u>.")
        .arg(producerName) //sequence from Read Fasta 1
        .arg(strandName) //both strands
        .arg(ttName) //Standard Genetic Code
        .arg(cfg.minLen) //100
        .arg(extra) //  take into account alternative start codons.
        .arg(resultName);
    
    return doc;
}

/*****************************
 * ORFWorker
 *****************************/
ORFWorker::ORFWorker(Actor* a) : BaseWorker(a), input(NULL), output(NULL) {
}

void ORFWorker::init() {
    input = ports.value(CoreLibConstants::IN_SEQ_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID);
}

bool ORFWorker::isReady() {
    return (input && input->hasMessage());
}

Task* ORFWorker::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    cfg.strand = ORFAlgorithmStrand(actor->getParameter(STRAND_ATTR)->getAttributeValue<int>());
    cfg.minLen = actor->getParameter(LEN_ATTR)->getAttributeValue<int>();
    cfg.mustFit = actor->getParameter(FIT_ATTR)->getAttributeValue<bool>();
    cfg.mustInit = actor->getParameter(INIT_ATTR)->getAttributeValue<bool>();
    cfg.allowAltStart = actor->getParameter(ALT_ATTR)->getAttributeValue<bool>();
    resultName = actor->getParameter(NAME_ATTR)->getAttributeValue<QString>();
    if(resultName.isEmpty()){
        algoLog.error(tr("ORF: result name is empty, default name used"));
        resultName = "misc_feature";
    }
    transId = actor->getParameter(ID_ATTR)->getAttributeValue<QString>();
    if (cfg.minLen < 0){
        algoLog.error(tr("ORF: Incorrect value: min-length must be greater then zero"));
        return new FailTask(tr("Incorrect value: min-length must be greater then zero"));
    }
    DNASequence seq = inputMessage.getData().toMap().value(BioActorLibrary::SEQ_SLOT().getId()).value<DNASequence>();
    
    if (!seq.isNull() && seq.alphabet->getType() == DNAAlphabet_NUCL) {
        ORFAlgorithmSettings config(cfg);
        config.searchRegion.len = seq.length();
        if (config.strand != ORFAlgorithmStrand_Direct) {
            QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation(seq.alphabet, DNATranslationType_NUCL_2_COMPLNUCL);
            if (!compTTs.isEmpty()) {
                config.complementTT = compTTs.first();
            } else {
                config.strand = ORFAlgorithmStrand_Direct;
            }
        }
        config.proteinTT = AppContext::getDNATranslationRegistry()->
            lookupTranslation(seq.alphabet, DNATranslationType_NUCL_2_AMINO, transId);
        if (config.proteinTT) {
            Task* t = new ORFFindTask(config, QByteArray(seq.constData(), seq.length()));
            connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
            return t;
        }
    }
    QString err = tr("Bad sequence supplied to ORFWorker: %1").arg(seq.getName());
    //if (failFast) {
        return new FailTask(err);
    /*} else {
        algoLog.error(err);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), QVariant()));
        if (input->isEnded()) {
            output->setEnded();
        }
        return NULL;
    }*/
}

void ORFWorker::sl_taskFinished() {
    ORFFindTask* t = qobject_cast<ORFFindTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    QList<ORFFindResult> res = t->popResults();
    if (output) {
        QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(ORFFindResult::toTable(res, resultName));
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
        if (input->isEnded()) {
            output->setEnded();
        }
        algoLog.info(tr("Found %1 ORFs").arg(res.size()));
    }
}

bool ORFWorker::isDone() {
    return !input || input->isEnded();
}

void ORFWorker::cleanup() {
}

} //namespace LocalWorkflow
} //namespace U2
