/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TCoffeeWorker.h"
#include "TaskLocalStorage.h"
#include "TCoffeeSupport.h"

#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>
#include <U2Core/AppContext.h>
#include <U2Core/AppSettings.h>
#include <U2Core/UserApplicationsSettings.h>

#include <U2Core/ExternalToolRegistry.h>
#include <U2Core/Log.h>

namespace U2 {
namespace LocalWorkflow {

/****************************
 * TCoffeeWorkerFactory
 ****************************/
const QString TCoffeeWorkerFactory::ACTOR_ID("tcoffee");
const QString GAP_OPEN_PENALTY("gap-open-penalty");
const QString GAP_EXT_PENALTY("gap-ext-penalty");
const QString NUM_ITER("iterations-max-num");
const QString EXT_TOOL_PATH("path");
const QString TMP_DIR_PATH("temp-dir");

void TCoffeeWorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    Descriptor ind(CoreLibConstants::IN_MSA_PORT_ID, TCoffeeWorker::tr("Input MSA"), TCoffeeWorker::tr("Multiple sequence alignment to be processed."));
    Descriptor oud(CoreLibConstants::OUT_MSA_PORT_ID, TCoffeeWorker::tr("Multiple sequence alignment"), TCoffeeWorker::tr("Result of alignment."));
    
    QMap<Descriptor, DataTypePtr> inM;
    inM[BioActorLibrary::MA_SLOT()] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
    p << new PortDescriptor(ind, DataTypePtr(new MapDataType("tcoffee.in.msa", inM)), true /*input*/);
    QMap<Descriptor, DataTypePtr> outM;
    outM[BioActorLibrary::MA_SLOT()] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
    p << new PortDescriptor(oud, DataTypePtr(new MapDataType("tcoffee.out.msa", outM)), false /*input*/, true /*multi*/);
    
    Descriptor gop(GAP_OPEN_PENALTY, TCoffeeWorker::tr("Gap Open Penalty"),
                   TCoffeeWorker::tr("Gap Open Penalty. Must be negative, best matches get a score of 1000."));
    Descriptor gep(GAP_EXT_PENALTY, TCoffeeWorker::tr("Gap Extension Penalty"),
                   TCoffeeWorker::tr("Gap Extension Penalty. Positive values give rewards to gaps and prevent the alignment of unrelated segments."));
    Descriptor tgp(NUM_ITER, TCoffeeWorker::tr("Max Iteration"),
                   TCoffeeWorker::tr("Number of iteration on the progressive alignment<br>"
                                     "0 - no iteration, -1 - Nseq iterations"));
    Descriptor etp(EXT_TOOL_PATH, TCoffeeWorker::tr("Tool Path"),
                   TCoffeeWorker::tr("External tool path"));
    Descriptor tdp(TMP_DIR_PATH, TCoffeeWorker::tr("Temporary directory"),
                   TCoffeeWorker::tr("Directory for temporary files"));

    a << new Attribute(gop, CoreDataTypes::NUM_TYPE(), false, QVariant(-50));
    a << new Attribute(gep, CoreDataTypes::NUM_TYPE(), false, QVariant(0));
    a << new Attribute(tgp, CoreDataTypes::NUM_TYPE(), false, QVariant(0));
    a << new Attribute(etp, CoreDataTypes::STRING_TYPE(), true, QVariant("default"));
    a << new Attribute(tdp, CoreDataTypes::STRING_TYPE(), true, QVariant("default"));

    Descriptor desc(ACTOR_ID, TCoffeeWorker::tr("Align with T-Coffee"),
        TCoffeeWorker::tr("T-Coffee is a multiple sequence alignment package. "));
    ActorPrototype* proto = new IntegralBusActorPrototype(desc, p, a);

    QMap<QString, PropertyDelegate*> delegates;
    {
        QVariantMap m; m["minimum"] = int(-10000); m["maximum"] = int(0); m["singleStep"] = int(50);
        delegates[GAP_OPEN_PENALTY] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = int(-5000); m["maximum"] = int(5000); m["singleStep"] = int(10);
        delegates[GAP_EXT_PENALTY] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = int(-1); m["maximum"] = int(100);
        delegates[NUM_ITER] = new SpinBoxDelegate(m);
    }
    delegates[EXT_TOOL_PATH] = new URLDelegate("", "executable", false);
    delegates[TMP_DIR_PATH] = new URLDelegate("", "TmpDir", false, true);

    proto->setEditor(new DelegateEditor(delegates));
    proto->setPrompter(new TCoffeePrompter());
    proto->setIconPath(":external_tool_support/images/tcoffee.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_ALIGNMENT(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new TCoffeeWorkerFactory());
}

/****************************
* TCoffeePrompter
****************************/
TCoffeePrompter::TCoffeePrompter(Actor* p) : PrompterBase<TCoffeePrompter>(p) {
}
QString TCoffeePrompter::composeRichDoc() {
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(CoreLibConstants::IN_MSA_PORT_ID));
    Actor* producer = input->getProducer(CoreLibConstants::IN_MSA_PORT_ID);
    QString producerName = producer ? tr(" from %1").arg(producer->getLabel()) : "";

    QString doc = tr("For each MSA<u>%1</u>, build the alignment using <u>\"T-Coffee\"</u> and send it to output.")
        .arg(producerName);

    return doc;
}
/****************************
* TCoffeeWorker
****************************/
TCoffeeWorker::TCoffeeWorker(Actor* a) : BaseWorker(a), input(NULL), output(NULL) {
}

void TCoffeeWorker::init() {
    input = ports.value(CoreLibConstants::IN_MSA_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_MSA_PORT_ID);
}

bool TCoffeeWorker::isReady() {
    return (input && input->hasMessage());
}

Task* TCoffeeWorker::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    cfg.gapOpenPenalty=actor->getParameter(GAP_OPEN_PENALTY)->getAttributeValue<float>();
    cfg.gapExtenstionPenalty=actor->getParameter(GAP_EXT_PENALTY)->getAttributeValue<float>();
    cfg.numIterations=actor->getParameter(NUM_ITER)->getAttributeValue<int>();
    QString path=actor->getParameter(EXT_TOOL_PATH)->getAttributeValue<QString>();
    if(QString::compare(path, "default", Qt::CaseInsensitive) != 0){
        AppContext::getExternalToolRegistry()->getByName(TCOFFEE_TOOL_NAME)->setPath(path);
    }
    path=actor->getParameter(TMP_DIR_PATH)->getAttributeValue<QString>();
    if(QString::compare(path, "default", Qt::CaseInsensitive) != 0){
        AppContext::getAppSettings()->getUserAppsSettings()->setTemporaryDirPath(path);
    }
    MAlignment msa = inputMessage.getData().toMap().value(BioActorLibrary::MA_SLOT().getId()).value<MAlignment>();
    
    if( msa.isEmpty() ) {
        algoLog.error( tr("Empty msa supplied to tcoffee") );
        return NULL;
    }
    Task* t = new TCoffeeSupportTask(new MAlignmentObject(msa), cfg);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void TCoffeeWorker::sl_taskFinished() {
    TCoffeeSupportTask* t = qobject_cast<TCoffeeSupportTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    QVariant v = qVariantFromValue<MAlignment>(t->resultMA);
    output->put(Message(BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), v));
    if (input->isEnded()) {
        output->setEnded();
    }
    algoLog.info(tr("Aligned %1 with T-Coffee").arg(t->resultMA.getName()));
}

bool TCoffeeWorker::isDone() {
    return !input || input->isEnded();
}

void TCoffeeWorker::cleanup() {
}

} //namespace LocalWorkflow
} //namespace U2
