/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "EnzymesPlugin.h"
#include "FindEnzymesDialog.h"
#include "ConstructMoleculeDialog.h"
#include "DigestSequenceDialog.h"
#include "EnzymesTests.h"
#include "EnzymesQuery.h"

#include <U2Core/AppContext.h>
#include <U2Gui/MainWindow.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/GAutoDeleteList.h>

#include <U2Gui/GUIUtils.h>
#include <U2Misc/DialogUtils.h>

#include <U2View/ADVConstants.h>
#include <U2View/AnnotatedDNAView.h>
#include <U2View/ADVSequenceObjectContext.h>
#include <U2View/ADVUtils.h>


#include <U2Test/GTestFrameworkComponents.h>

#include <QtCore/QDir>
#include <QtGui/QMenu>
#include <QtGui/QMessageBox>

namespace U2 {

extern "C" Q_DECL_EXPORT Plugin * U2_PLUGIN_INIT_FUNC() {
    EnzymesPlugin * plug = new EnzymesPlugin();
    return plug;
}

EnzymesPlugin::EnzymesPlugin() : Plugin( tr("Restriction analysis"), 
                                        tr("Finds and annotates restriction sites on a DNA sequence.") ),
                                        ctxADV(NULL)
{
    if (AppContext::getMainWindow()) {
        createToolsMenu();     
        
        QList<QAction*> actions;
        actions.append(openDigestSequenceDialog);
        actions.append(openConstructMoleculeDialog);
        
        ctxADV = new EnzymesADVContext(this,actions);
        ctxADV->init();

    }

    EnzymesSelectorWidget::initSettings();

    GTestFormatRegistry* tfr = AppContext::getTestFramework()->getTestFormatRegistry();
    XMLTestFormat *xmlTestFormat = qobject_cast<XMLTestFormat*>(tfr->findFormat("XML"));
    assert(xmlTestFormat!=NULL);

    QDActorPrototypeRegistry* qdpr = AppContext::getQDActorProtoRegistry();
    qdpr->registerProto(new QDEnzymesActorPrototype());

    GAutoDeleteList<XMLTestFactory>* l = new GAutoDeleteList<XMLTestFactory>(this);
    l->qlist = EnzymeTests::createTestFactories();

    foreach(XMLTestFactory* f, l->qlist) { 
        bool res = xmlTestFormat->registerTestFactory(f);
        assert(res);
        Q_UNUSED(res);
    }
}

void EnzymesPlugin::createToolsMenu()
{
    openDigestSequenceDialog = new QAction(tr("Digest Into Fragments"), this);
    openConstructMoleculeDialog = new QAction(tr("Construct molecule"), this);
    
    MWMenuManager* menu = AppContext::getMainWindow()->getMenuManager();
    QMenu* tools = menu->getTopLevelMenu(MWMENU_TOOLS);
    
    QMenu* cloningMenu = tools->addMenu(QIcon(":core/images/dna_helix.png"), tr("Cloning"));
    
    cloningMenu->addAction(openDigestSequenceDialog);
    connect(openDigestSequenceDialog, SIGNAL(triggered()), SLOT(sl_onOpenDigestSequenceDialog()));   
    
    cloningMenu->addAction(openConstructMoleculeDialog);
    connect(openConstructMoleculeDialog, SIGNAL(triggered()), SLOT(sl_onOpenConstructMoleculeDialog()));

}

void EnzymesPlugin::sl_onOpenDigestSequenceDialog()
{
    GObjectViewWindow* w = GObjectViewUtils::getActiveObjectViewWindow();
    
    if (w == NULL) {
        QMessageBox::information(QApplication::activeWindow(), openDigestSequenceDialog->text(),
            tr("There is no active sequence object.\nTo start partition open sequence document.") );
        return;
    }

    AnnotatedDNAView* view = qobject_cast<AnnotatedDNAView*>(w->getObjectView());
    if (view == NULL) {
        QMessageBox::information(QApplication::activeWindow(), openDigestSequenceDialog->text(),
            tr("There is no active sequence object.\nTo start partition open sequence document.") );
        return;

    }

    DigestSequenceDialog dlg(view->getSequenceInFocus(), QApplication::activeWindow());
    dlg.exec();
        
    
}

void EnzymesPlugin::sl_onOpenConstructMoleculeDialog()
{
    Project* p = AppContext::getProject();
    if (p == NULL) {
        QMessageBox::information(QApplication::activeWindow(), openConstructMoleculeDialog->text(),
            tr("There is no active project.\nTo start ligation create a project or open an existing.") );
        return;

    }

    QList<DNAFragment> fragments = DNAFragment::findAvailableFragments();

    if (fragments.isEmpty()) {
        QMessageBox::information(QApplication::activeWindow(), openConstructMoleculeDialog->text(),
            tr("There are no available DNA fragments.\nDefine fragments by using \"Digest Into Fragments\" tool.") );
        return;
    }
    
    ConstructMoleculeDialog dlg(fragments, QApplication::activeWindow());
    dlg.exec();
}

EnzymesPlugin::~EnzymesPlugin()
{

}

//////////////////////////////////////////////////////////////////////////


EnzymesADVContext::EnzymesADVContext(QObject* p,const QList<QAction*>& actions) : GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID), cloningActions(actions)
{
}

void EnzymesADVContext::initViewContext(GObjectView* view) {
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
    ADVGlobalAction* a = new ADVGlobalAction(av, QIcon(":enzymes/images/enzymes.png"), tr("Find restriction sites..."), 50);
    a->addAlphabetFilter(DNAAlphabet_NUCL);
    connect(a, SIGNAL(triggered()), SLOT(sl_search()));
}


void EnzymesADVContext::sl_search() {
    GObjectViewAction* action = qobject_cast<GObjectViewAction*>(sender());
    assert(action!=NULL);
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(action->getObjectView());
    assert(av!=NULL);

    ADVSequenceObjectContext* seqCtx = av->getSequenceInFocus();
    assert(seqCtx->getAlphabet()->isNucleic());
    FindEnzymesDialog d(seqCtx);
    d.exec();
}

void EnzymesADVContext::buildMenu( GObjectView* v, QMenu* m )
{
    
    QMenu* cloningMenu = new QMenu(tr("Cloning"), m);
    cloningMenu->addActions(cloningActions);
    
    QAction* exportMenuAction = GUIUtils::findAction(m->actions(), ADV_MENU_EXPORT);
    m->insertMenu(exportMenuAction, cloningMenu);

}

} //namespace
