/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QMessageBox>
#include <QtCore/QFileInfo>
#include <QtCore/QDir>

#include <U2Core/AppContext.h>
#include <U2Core/DNATranslation.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/Task.h>

#include "EnzymeModel.h"
#include "EnzymesIO.h"
#include "EditFragmentDialog.h"

namespace U2 {


static QList<SEnzymeData> loadEnzymesData() {   
    QList<SEnzymeData> res;
    TaskStateInfo ti;

    QString dataDir = QDir::searchPaths( PATH_PREFIX_DATA ).first() + "/enzymes/";
    QString url = dataDir + DEFAULT_ENZYMES_FILE;

    if (QFileInfo(url).exists()) {
        res = EnzymesIO::readEnzymes(url, ti);
    }
    
    return res;
}

static const QList<SEnzymeData>& getEnzymesList() {
    static QList<SEnzymeData> data = loadEnzymesData();
    return data;
}

EditFragmentDialog::EditFragmentDialog( const DNAFragment& fragment, QWidget* p )
: QDialog(p), dnaFragment(fragment)
{
    
    setupUi(this);

    static const int REGION_LEN = 10;
    
    seq = dnaFragment.getSequence();
    transl = AppContext::getDNATranslationRegistry()->lookupComplementTranslation(dnaFragment.getAlphabet());
    QByteArray data(seq.toAscii());
    transl->translate(data.data(), data.length());
    trseq = QString(data);

    if (seq.length() > REGION_LEN*2) {
        QString leftSeqPart = seq.mid(0, REGION_LEN);
        QString rightSeqPart = seq.mid(seq.length() - REGION_LEN - 1);
        seq = QString("%1 ... %2").arg(leftSeqPart).arg(rightSeqPart);

        leftSeqPart = trseq.mid(0, REGION_LEN);
        rightSeqPart = trseq.mid(trseq.length() - REGION_LEN - 1);
        trseq = QString("%1 ... %2").arg(leftSeqPart).arg(rightSeqPart);
    } 

    lCustomOverhangEdit->setText(dnaFragment.getLeftOverhang());
    lCustomOverhangEdit->setMaxLength(8);
    rCustomOverhangEdit->setText(dnaFragment.getRightOverhang());
    lCustomOverhangEdit->setMaxLength(8);


    dnaFragment.getLeftTermType() == OVERHANG_TYPE_BLUNT ? lBluntButton->toggle() : lStickyButton->toggle();
    dnaFragment.getRightTermType() == OVERHANG_TYPE_BLUNT ? rBluntButton->toggle() : rStickyButton->toggle();

    updatePreview();

    connect(rStickyButton, SIGNAL(clicked()), SLOT(sl_updatePreview()));
    connect(rBluntButton, SIGNAL(clicked()), SLOT(sl_updatePreview()));
    connect(lStickyButton, SIGNAL(clicked()), SLOT(sl_updatePreview()));
    connect(lBluntButton, SIGNAL(clicked()), SLOT(sl_updatePreview()));
    connect(lResetButton, SIGNAL(clicked()), SLOT(sl_onLeftResetClicked()));
    connect(rResetButton, SIGNAL(clicked()), SLOT(sl_onRightResetClicked()));
    connect(lCustomOverhangEdit, SIGNAL(textEdited(const QString&)), SLOT(sl_updatePreview()));
    connect(rCustomOverhangEdit, SIGNAL(textEdited(const QString&)), SLOT(sl_updatePreview()));



}

void EditFragmentDialog::accept()
{
    QString lTermType = lBluntButton->isChecked() ? OVERHANG_TYPE_BLUNT : OVERHANG_TYPE_STICKY;
    dnaFragment.setLeftTermType(lTermType);
    
    QString rTermType = rBluntButton->isChecked() ? OVERHANG_TYPE_BLUNT : OVERHANG_TYPE_STICKY; 
    dnaFragment.setRightTermType(rTermType);
    
    if (lCustomOverhangBox->isChecked()) {
        if (!isValidOverhang(lCustomOverhangEdit->text())) {
            QMessageBox::warning(this, windowTitle(), tr("Invalid left overhang: unsupported alphabet!"));
            lCustomOverhangEdit->setFocus();
            lCustomOverhangEdit->setSelection(0, lCustomOverhangEdit->text().length());
            return;
        }
        
        dnaFragment.setLeftOverhang(lCustomOverhangEdit->text().toUpper());
    }
    if (rCustomOverhangBox->isChecked()) {
        if (!isValidOverhang(rCustomOverhangEdit->text())) {
            QMessageBox::warning(this, windowTitle(), tr("Invalid right overhang: unsupported alphabet!"));
            rCustomOverhangEdit->setFocus();
            rCustomOverhangEdit->setSelection(0, rCustomOverhangEdit->text().length());
            return;
        }
        dnaFragment.setRightOverhang(rCustomOverhangEdit->text().toUpper());
    }
    
    QDialog::accept();

}

void EditFragmentDialog::updatePreview()
{
    QString preview;

    preview += tr("Fragment of %1<br>").arg(dnaFragment.getSequenceDocName());

    const QString uLeftOverhang = lCustomOverhangEdit->text().toUpper();
    const QString bRightOverhang = rCustomOverhangEdit->text().toUpper();

    QByteArray urOverhangData = rBluntButton->isChecked() ? bRightOverhang.toAscii() : QByteArray();
    transl->translate(urOverhangData.data(), urOverhangData.length());
    const QString uRightOverhang(urOverhangData.toUpper());

    QByteArray blOverhangData = lBluntButton->isChecked() ? uLeftOverhang.toAscii() : QByteArray();
    transl->translate(blOverhangData.data(), blOverhangData.length());
    const QString bLeftOverhang(blOverhangData.toUpper());

    preview+=("<table cellspacing=\"10\" >");
    preview += tr("<tr> <td align=\"center\"> 5'End </td><td></td> <td align=\"center\"> 3'End </td> </tr>");

    preview += QString("<tr> <td align=\"center\" >%1</td><td align=\"center\" >%2</td><td align=\"center\" >%3</td> </tr>").
        arg(uLeftOverhang).arg(seq).arg(uRightOverhang);
    preview += QString("<tr> <td align=\"center\" >%1</td><td align=\"center\" >%2</td><td align=\"center\" >%3</td> </tr>").
        arg(bLeftOverhang).arg(trseq).arg(bRightOverhang);


    preview+=("</table>");
    previewEdit->setText(preview);
}

void EditFragmentDialog::sl_updatePreview()
{
    updatePreview();
}

void EditFragmentDialog::sl_customOverhangSet( const QString& text )
{
    updatePreview();
}

void EditFragmentDialog::sl_onLeftResetClicked()
{
    resetLeftOverhang();
    updatePreview();
}

void EditFragmentDialog::sl_onRightResetClicked()
{
    resetRightOverhang();
    updatePreview();

}

bool EditFragmentDialog::isValidOverhang( const QString& text )
{
    QByteArray seq(text.toAscii());
    DNAAlphabetRegistry *r = AppContext::getDNAAlphabetRegistry();
    DNAAlphabet *alph = r->findAlphabet(seq);
    if (alph == NULL ) {
        return false;
    }

    return alph->isNucleic() ? true : false;

}

void EditFragmentDialog::resetLeftOverhang()
{
    QString enzymeId = dnaFragment.getLeftTerminus();
    if (enzymeId.isEmpty()) {
        return;
    }
    const QList<SEnzymeData>& enzymes = getEnzymesList();
    SEnzymeData enz = EnzymesIO::findEnzymeById(enzymeId, enzymes);
    
    int startPos = dnaFragment.getFragmentRegions().first().startPos - enz->cutDirect;
    int fragLen = enz->seq.length() - enz->cutDirect - enz->cutComplement;

    QByteArray overhang = dnaFragment.getSourceSequence().mid(startPos - fragLen + 1, fragLen);

    lCustomOverhangEdit->setText(overhang);
}

void EditFragmentDialog::resetRightOverhang()
{
    QString enzymeId = dnaFragment.getRightTerminus();
    if (enzymeId.isEmpty()) {
        return;
    }
    
    const QList<SEnzymeData>& enzymes = getEnzymesList();
    SEnzymeData enz = EnzymesIO::findEnzymeById(enzymeId, enzymes);

    int startPos = dnaFragment.getFragmentRegions().last().endPos();
    int fragLen = enz->seq.length() - enz->cutComplement - enz->cutDirect;

    QByteArray overhang = dnaFragment.getSourceSequence().mid(startPos, fragLen);

    rCustomOverhangEdit->setText(overhang);
}


} // namespace
