/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <U2Core/DocumentModel.h>

#include <U2Core/AnnotationTableObject.h>
#include <U2Core/DNASequenceObject.h>
#include <U2Core/GObjectUtils.h>
#include <U2Core/GObjectRelationRoles.h>

#include "EnzymeModel.h"
#include "DNAFragment.h"

namespace U2 {

DNAFragment::DNAFragment( Annotation* fragment, DNASequenceObject* sObj, const QList<AnnotationTableObject*> relatedAnns )
    : annotatedFragment(fragment), dnaObj(sObj), relatedAnnotations(relatedAnns)
{   
    assert(fragment != NULL);
    assert(sObj != NULL);
}

bool static isDNAFragment( const Annotation* a )
{
    QString aName = a->getAnnotationName();
    if (aName.startsWith("Fragment")) {
        return true;
    } else {
        return false;
    }
}

QList<DNAFragment> DNAFragment::findAvailableFragments()
{
    QList<GObject*> aObjects = GObjectUtils::findAllObjects(UOF_LoadedOnly,GObjectTypes::ANNOTATION_TABLE);
    QList<GObject*> sObjects = GObjectUtils::findAllObjects(UOF_LoadedOnly,GObjectTypes::SEQUENCE);
    
    return findAvailableFragments(aObjects,sObjects);
}

QList<DNAFragment> DNAFragment::findAvailableFragments( const QList<GObject*>& aObjects, const QList<GObject*>& sObjects )
{
    QList<DNAFragment> fragments;
    foreach (GObject* obj , aObjects) {
        AnnotationTableObject* aObj = qobject_cast<AnnotationTableObject*>(obj);
        assert(aObj != NULL);
        QList<Annotation*> annotations = aObj->getAnnotations();
        foreach (Annotation* a, annotations) {
            if (isDNAFragment(a)) {
                // Skip negative split
                if (GObjectUtils::annotationHasNegativeSplit(a)) {
                    continue;
                }
                // Find related sequence object
                DNASequenceObject* dnaObj = NULL;
                QList<GObjectRelation> relations = aObj->getObjectRelations();
                foreach (const GObjectRelation& relation, relations ) {
                    if (relation.role != GObjectRelationRole::SEQUENCE) {
                        continue;
                    }
                    GObject* relatedObj = GObjectUtils::selectObjectByReference(relation.ref, sObjects, UOF_LoadedOnly);
                    dnaObj = qobject_cast<DNASequenceObject*>(relatedObj);
                }
                if (dnaObj == NULL) {
                    continue;
                }
                // Find related annotation tables 
                QList<GObject*> relatedAnns = GObjectUtils::findObjectsRelatedToObjectByRole(dnaObj, GObjectTypes::ANNOTATION_TABLE, GObjectRelationRole::SEQUENCE, aObjects, UOF_LoadedOnly);
                // Add fragment
                DNAFragment fragment;
                fragment.annotatedFragment = a;
                fragment.dnaObj = dnaObj;
                foreach(GObject* relAnn, relatedAnns) {
                    AnnotationTableObject* related = qobject_cast<AnnotationTableObject*>(relAnn);
                    fragment.relatedAnnotations.append(related);    
                }
                fragments.append(fragment);
            }
        }
    }
    return fragments;
}

QList<LRegion> DNAFragment::getFragmentRegions() const
{
    assert(!isEmpty());
    QList<LRegion> result = annotatedFragment->getLocation();
    QString split = annotatedFragment->findFirstQualifierValue(QUALIFIER_SPLIT);
    if (!split.isEmpty()) {
        int splitVal = split.toInt();
        LRegion oldRegion = result.first();
        LRegion addRegion(0, splitVal - oldRegion.len);
        result.append(addRegion);
    }

    return result;
}

QString DNAFragment::getName() const
{
    assert(!isEmpty());
    return annotatedFragment->getAnnotationName();
}

DNAAlphabet* DNAFragment::getAlphabet() const
{
    assert(!isEmpty());
    return dnaObj->getAlphabet();
}

QString DNAFragment::getSequenceName() const
{
    return dnaObj->getGObjectName();
}

QByteArray DNAFragment::getSequence() const
{
    assert(!isEmpty());
    LRegion region = annotatedFragment->getLocation().first();
    QByteArray seq = dnaObj->getSequence().mid(region.startPos, region.len);
    QString split = annotatedFragment->findFirstQualifierValue(QUALIFIER_SPLIT);
    if (!split.isEmpty()) {
        int splitVal = split.toInt();
        QByteArray addSeq = dnaObj->getSequence().mid(0, splitVal - region.len);
        seq.append(addSeq);
    }
    return seq;
}

QString DNAFragment::getSequenceDocName() const
{
    return dnaObj->getDocument()->getName();
}

QString DNAFragment::getLeftTerminus() const
{
    return annotatedFragment->findFirstQualifierValue(QUALIFIER_LEFT_TERM);
}

QString DNAFragment::getRightTerminus() const
{
    if (isSplitted()) {
        return annotatedFragment->findFirstQualifierValue(QUALIFIER_SPLIT_TERM);
    } else {
        return annotatedFragment->findFirstQualifierValue(QUALIFIER_RIGHT_TERM);
    }
}

bool DNAFragment::isSplitted() const
{
    return !annotatedFragment->findFirstQualifierValue(QUALIFIER_SPLIT).isEmpty();
}

QString DNAFragment::getLeftOverhang() const
{
    return annotatedFragment->findFirstQualifierValue(QUALIFIER_LEFT_OVERHANG);
}

QString DNAFragment::getRightOverhang() const
{
    if (isSplitted()) {
        return annotatedFragment->findFirstQualifierValue(QUALIFIER_SPLIT_OVERHANG);
    } else {
        return annotatedFragment->findFirstQualifierValue(QUALIFIER_RIGHT_OVERHANG);
    }
}

QString DNAFragment::getLeftTermType() const
{
    return annotatedFragment->findFirstQualifierValue(QUALIFIER_LEFT_TYPE);
}

QString DNAFragment::getRightTermType() const
{
    const QString qName = isSplitted() ? QUALIFIER_SPLIT_TYPE : QUALIFIER_RIGHT_TYPE;
    return annotatedFragment->findFirstQualifierValue(qName);
}

void DNAFragment::setLeftTermType( const QString& termType )
{
    GObjectUtils::replaceAnnotationQualfier(annotatedFragment, QUALIFIER_LEFT_TYPE, termType);
}

void DNAFragment::setRightTermType( const QString& termType )
{
    const QString qName = isSplitted() ?  QUALIFIER_SPLIT_TYPE : QUALIFIER_RIGHT_TYPE;
    GObjectUtils::replaceAnnotationQualfier(annotatedFragment, qName, termType);   
}


void DNAFragment::setLeftOverhang( const QString& overhang )
{
    GObjectUtils::replaceAnnotationQualfier(annotatedFragment, QUALIFIER_LEFT_OVERHANG, overhang);
}

void DNAFragment::setRightOverhang( const QString& overhang )
{
    const QString qName = isSplitted() ?  QUALIFIER_SPLIT_OVERHANG : QUALIFIER_RIGHT_OVERHANG;
    GObjectUtils::replaceAnnotationQualfier(annotatedFragment, qName, overhang);   
}

int DNAFragment::getLength() const
{
    assert(!isEmpty());
    QList<LRegion> regions = getFragmentRegions();
    int len = 0;
    foreach (const LRegion& r, regions) {
        len += r.len;
    }
    return len;
}

const QByteArray& DNAFragment::getSourceSequence() const
{
    assert(!isEmpty());
    return dnaObj->getSequence(); 
}


} // namespace
