/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

#include <U2Misc/DialogUtils.h>
#include <U2Core/AppContext.h>
#include <U2Core/GUrlUtils.h>
#include <U2Core/DocumentUtils.h>

#include "ConstructMoleculeDialog.h"
#include "EditFragmentDialog.h"

namespace U2 {


ConstructMoleculeDialog::ConstructMoleculeDialog(const QList<DNAFragment>& fragmentList,  QWidget* p )
: QDialog(p), fragments(fragmentList)
{
    setupUi(this);
    
    foreach (const DNAFragment& frag, fragments) {
        QString fragItem = QString("%1 (%2) %3").arg(frag.getSequenceName())
            .arg(frag.getSequenceDocName())
            .arg(frag.getName());
        fragmentListWidget->addItem(fragItem);
    }

    
    LastOpenDirHelper lod;
    GUrl url = GUrlUtils::rollFileName(lod.dir + "/new_mol.gb", DocumentUtils::getNewDocFileNameExcludesHint());
    filePathEdit->setText(url.getURLString());
    fragmentListWidget->setSelectionMode(QAbstractItemView::ExtendedSelection);
    molConstructWidget->setColumnWidth(1, molConstructWidget->width()*0.5);
   
    connect(browseButton, SIGNAL(clicked()), SLOT(sl_onBrowseButtonClicked()));
    connect(takeButton, SIGNAL(clicked()), SLOT(sl_onTakeButtonClicked()));
    connect(takeAllButton, SIGNAL(clicked()), SLOT(sl_onTakeAllButtonClicked()));
    connect(clearButton, SIGNAL(clicked()), SLOT(sl_onClearButtonClicked()));
    connect(upButton, SIGNAL(clicked()), SLOT(sl_onUpButtonClicked()) );
    connect(downButton, SIGNAL(clicked()), SLOT(sl_onDownButtonClicked()));
    connect(removeButton, SIGNAL(clicked()), SLOT(sl_onRemoveButtonClicked()));
    connect(makeCircularBox, SIGNAL(clicked()), SLOT(sl_makeCircularBoxClicked()));
    connect(makeBluntBox, SIGNAL(clicked()), SLOT(sl_forceBluntBoxClicked()) );
    connect(editFragmentButton, SIGNAL(clicked()), SLOT(sl_onEditFragmentButtonClicked()));
    
    molConstructWidget->installEventFilter(this);

}

void ConstructMoleculeDialog::accept()
{
    if (selected.isEmpty()) {
        QMessageBox::information(this, windowTitle(), tr("No fragments are selected!\n Please construct molecule from available fragments."));
        return;        
    }
    
    QList<DNAFragment> toLigate;
    foreach(int idx, selected) 
    {
        toLigate.append(fragments[idx]);
    }
        
    LigateFragmentsTaskConfig cfg;
    cfg.checkOverhangs = !makeBluntBox->isChecked();
    cfg.makeCircular = makeCircularBox->isChecked();
    cfg.docUrl = filePathEdit->text();
    cfg.openView = openViewBox->isChecked();
    cfg.saveDoc = saveImmediatlyBox->isChecked();
    cfg.annotateFragments = annotateFragmentsBox->isChecked();
    
    Task* task = new LigateFragmentsTask(toLigate, cfg); 
    AppContext::getTaskScheduler()->registerTopLevelTask(task);

    QDialog::accept();
}


void ConstructMoleculeDialog::sl_onBrowseButtonClicked()
{
    LastOpenDirHelper lod;
    lod.url = QFileDialog::getSaveFileName(this, tr("Set new molecule file name"), lod.dir, tr("Genbank (*.gb )"));
    if (!lod.url.isEmpty()) {
        GUrl result = lod.url;
        filePathEdit->setText(result.getURLString());
    }
}

void ConstructMoleculeDialog::sl_onTakeButtonClicked()
{
    QList<QListWidgetItem*> items = fragmentListWidget->selectedItems();

    foreach (QListWidgetItem* item, items) {
        int curRow = fragmentListWidget->row(item);
        if (!selected.contains(curRow)) {
            selected.append(curRow);
        }
    }

    update();

}

void ConstructMoleculeDialog::sl_onTakeAllButtonClicked()
{
    selected.clear();
    int count = fragmentListWidget->count();
    
    for (int i = 0; i < count; ++i) {
        selected.append(i);
    }
    update();   
}

void ConstructMoleculeDialog::sl_onClearButtonClicked()
{
    selected.clear();
    update();
}

void ConstructMoleculeDialog::sl_onUpButtonClicked()
{
    QTreeWidgetItem* item = molConstructWidget->currentItem();
    if (item == NULL || selected.size() == 1) {
        return;
    }

    int index = molConstructWidget->indexOfTopLevelItem(item);
    int newIndex = index - 1 == -1 ? selected.size() - 1 : index - 1;

    qSwap(selected[index], selected[newIndex]);

    update();

    molConstructWidget->setCurrentItem(molConstructWidget->topLevelItem(newIndex), true);  

}

void ConstructMoleculeDialog::sl_onDownButtonClicked()
{
    QTreeWidgetItem* item = molConstructWidget->currentItem();
    if (item == NULL || selected.size() == 1) {
        return;
    }

    int index = molConstructWidget->indexOfTopLevelItem(item);
    int newIndex = index + 1 == selected.count() ? 0 : index + 1;

    qSwap(selected[index], selected[newIndex]);
    
    update();

    molConstructWidget->setCurrentItem(molConstructWidget->topLevelItem(newIndex), true);
    
}

void ConstructMoleculeDialog::sl_onRemoveButtonClicked()
{
    QTreeWidgetItem* item = molConstructWidget->currentItem();
    if (item == NULL) {
        return;
    }
    int index = molConstructWidget->indexOfTopLevelItem(item);
    selected.removeAt(index);

    update();
}

void ConstructMoleculeDialog::update()
{
    static const QString BLUNT(tr("Blunt"));

    molConstructWidget->clear();

    foreach (int index, selected ) {
        QListWidgetItem* item = fragmentListWidget->item(index);
        assert(item != NULL);
        if (item != NULL) {
            QTreeWidgetItem* newItem = new QTreeWidgetItem(molConstructWidget);
            const DNAFragment& fragment = fragments.at(index);
            newItem->setText(0, fragment.getLeftTermType() == OVERHANG_TYPE_BLUNT ? BLUNT : fragment.getLeftOverhang());
            newItem->setText(1,  item->text());
            newItem->setText(2, fragment.getRightTermType() == OVERHANG_TYPE_BLUNT ? BLUNT : fragment.getRightOverhang());
            molConstructWidget->addTopLevelItem(newItem);
        }
    }

    bool checkTermsConsistency = !makeBluntBox->isChecked();

    if (checkTermsConsistency) { 
        QTreeWidgetItem* prevItem = NULL;
        int count = molConstructWidget->topLevelItemCount();
        for(int i = 0; i < count; ++i) {
            QTreeWidgetItem* item = molConstructWidget->topLevelItem(i);
            if (prevItem != NULL) {
                QColor color = prevItem->text(2) == item->text(0) ? Qt::green : Qt::red;
                prevItem->setTextColor(2, color);
                item->setTextColor(0, color);
            }
            prevItem = item;

        }
        if (makeCircularBox->isChecked() && count > 0) {
            QTreeWidgetItem* first = molConstructWidget->topLevelItem(0);
            QTreeWidgetItem* last = molConstructWidget->topLevelItem(count - 1);
            QColor color = first->text(0) == last->text(2) ? Qt::green : Qt::red;
            first->setTextColor(0, color);
            last->setTextColor(2, color);
        }
    }

}

void ConstructMoleculeDialog::sl_makeCircularBoxClicked()
{
    update();
}

void ConstructMoleculeDialog::sl_forceBluntBoxClicked()
{
    update();
}

void ConstructMoleculeDialog::sl_onEditFragmentButtonClicked()
{
    QTreeWidgetItem* item = molConstructWidget->currentItem();
    if (item == NULL) {
        return;
    }
    
    int idx = molConstructWidget->indexOfTopLevelItem(item);
    const DNAFragment& fragment = fragments.at(selected[idx]);

    EditFragmentDialog dlg(fragment, this);
    if (dlg.exec() == -1 ) {
        return;
    }

    update();


}



bool ConstructMoleculeDialog::eventFilter( QObject* obj , QEvent* event )
{
    if (obj == molConstructWidget && event->type() == QEvent::FocusOut) {
        molConstructWidget->clearSelection();
    }
    
    return QDialog::eventFilter(obj, event);

}




} // U2
