/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CircularViewPlugin.h"
#include "CircularView.h"
#include "CircularViewSplitter.h"

#include <U2Core/GObject.h>
#include <U2Core/DocumentModel.h>
#include <U2Core/GObjectTypes.h>
#include <U2Core/DNASequenceObject.h>
#include <U2Gui/MainWindow.h>
#include <U2Core/DocumentSelection.h>

#include <U2View/AnnotatedDNAView.h>
#include <U2View/ADVSingleSequenceWidget.h>
#include <U2View/ADVConstants.h>
#include <U2View/ADVSequenceObjectContext.h>

#include <U2Gui/GUIUtils.h>

#include <QtGui/QMessageBox>
#include <QtGui/QMenu>

namespace U2 {

extern "C" Q_DECL_EXPORT Plugin* U2_PLUGIN_INIT_FUNC() {
    if (AppContext::getMainWindow()) {
        CircularViewPlugin* plug = new CircularViewPlugin();
        return plug;
    }
    return NULL;
}


CircularViewPlugin::CircularViewPlugin() : Plugin(tr("CircularView"), tr("Enables drawing of DNA sequences using circular representation")) {
    viewCtx = new CircularViewContext(this);
    viewCtx->init();
}

CircularViewPlugin::~CircularViewPlugin() {
}

#define CIRCULAR_ACTION_NAME   "CIRCULAR_ACTION"
#define CIRCULAR_VIEW_NAME     "CIRCULAR_VIEW"

CircularViewContext::CircularViewContext(QObject* p)
: GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID), exportAction(NULL) {
}

void CircularViewContext::initViewContext(GObjectView* v) {

    exportAction = new GObjectViewAction(this, v, tr("Save circular view as image"));
    exportAction->setIcon(QIcon(":/core/images/cam2.png"));

    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(v);

    foreach(ADVSequenceWidget* w, av->getSequenceWidgets()) {
        sl_sequenceWidgetAdded(w);
    }
    connect(av, SIGNAL(si_sequenceWidgetAdded(ADVSequenceWidget*)), SLOT(sl_sequenceWidgetAdded(ADVSequenceWidget*)));
}

#define MIN_LENGTH_TO_AUTO_SHOW (1000*1000)

void CircularViewContext::sl_sequenceWidgetAdded(ADVSequenceWidget* w) {
    ADVSingleSequenceWidget* sw = qobject_cast<ADVSingleSequenceWidget*>(w);
    if (sw == NULL || sw->getSequenceObject() == NULL || !sw->getSequenceObject()->getAlphabet()->isNucleic()) {
        return;
    }
    
    CircularViewAction* action = new CircularViewAction();
    action->setIcon(QIcon(":circular_view/images/circular.png"));
    action->setCheckable(true);
    action->setChecked(false);
    action->addToMenu = true;
    action->addToBar = true;
    connect(action, SIGNAL(triggered()), SLOT(sl_showCircular()));
    
    sw->addADVSequenceWidgetAction(action);

    const DNASequence& dna = sw->getSequenceContext()->getSequenceObject()->getDNASequence();
    if (dna.length() < MIN_LENGTH_TO_AUTO_SHOW) {
        if (dna.circular) {
            action->trigger();
        }
    }

}

CircularViewSplitter* CircularViewContext::getView(GObjectView* view, bool create) {

    CircularViewSplitter* circularView = NULL;
    QList<QObject*> resources = viewResources.value(view);
    foreach(QObject* r, resources) {
        circularView = qobject_cast<CircularViewSplitter*>(r);
        if (circularView!=NULL) {
            return circularView;
        }
    }
    if (create) {
        AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
        circularView = new CircularViewSplitter(av);
        av->insertWidgetIntoSplitter(circularView);
        resources.append(circularView);
        viewResources.insert(view, resources);

        if (!exportAction) {
            initViewContext(view);
        }
        connect(exportAction, SIGNAL(triggered()), circularView, SLOT(sl_export()));
    }
    return circularView;
}

//////////////////////////////////////////////////////////////////////////
void CircularViewContext::buildMenu(U2::GObjectView *v, QMenu *m) {
    bool empty = true;
    QList<QObject*> resources = viewResources.value(v);
    foreach(QObject* r, resources) {
        CircularViewSplitter* circularView = qobject_cast<CircularViewSplitter*>(r);
        if (circularView!=NULL) {
            if(!circularView->isEmpty()) {
                empty = false;
            }
            break;
        }
    }
    if(empty) {
        return;
    }
    QMenu* exportMenu = GUIUtils::findSubMenu(m, ADV_MENU_EXPORT);
    assert(exportMenu!=NULL);
    exportMenu->addAction(exportAction);
}
//////////////////////////////////////////////////////////////////////////

void CircularViewContext::removeCircularView(GObjectView* view) {
    QList<QObject*> resources = viewResources.value(view);
    foreach(QObject* r, resources) {
        CircularViewSplitter* circularView = qobject_cast<CircularViewSplitter*>(r);
        if (circularView!=NULL) {
            assert(circularView->isEmpty());
            AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
            av->unregisterSplitWidget(circularView);
            resources.removeOne(circularView);
            viewResources.insert(view, resources);
            delete circularView;
        }
    }
}

void CircularViewContext::sl_showCircular() {
    CircularViewAction* a = qobject_cast<CircularViewAction*>(sender());
    ADVSingleSequenceWidget* sw = qobject_cast<ADVSingleSequenceWidget*>(a->seqWidget);
    assert(a!=NULL);
    if (a->isChecked()) {
        a->setText(tr("Remove circular view"));
        assert(a->view == NULL);
        CircularViewSplitter* splitter = getView(sw->getAnnotatedDNAView(), true);
        a->view = new CircularView(sw, sw->getSequenceContext());
        splitter->addView(a->view);
    } else {
        a->setText(tr("Show circular view"));
        assert(a->view!=NULL);
        CircularViewSplitter* splitter = getView(sw->getAnnotatedDNAView(), false);
        if(splitter != NULL) {
            splitter->removeView(a->view);
            delete a->view;
            if(splitter->isEmpty()) {
                removeCircularView(sw->getAnnotatedDNAView());
            }
        }
        a->view = NULL;
    }
}

CircularViewAction::CircularViewAction() : ADVSequenceWidgetAction(CIRCULAR_ACTION_NAME, tr("Show circular view")), view(NULL) 
{
}

CircularViewAction::~CircularViewAction() {
    /*if (view!=NULL) {
        delete view;
    }*/
}


}//namespace
