/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _U2_DBI_H_
#define _U2_DBI_H_

#include "U2Type.h"

#include <U2Core/U2Annotation.h>
#include <U2Core/U2Assembly.h>
#include <U2Core/U2Attribute.h>
#include <U2Core/U2Msa.h>
#include <U2Core/U2Sequence.h>

#include <QtCore/QHash>

namespace U2 {

// For the classes below, see description in class definition
class U2ObjectRDbi;
class U2ObjectRWDbi;
class U2FolderDbi;
class U2SequenceRDbi;
class U2SequenceRWDbi;
class U2AnnotationRDbi;
class U2AnnotationRWDbi;
class U2MsaRDbi;
class U2MsaRWDbi;
class U2AssemblyRDbi;
class U2AssemblyRWDbi;
class U2AttributeRDbi;
class U2AttributeRWDbi;

/** A constant to retrieve all available data. */
#define NO_LIMIT -1


/**
    Database operation status.
    If operation failed, the reason is written in 'error' var
    If operation hangs, it can be canceled with 'cancelFlag'
*/
class U2CORE_EXPORT U2DbiOpStatus {
public:
    U2DbiOpStatus(QString& err, bool& cancel) : error(err), cancelFlag(cancel) {}
    bool hasError() const {return !error.isEmpty();}
    void setError(const QString & err) { assert(error.isEmpty()); error.append(err);}
    
private:
    QString&    error;

public:
    const bool& cancelFlag;
};

/** Default helper stub for U2DbiOpStatus */
class U2CORE_EXPORT U2DbiOpStatusImpl : public U2DbiOpStatus {
public:
    U2DbiOpStatusImpl() : U2DbiOpStatus(_error, _cancelFlag){}

//private:
    QString    _error;
    bool       _cancelFlag;
};

/**
Operational state of the database.
*/
enum U2CORE_EXPORT U2DbiState {
    U2DbiState_Void,
    U2DbiState_Starting,
    U2DbiState_Ready,
    U2DbiState_Stopping
};

/**
    Database access interface. 
    Database examples: fasta file, genbank file, BAM file, SQLite file
*/
class U2CORE_EXPORT U2Dbi : public QObject {
    Q_OBJECT
public:

    //////////////////////////////////////////////////////////////////////////
    // base methods that any DBI must support


    /** 
        Boots the database up to functional state. 
        Can be called again after successful shutdown, to re-load the database.
        Some property names are reserved:
            'url' - url of the DBI
            'create'  - '1' or '0'- if DBI does not exist, asks to create instance if possible
            'user', 'password' - user and password to access to the DBI
        The 'persistentData' parameter provides database state saved from previous session in a project or workspace, if any.

        @see shutdown()
        @see si_ready()
    */
    virtual void init(const QHash<QString, QString>& properties, const QVariantMap& persistentData, U2DbiOpStatus& os) = 0;

    /** Stops the database and frees up used resources. 
    Returns persistent database state for external storage in a project or workspace, if any. 
    For example, plain-file-based DBI can store certain settings or preferences between sessions, 
    which may not fit into particular file format.

    @see init()
    @see si_closed()
    */
    virtual QVariantMap shutdown(U2DbiOpStatus& os) = 0;
    

    /** Unique database id. Used for cross-database references. */
    virtual QString getDbiId() const = 0;

    /** Returns database meta-info. Any set of properties to be shown to user */
    virtual QHash<QString, QString> getDbiMetaInfo(U2DbiOpStatus&) = 0;

    /** Returns type of the entity referenced by the given ID */
    virtual U2DataType getEntityTypeById(U2DataId id) const = 0;

    /** 
        Checks current database state. Sets error message if DBI is not functional at the moment.
        //TODO better define message to state mapping.
    */
    virtual U2DbiState checkDbiState(U2DbiOpStatus& os) = 0;

    /** Database interface to access objects. Must not be NULL! */
    virtual U2ObjectRDbi* getObjectRDbi() = 0;

    /**  Generic database interface to remove objects from database */
    virtual U2ObjectRWDbi* getObjectRWDbi() {return NULL;}

    /** Database interface to modify folder information. */
    virtual U2FolderDbi* getFolderDbi() const {return NULL;};

    /**  U2Sequence read ops. Not null if DBI supports the whole set of sequence reading operations */
    virtual U2SequenceRDbi* getSequenceRDbi() {return NULL;}

    /** U2Sequence read and write ops. Not null if DBI supports the whole set of sequence writing operations */
    virtual U2SequenceRWDbi* getSequenceRWDbi() {return NULL;}

    /**  U2Annotation read ops. Not null if DBI supports the whole set of annotation reading operations */
    virtual U2AnnotationRDbi* getAnnotationRDbi() {return NULL;}

    /** U2Annotation write ops. Not null if DBI supports the whole set of annotation writing operations */
    virtual U2AnnotationRWDbi* getAnnotationRWDbi() {return NULL;}

    /** 
        U2Msa read ops. Not null if DBI supports the whole set of msa reading operations
        U2MsaRDbi requires U2SequenceRDbi support
    */
    virtual U2MsaRDbi* getMsaRDbi() {return NULL;}

    /** 
        U2Msa read ops. Not null if DBI supports the whole set of msa reading operations
        U2MsaRWDbi requires U2SequenceRWDbi support
    */        
    virtual U2MsaRWDbi* getMsaRWDbi() {return NULL;}

    /** 
        U2Assembly read ops. Not null if DBI supports the whole set of assembly reading operations
        U2AssemblyRDbi requires U2SequenceRDbi support
    */
    virtual U2AssemblyRDbi* getAssemblyRDbi() {return NULL;}

    /** 
        U2Msa read ops. Not null if DBI supports the whole set of msa reading operations
        Support of U2MsaRWDbi requires U2SequenceRWDbi support
    */        
    virtual U2AssemblyRWDbi* getAssemblyRWDbi() {return NULL;}


    /**  U2Attribute read ops. Not null if DBI supports the whole set of attributes reading operations */
    virtual U2AttributeRDbi* getAttributeRDbi() {return NULL;}

    /**  U2Annotation write ops. Not null if DBI supports the whole set of attribute writing operations */
    virtual U2AttributeRWDbi* getAttributeRWDbi() {return NULL;}

signals:
    /** Emitted after the database completes initialization */
    void si_ready();

    /** Emitted after the database completes shutdown */
    void si_closed();
};

/** 
    Base class for all *Dbi classes provided by U2Dbi
    Contains reference to root-level dbi
*/
class U2CORE_EXPORT U2ChildDbi : public QObject {
    Q_OBJECT

protected:
    U2ChildDbi(U2Dbi* rootDbi) : QObject(rootDbi) {}

    U2Dbi* getRootDbi() const { return qobject_cast<U2Dbi*>(parent()); }
};


/**
    An interface used to remove objects from database
*/
class U2CORE_EXPORT U2ObjectRDbi : public U2ChildDbi {
    Q_OBJECT

protected:
    U2ObjectRDbi(U2Dbi* rootDbi) : U2ChildDbi(rootDbi) {}

public:
    
    /**  Returns number of top-level U2Objects in database */
    virtual qint64 countObjects(U2DbiOpStatus& os) = 0;

    /**  Returns number of top-level U2Objects with the specified type in database */
    virtual qint64 countObjects(U2DataType type, U2DbiOpStatus& os) = 0;

    /** Lists database top-level objects, starts with 'offset' and limits by 'count'. 
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2DataId> getObjects(qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;

    /** Lists database top-level objects of the specified type, starts with 'offset' and limits by 'count'. 
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2DataId> getObjects(U2DataType type, qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;

    /**  Returns parents for entity.
        If entity is object, returns other object this object is a part of
        If object is not a part of any other object and does not belongs to any folder - it's automatically removed.
     */
    virtual QList<U2DataId> getParents(U2DataId entityId, U2DbiOpStatus& os) = 0;


    /**  Returns list of folders stored in database. 
        Folders are represented as paths, separated by '/' character.
        At least one root folder is required. 
    */
    virtual QStringList getFolders(U2DbiOpStatus& os) = 0;

    /** Returns number of top-level U2Objects in folder */
    virtual qint64 countObjects(const QString& folder, U2DbiOpStatus& os) = 0;

    /** Lists database top-level objects of the specified type, starts with 'offset' and limits by 'count'. 
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2DataId> getObjects(const QString& folder, qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;

    /**  Returns all folders this object must be shown in  */
    virtual QStringList getObjectFolders(U2DataId objectId, U2DbiOpStatus& os) = 0;

signals:

    /** Emitted after top-level objects are completely removed from the database. */
    void si_objectsErased(const QList<U2DataId>& objectIds);

    // Emitted when folder is created
    void si_folderAdded(const QString& folder);

    // Emitted when folder is removed
    void si_folderRemoved(const QString& folder);

    // Emitted when some objects are moved from/to folder
    void si_folderContentChanged(const QString& folder, const QList<U2DataId>& addedObjects, const QList<U2DataId>& removedObjects);
};

/**
    An interface used to add/remove objects from database
*/
class U2CORE_EXPORT U2ObjectRWDbi : public U2ObjectRDbi {
    Q_OBJECT

protected:
    U2ObjectRWDbi(U2Dbi* rootDbi) : U2ObjectRDbi(rootDbi) {}

public:
    
    /** 
        Removes object from the specified folder. If folder is empty - removes object from all folders.
        Note: the object & all related data is automatically removed from database when
        object is not placed in any folder or is not a part of any other more complex object (ex: sequence in msa)
    */
    virtual void removeObject(U2DataId dataId, const QString& folder, U2DbiOpStatus& os) = 0;
    
    /** 
        Removes collection of objects from the specified folder. If folder is empty - removes object from all folders.
        Note: the object & all related data is automatically removed from database when
        object is not placed in any folder or is not a part of any other more complex object (ex: sequence in msa)
    */
    virtual void removeObjects(const QList<U2DataId>& dataIds, const QString& folder, U2DbiOpStatus& os) = 0;
};


/**
    An interface used to add/remove folders
    Requires U2ObjectRWDbi to present
*/
class U2CORE_EXPORT U2FolderDbi : public U2ObjectRWDbi {
    Q_OBJECT;

protected:
    U2FolderDbi(U2Dbi* rootDbi): U2ObjectRWDbi(rootDbi) {}

public:
    /** Creates folder in the database.
    The specified path must be a valid unique path, not existing in the database.
    It is not required that parent folders must exist, they are created automatically.
    */
    virtual void createFolder(const QString& path, U2DbiOpStatus& os) = 0;

    /** Removes folder. The folder must be existing path. Runs GC check for all objects in the folder */
    virtual void removeFolder(const QString& folder, U2DbiOpStatus& os) = 0;

    /** Adds objects to the specified folder.
    All objects must exist and have a top-level type.*/
    virtual void addObjectsToFolder(const QList<U2DataId>& objectIds, const QString& toFolder, U2DbiOpStatus& os) = 0;

    /** Moves objects between folders.
    'fromFolder' must be existing path containing all specified objects.
    'toFolder' must be existing path or empty string.
    If 'toFolder' is empty, removes the objects from 'fromFolder' and 
    deletes non-top-level objects without parents, if any appear in the specified list.
    Otherwise, moves the specified objects between the specified folders, omitting duplicates.
    */
    virtual void moveObjects(const QList<U2DataId>& objectIds, const QString& fromFolder, const QString& toFolder, U2DbiOpStatus& os) = 0;
};

/**
    An interface to obtain 'read' access to sequence objects
*/
class U2CORE_EXPORT U2SequenceRDbi : public U2ChildDbi {
    Q_OBJECT

protected:
    U2SequenceRDbi(U2Dbi* rootDbi) : U2ChildDbi(rootDbi){}

public:
    /** Reads sequence object from database */
    virtual U2Sequence getSequenceObject(U2DataId sequenceId, U2DbiOpStatus& os) = 0;
    
    /**  
    Reads specified sequence data region from database.
    The region must be valid region within sequence bounds.
    */
    virtual QByteArray getSequenceData(U2DataId sequenceId, const U2Region& region, U2DbiOpStatus& os) = 0;

signals:
    /** Emitted when sequence is modified */
    void si_sequenceModified(U2DataId sequenceId);
};

/**
    An interface to obtain 'write' access to sequence objects
*/
class U2CORE_EXPORT U2SequenceRWDbi : public U2SequenceRDbi {
    Q_OBJECT;

protected:
    U2SequenceRWDbi(U2Dbi* rootDbi) : U2SequenceRDbi(rootDbi){}

public:

    /**  Adds new (empty) sequence instance into database, sets the assigned id on the passed U2Sequence instance. 
        The folder must exist in the database.
        Use 'updateSequenceData' method to supply data to the created sequence.

        //TODO do we ever need to allow empty folder??
    */
    virtual void createSequenceObject(U2Sequence& sequence, const QString& folder, U2DbiOpStatus& os) = 0;

    /** 
        Updates sequence region. 
        The region must be valid region within sequence bounds.
        Note: regionToReplace length can differ from dataToInsert length, so the method can be used to add/remove sequence regions.

        //TODO think about annotations: should we fix locations automatically?? If yes, emit notifications??
        // varlax: I think this should be left to user, no automatic fixes.
    */
    virtual void updateSequenceData(U2DataId sequenceId, const U2Region& regionToReplace, const QByteArray& dataToInsert, U2DbiOpStatus& os) = 0;
};

/**
    Types of possible annotation modifications
*/
enum U2CORE_EXPORT U2AnnotationModificationType {
    /** Emitted if annotation name is changed  */
    U2AnnotationModification_NameChanged = 1, 
    
    /** Emitted if new annotation qualified is added */
    U2AnnotationModification_QualifierCreated = 2,
    
    /** Emitted if annotation qualified is removed */
    U2AnnotationModification_QualifierRemoved = 3,
    
    /** Emitted if annotation strand or location region is changed */
    U2AnnotationModification_LocationChanged = 4,
    
    /** Groups changes */
    U2AnnotationModification_RemovedFromGroup = 5,
    U2AnnotationModification_AddedToGroup = 6
};

/**
    Base class for all classes that describe annotation modification
*/
class U2AnnotationModification {
public:
    U2AnnotationModification(U2AnnotationModificationType _type, const U2Annotation& a) : type(_type), annotation(a) {}
    U2AnnotationModificationType type;
    U2Annotation annotation;
};

/**
    Group modification info: contains additional info about
*/
class U2AnnotationGroupModification : public U2AnnotationModification {
public:
    U2AnnotationGroupModification(U2AnnotationModificationType _type, const U2Annotation& a, const QString& _group) 
        : U2AnnotationModification(_type, a), group(_group) {}

    QString         group;
    U2Annotation    annotation;
};

/**
    Qualifier modification info: contains additional info to base class about qualifier modified
*/
class  U2CORE_EXPORT U2QualifierModification : public U2AnnotationModification {
public:
    U2QualifierModification(U2AnnotationModificationType _type, const U2Annotation& a, const U2Qualifier& q) 
        : U2AnnotationModification(_type, a), qualifier(q) {}

    /** Qualifier modified: added or removed */
    U2Qualifier qualifier;
};


/**
    An interface to obtain 'read' access to sequence annotations
*/
class U2CORE_EXPORT U2AnnotationRDbi : public U2ChildDbi {
    Q_OBJECT
protected:
    U2AnnotationRDbi(U2Dbi* rootDbi) : U2ChildDbi(rootDbi){}

public:
    /** 
        Returns number of annotations for the given sequence object  that belongs to the group specified
        If group name is empty - all annotations are counted
    */
    virtual qint64 countAnnotations(U2DataId sequenceId, const QString& group,  U2DbiOpStatus& os) = 0;
    
    /** 
        Returns number of annotations for the given sequence object in the given region.
        Counts all annotations whose location intersects the region.
    */
    virtual qint64 countAnnotations(U2DataId sequenceId, const U2Region& region, U2DbiOpStatus& os) = 0;

    /** 
        Returns annotations for the given sequence object that belongs to the group specified
        If group is empty searches globally in all groups
        Orders result by qualifier if not empty
        The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. 
    */
    virtual QList<U2DataId> getAnnotations(U2DataId sequenceId, const QString& group, const QString& orderByQualifier,
                                            qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;
    
    /** 
        Returns annotations for the given sequence object in the given region.
        Counts all annotations whose location intersects the region.
        Orders result by qualifier if not empty.
        The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. 
    */
    virtual QList<U2DataId> getAnnotations(U2DataId sequenceId, const U2Region& region, const QString& orderByQualifier, 
                                            qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;

    /** Reads annotation entity by id */
    virtual U2Annotation getAnnotation(U2DataId annotationId, U2DbiOpStatus& os) = 0;

signals:
    void si_annotationsCreated(QList<U2Annotation>& annotations);

    void si_annotationsRemoved(QList<U2Annotation>& annotations);

    void si_annotationModified(const U2AnnotationModification* modInfo);
};

/**
    An interface to obtain 'write' access to sequence annotations
*/
class U2CORE_EXPORT U2AnnotationRWDbi : public U2AnnotationRDbi {
    Q_OBJECT

protected:
    U2AnnotationRWDbi(U2Dbi* rootDbi) : U2AnnotationRDbi(rootDbi){}

public:

    /** Adds new annotation. Assigns Id to annotation */
    virtual void createAnnotation(U2Annotation& a, U2DbiOpStatus& os) = 0;

    /** Adds list of new annotations. Assigns Ids to annotations added */
    virtual void createAnnotations(QList<U2Annotation>& annotations, U2DbiOpStatus& os) = 0;

    
    /* Removes annotation from database */
    virtual void removeAnnotation(U2DataId annotationId, U2DbiOpStatus& os) = 0;

    /**  Removes annotations from database  */
    virtual void removeAnnotations(const QList<U2DataId>& annotationIds, U2DbiOpStatus& os) = 0;
    
    /** Changes annotations location */
    virtual U2Annotation updateLocation(U2DataId annotationId, const U2Location& location, U2DbiOpStatus& os) = 0;  
    
    /** Changes annotations name */
    virtual U2Annotation updateName(U2DataId annotationId, const QString& newName, U2DbiOpStatus& os) = 0;  
    
    /** Adds new qualifier to annotation  */
    virtual U2Annotation createQualifier(U2DataId annotationId, const U2Qualifier& q, U2DbiOpStatus& os) = 0;
    
    /** Removes existing qualifier from annotation  */    
    virtual U2Annotation removeQualifier(U2DataId annotationId, const U2Qualifier& q, U2DbiOpStatus& os) = 0; 

    /** Adds annotation to the specified group */    
    virtual U2Annotation addToGroup(U2DataId annotationId, const QString& group, U2DbiOpStatus& os) = 0; 
    
    /** 
        Removes annotation from the specified group 
        If annotation belongs to no group, it is removed
    */
    virtual U2Annotation removeFromGroup(U2DataId annotationId, const QString& group, U2DbiOpStatus& os) = 0; 
};


/**
    An interface to obtain 'read' access to multiple sequence alignment
*/
class U2CORE_EXPORT U2MsaRDbi : public U2ChildDbi {
    Q_OBJECT

protected:
    U2MsaRDbi(U2Dbi* rootDbi) : U2ChildDbi(rootDbi) {} 

public:
    /** Reads Msa objects by id */
    virtual U2Msa getMsaObject(U2DataId id, U2DbiOpStatus& os) = 0;

    /** Returns region of Msa rows. Total number of MSA rows is equal to number of sequences in MSA.
    The 'firstRow' and 'numRows' must specify valid subset of rows in the alignment. */
    virtual QList<U2MsaRow> getRows(U2DataId msaId, qint32 firstRow, qint32 numRows, U2DbiOpStatus& os) = 0;
    
    /** 
        Return number of sequences in alignment that intersect given coord.
        'Intersect' here means that first non gap character is <= coord <= last non gap character.
        The coord should be a valid coordinate within alignment bounds, i.e. non-negative and less than alignment length.
    */
    virtual qint32 countSequencesAt(U2DataId msaId, qint64 coord, U2DbiOpStatus& os) = 0;

    /** Return 'count' sequences starting with 'offset' that intersect given coordinate.
    The coord should be a valid coordinate within alignment bounds, i.e. non-negative and less than alignment length.
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2DataId> getSequencesAt(U2DataId msaId, qint64 coord, qint32 offset, qint32 count, U2DbiOpStatus& os) = 0;
    
    /** Return number of sequences in alignment that intersect given region.
    The region should be a valid region within alignment bounds, i.e. non-negative and less than alignment length.
        'Intersect' here means that first non gap character is <= coord <= last non gap character
    */
    virtual qint32 countSequencesAt(U2DataId msaId, const U2Region& r, U2DbiOpStatus& os) = 0;

    /** Return 'count' sequences starting with 'offset' that intersect given region.
    The region should be a valid region within alignment bounds, i.e. non-negative and less than alignment length.
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2DataId> getSequencesAt(U2DataId msaId, const U2Region& r, qint32 offset, qint32 count, U2DbiOpStatus& os) = 0;

    /** Return number of sequences in alignment that that have non-gap character at the given coord.
    The coord should be a valid coordinate within alignment bounds, i.e. non-negative and less than alignment length.*/
    virtual qint32 countSequencesWithoutGapAt(U2DataId msaId, qint64 coord, U2DbiOpStatus& os) = 0;
    
    /** Return 'count' sequence starting with 'offset' alignment that that have non-gap character at the given coord.
    The coord should be a valid coordinate within alignment bounds, i.e. non-negative and less than alignment length.
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2DataId> getSequencesWithoutGapAt(U2DataId msaId, qint64 coord, qint32 offset, qint32 count, U2DbiOpStatus& os) = 0;


signals:
    /** 
        Emitted when msa structure is changed.
        Is not emitted for internal sequences changes, 
            but emitted when offset of some sequence is changed, set of sequences is changed or msa length is changed
        Note: msa length can be changed as the result of sequence internal change. 
            DB must track all sequence length changes and emit valid si_msaStructureChanged signals if needed.
    */
    void si_msaStructureChanged(const U2Msa& msa); //TODO replace with U2DataId ??

    /** Emitted when content of the sequence changed */
    void si_msaSequenceChanged(const U2Msa& msa, U2DataId sequenceId); //TODO replace with U2DataId ??

};

/**
    An interface to obtain 'write' access to multiple sequence alignment
*/
class U2CORE_EXPORT U2MsaRWDbi : public U2MsaRDbi {
    Q_OBJECT;

protected:
    U2MsaRWDbi(U2Dbi* rootDbi) : U2MsaRDbi(rootDbi) {} 

public:
    /** Creates new empty Msa object  */
    virtual void createMsaObject(U2Msa& msa, const QString& folder, U2DbiOpStatus& os) = 0;
    
    /** 
        Removes sequences from MSA
        Automatically removes affected sequences that are not anymore located in some folder nor Msa object
    */
    virtual void removeSequences(U2Msa& msa, const QList<U2DataId> sequenceIds, U2DbiOpStatus& os) = 0;

    /**  Adds sequences to MSA */
    virtual void addSequences(U2Msa& msa, const QList<U2MsaRow>& rows, U2DbiOpStatus& os) = 0;

};

/**
    An interface to obtain 'read' access to assembly data
*/
class U2CORE_EXPORT U2AssemblyRDbi : public U2ChildDbi {
    Q_OBJECT

protected:
    U2AssemblyRDbi(U2Dbi* rootDbi) : U2ChildDbi(rootDbi) {} 

public:
    /** Reads assembly objects by id */
    virtual U2Assembly getAssemblyObject(U2DataId id, U2DbiOpStatus& os) = 0;

    /** 
        Return number of reads in assembly that intersect given region.
        The region should be a valid region within alignment bounds, i.e. non-negative and less than alignment length.
        'Intersect' here means that region(leftmost pos, rightmost pos) intersects with 'r'
    */
    virtual qint64 countReadsAt(U2DataId assemblyId, const U2Region& r, U2DbiOpStatus& os) = 0;

    /** Return 'count' row ids starting with 'offset' that intersect given region.
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2DataId> getReadIdsAt(U2DataId assemblyId, const U2Region& r, qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;

    /** Return 'count' rows starting with 'offset' that intersect given region.
    The 'offset' and 'count' can be arbitrarily large but should not be negative. Also, 'count' can have special value 'NO_LIMIT'. */
    virtual QList<U2AssemblyRow> getReadsAt(U2DataId assemblyId, const U2Region& r, qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;

    /** Return assembly row structure by id */
    virtual U2AssemblyRow getReadById(U2DataId rowId, U2DbiOpStatus& os) = 0;

    /** 
        Return max packed row at the given coordinate
        'Intersect' here means that region(leftmost pos, rightmost pos) intersects with 'r'
    */
    virtual qint64 getMaxPackedRow(U2DataId assemblyId, const U2Region& r, U2DbiOpStatus& os) = 0;

    /** Return reads with packed row value >= min, <= max that intersect given region */
    virtual QList<U2AssemblyRow> getReadsByRow(U2DataId assemblyId, const U2Region& r, qint64 minRow, qint64 maxRow, U2DbiOpStatus& os) = 0;


    /** Count 'length of assembly' - position of the rightmost base of all reads */
    virtual quint64 getMaxEndPos(U2DataId assemblyId, U2DbiOpStatus& os) = 0;

signals:
    /** 
        Emitted when assembly structure is changed: reads added/removed
        Is not emitted for internal sequences changes, 
    */
    void si_assemblyChanged(U2DataId assembly); //TODO replace with U2DataId ??

    /** Emitted when content of the sequence changed */
    void si_assemblyReadChanged(U2DataId assembly, U2DataId sequenceId); //TODO replace with U2DataId ??

};

/**
    An interface to obtain 'write' access to multiple sequence alignment
*/
class U2CORE_EXPORT U2AssemblyRWDbi : public U2AssemblyRDbi {
    Q_OBJECT;

protected:
    U2AssemblyRWDbi(U2Dbi* rootDbi) : U2AssemblyRDbi(rootDbi) {} 

public:
    /** Creates new empty assembly object  */
    virtual void createAssemblyObject(U2Assembly& assembly, const QString& folder, U2DbiOpStatus& os) = 0;
    
    /** 
        Removes sequences from assembly
        Automatically removes affected sequences that are not anymore accessible from folders
    */
    virtual void removeReads(U2DataId assemblyId, const QList<U2DataId>& rowIds, U2DbiOpStatus& os) = 0;

    /**  
        Adds sequences to assembly
        Rows got their ids assigned.
        If 'pack' is true, rows are packed by DBI and 'packedViewRow' is assigned
        If 'pack' is false DBI stores in DB row's 'packedViewRow' field value
    */
    virtual void addReads(U2DataId assemblyId, QList<U2AssemblyRow>& rows, bool pack, U2DbiOpStatus& os) = 0;

};



/**
    A configuration for sorting by attribute operation
*/
class U2CORE_EXPORT U2DbiSortConfig {
public:
    U2DbiSortConfig() : ascending(true), ignoreFolder(true) {}

    /** Sort column*/
    QString     sortColumnName;
    
    /** Type of the sort column */
    U2DataType  columnType;
    
    /** Tells  if sorting is ascending or descending */
    bool        ascending;
    
    /** Ignore folder information during sorting. If 'false' only objects from the given folders are sorted and put into result-set*/
    bool        ignoreFolder;
    
    /** Folder to localize sorting  */
    QString     folder;
};

/**
    An interface to obtain 'read' access to object attributes
*/
class U2CORE_EXPORT U2AttributeRDbi: public U2ChildDbi {
    Q_OBJECT

protected:
    U2AttributeRDbi(U2Dbi* rootDbi) : U2ChildDbi(rootDbi){}

public:
    /** Returns all attribute names available in the database */
    virtual QStringList getAvailableAttributeNames(U2DbiOpStatus& os) = 0;
    
    /** Returns all attribute ids for the given object */
    virtual QList<U2DataId> getObjectAttributes(const U2DataId& objectId, U2DbiOpStatus& os) = 0;

    /** Returns all attribute ids for the given object */
    virtual QList<U2DataId> getObjectPairAttributes(const U2DataId& objectId, const U2DataId& childId, U2DbiOpStatus& os) = 0;

    /** Loads int32 attribute by id */
    virtual U2Int32Attribute getInt32Attribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;

    /** Loads int64 attribute by id */
    virtual U2Int64Attribute getInt64Attribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;

    /** Loads real64 attribute by id */
    virtual U2Real64Attribute getReal64Attribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;

    /** Loads String attribute by id */
    virtual U2StringAttribute getStringAttribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;

    /** Loads byte attribute by id */
    virtual U2ByteArrayAttribute getByteArrayAttribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;
    
    /** Loads date-time attribute by id */
    virtual U2DateTimeAttribute getDateTimeAttribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;

    /** Loads range int32-values attribute by id */
    virtual U2RangeInt32StatAttribute getRangeInt32StatAttribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;

    /** Loads range real64-values attribute by id */
    virtual U2RangeReal64StatAttribute getRangeReal64StatAttribute(U2DataId attributeId, U2DbiOpStatus& os) = 0;

    /** Sorts all objects in database according to U2DbiSortConfig provided  */
    virtual QList<U2DataId> sort(const U2DbiSortConfig& sc, qint64 offset, qint64 count, U2DbiOpStatus& os) = 0;
};

/** 
    An interface to create / remove attributes
    Note: when attribute is created, the old one with the same name must be automatically deleted
*/
class U2CORE_EXPORT U2AttributeRWDbi: public U2AttributeRDbi {
    Q_OBJECT

protected:
    U2AttributeRWDbi(U2Dbi* rootDbi) : U2AttributeRDbi(rootDbi){}

public:
    /** Removes attribute from database */
    virtual void removeAttribute(const U2DataId& attributeId, U2DbiOpStatus& os) = 0;
    
    /** Creates int32 attribute in database. ObjectId must be already set in attribute and present in the same database */
    virtual void createInt32Attribute(U2Int32Attribute& a, U2DbiOpStatus& os) = 0;

    /** Creates int64 attribute in database. ObjectId must be already set in attribute and present in the same database */    
    virtual void createInt64Attribute(U2Int64Attribute& a, U2DbiOpStatus& os) = 0;

    /** Creates real64 attribute in database. ObjectId must be already set in attribute and present in the same database */    
    virtual void createReal64Attribute(U2Real64Attribute& a, U2DbiOpStatus& os) = 0;

    /** Creates String attribute in database. ObjectId must be already set in attribute and present in the same database */    
    virtual void createStringAttribute(U2StringAttribute& a, U2DbiOpStatus& os) = 0;

    /** Creates Byte attribute in database. ObjectId must be already set in attribute and present in the same database */    
    virtual void createByteArrayAttribute(U2ByteArrayAttribute& a, U2DbiOpStatus& os) = 0;

    /** Creates Date-time attribute in database. ObjectId must be already set in attribute and present in the same database */    
    virtual void createDateTimeAttribute(U2DateTimeAttribute& a, U2DbiOpStatus& os) = 0;

    /** Creates range int32-values attribute in database. ObjectId must be already set in attribute and present in the same database */    
    virtual void createRangeInt32StatAttribute(U2RangeInt32StatAttribute& a, U2DbiOpStatus& os) = 0;

    /** Creates range real32-values attribute in database. ObjectId must be already set in attribute and present in the same database */    
    virtual void createRangeReal64StatAttribute(U2RangeReal64StatAttribute& a, U2DbiOpStatus& os) = 0;
};

} //namespace

#endif
