/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "Overview.h"

#include "ADVSingleSequenceWidget.h"
#include "PanView.h"
#include "DetView.h"

#include <U2Core/DNASequenceSelection.h>

#include <U2Core/AppContext.h>

#include <U2Core/AnnotationTableObject.h>
#include <U2Core/AnnotationSettings.h>

#include <U2Misc/GScrollBar.h>
#include <U2Misc/GraphUtils.h>

#include <QtGui/QVBoxLayout>
#include <QtGui/QToolTip>

namespace U2 {

Overview::Overview(QWidget *p, ADVSequenceObjectContext *ctx) : GSequenceLineView(p, ctx) {
    renderArea = new OverviewRenderArea(this);
    visibleRange = LRegion(0, ctx->getSequenceLen());
    renderArea->setMouseTracking(true);

    ADVSingleSequenceWidget* ssw = qobject_cast<ADVSingleSequenceWidget*>(p);
    panView = ssw->getPanView();
    detView = ssw->getDetView();

    tb = new QToolButton(this);
    tb->setFixedWidth(16);
    tb->setFixedHeight(20);
    tb->setCheckable(true);
    tb->setIcon(QIcon(":core/images/sum.png"));
    tb->setToolTip(tr("Toggle annotation density graph"));

    connect(panView, SIGNAL(si_visibleRangeChanged()), SLOT(sl_visibleRangeChanged()));
    connect(detView, SIGNAL(si_visibleRangeChanged()), SLOT(sl_visibleRangeChanged()));
    connect(tb, SIGNAL(pressed()), SLOT(sl_tbToggled()));
    connect(ctx, SIGNAL(si_annotationObjectAdded(AnnotationTableObject*)),
        SLOT(sl_annotationObjectAdded(AnnotationTableObject*)));
    connect(ctx, SIGNAL(si_annotationObjectRemoved(AnnotationTableObject*)),
        SLOT(sl_annotationObjectRemoved(AnnotationTableObject*)));
    foreach(AnnotationTableObject* at, ctx->getAnnotationObjects()) {
        connect(at, SIGNAL(si_onAnnotationsAdded(const QList<Annotation*>&)),
            SLOT(sl_annotationsAdded(const QList<Annotation*>&)));
        connect(at, SIGNAL(si_onAnnotationsRemoved(const QList<Annotation*>&)),
            SLOT(sl_annotationsRemoved(const QList<Annotation*>&)));
        connect(at, SIGNAL(si_onAnnotationModified(const AnnotationModification&)),
            SLOT(sl_annotationModified(const AnnotationModification&)));
        connect(AppContext::getAnnotationsSettingsRegistry(),
            SIGNAL(si_annotationSettingsChanged(const QStringList&)),
            SLOT(sl_onAnnotationSettingsChanged(const QStringList&)));
    }
    connect(ctx->getSequenceGObject(), SIGNAL(si_sequenceChanged()), SLOT(sl_sequenceChanged()));
    sl_visibleRangeChanged();

    pack();
}

void Overview::sl_annotationObjectAdded(AnnotationTableObject* obj) {
    connect(obj, SIGNAL(si_onAnnotationsAdded(const QList<Annotation*>&)),
        SLOT(sl_annotationsAdded(const QList<Annotation*>&)));
    connect(obj, SIGNAL(si_onAnnotationsRemoved(const QList<Annotation*>&)),
        SLOT(sl_annotationsRemoved(const QList<Annotation*>&)));
    connect(obj, SIGNAL(si_onAnnotationModified(const AnnotationModification&)),
        SLOT(sl_annotationModified(const AnnotationModification&)));
    addUpdateFlags(GSLV_UF_AnnotationsChanged);
    update();
}

void Overview::sl_annotationObjectRemoved(AnnotationTableObject* obj) {
    Q_UNUSED(obj);

    addUpdateFlags(GSLV_UF_AnnotationsChanged);
    update();
}

void Overview::sl_annotationsAdded(const QList<Annotation*>& a) {
    Q_UNUSED(a);

    addUpdateFlags(GSLV_UF_AnnotationsChanged);
    update();
}

void Overview::sl_annotationsRemoved(const QList<Annotation*>& a) {
    Q_UNUSED(a);

    addUpdateFlags(GSLV_UF_AnnotationsChanged);
    update();
}

void Overview::sl_annotationModified(const AnnotationModification& md) {
    if (md.type==AnnotationModification_LocationChanged) {
        addUpdateFlags(GSLV_UF_AnnotationsChanged);
        update();
    }
}

void Overview::sl_onAnnotationSettingsChanged(const QStringList& changedSettings) {
    Q_UNUSED(changedSettings);

    addUpdateFlags(GSLV_UF_AnnotationsChanged);
    update();
}

void Overview::sl_sequenceChanged() {
    addUpdateFlags(GSLV_UF_NeedCompleteRedraw);
    update();
}

void Overview::pack() {
    assert(layout() == NULL);
    QHBoxLayout *layout = new QHBoxLayout();
    layout->setMargin(0);
    layout->setSpacing(0);
    layout->addWidget(renderArea);
    layout->addWidget(tb);
    setLayout(layout);
    scrollBar->setHidden(true);
    setFixedHeight(renderArea->height());
}

void Overview::sl_tbToggled() {
    OverviewRenderArea* ra = qobject_cast<OverviewRenderArea*>(renderArea);
    ra->showGraph = !ra->showGraph;
    addUpdateFlags(GSLV_UF_NeedCompleteRedraw);
    update();
}

void Overview::sl_visibleRangeChanged() {
    renderArea->update();
}

void Overview::mousePressEvent(QMouseEvent *me) {
    if (me->buttons() & Qt::LeftButton) {
        OverviewRenderArea* ra = static_cast<OverviewRenderArea*>(renderArea);
        QPoint renderAreaPos = toRenderAreaPoint(me->pos());

        //make selection if shift was pressed
        if(me->modifiers().testFlag(Qt::ShiftModifier)) {
            lastPressPos=ra->coordToPos(renderAreaPos.x());
            QWidget::mousePressEvent(me);
            return;
        }

        QRectF panSlider(ra->getPanSlider());
        QRectF detSlider(ra->getDetSlider());
        panSliderClicked = panSlider.contains(renderAreaPos);
        panSliderMovedRight = (panSlider.topRight().x() - renderAreaPos.x() < 10) && panSliderClicked;
        panSliderMovedLeft = (renderAreaPos.x() - panSlider.topLeft().x() < 10) && panSliderClicked;
        offset = renderArea->coordToPos(me->pos().x()) - renderArea->coordToPos(panSlider.left());
        //don't process detSlider when details view is collapsed
        ADVSingleSequenceWidget* parent = qobject_cast<ADVSingleSequenceWidget*>(parentWidget());
        assert(parent);
        if(parent->isDetViewCollapsed()) {
            detSliderClicked = false;
        }
        else {
            detSliderClicked = detSlider.contains(renderAreaPos);
        }
        //int panW = panSlider.width();
        //int raW = ra->width();
        if(panSliderClicked&&detSliderClicked) {
            /*panSliderClicked = panSlider.width()<detSlider.width();
            detSliderClicked = !panSliderClicked;*/
            panSliderClicked = false;
            detSliderClicked = true;
        }

        if(panSliderClicked) {
            mousePosToSlider = QPoint(renderAreaPos - panSlider.topLeft().toPoint());
        }
        else if (detSliderClicked) {
            mousePosToSlider = QPoint(renderAreaPos - detSlider.center().toPoint());
        }
        else {
            int seqLen = ctx->getSequenceLen();
            int panVisLen = panView->getVisibleRange().len;
            int detVisLen = detView->getVisibleRange().len;

            int panPos = renderArea->coordToPos(renderAreaPos.x() - panSlider.width()/2);
            panPos = qBound(0, panPos, seqLen-panVisLen);
            panView->setVisibleRange(LRegion(panPos, panVisLen));

            int detPos = renderArea->coordToPos(renderAreaPos.x());
            detPos = qBound(0, detPos, seqLen-detVisLen);
            detView->setVisibleRange(LRegion(detPos, detVisLen));
        }
    }
    QWidget::mousePressEvent(me);
}

void Overview::mouseReleaseEvent(QMouseEvent* me) {
    lastPressPos=-1;
    panSliderMovedRight = false;
    panSliderMovedLeft = false;
    QWidget::mouseReleaseEvent(me);
}

void Overview::mouseMoveEvent(QMouseEvent *me) {
    
    QPoint renderAreaPos = toRenderAreaPoint(me->pos());
    OverviewRenderArea* ra = static_cast<OverviewRenderArea*>(renderArea);
    QRectF panSlider(ra->getPanSlider());

    if(((panSlider.topRight().x() - renderAreaPos.x() < 10 && panSlider.topRight().x() - renderAreaPos.x() > -5)
        || (renderAreaPos.x() - panSlider.topLeft().x()> -5 && renderAreaPos.x() - panSlider.topLeft().x() < 10)) && panView->isVisible()) {
        setCursor(QCursor(Qt::SizeHorCursor));
    } else {
        setCursor(QCursor(Qt::ArrowCursor));
    }

    if (me->buttons() & Qt::LeftButton) {
        int pos = renderArea->coordToPos(renderAreaPos.x() - mousePosToSlider.x());
        if(lastPressPos!=-1) {//make selection
            LRegion selection;
            int mousePos = renderArea->coordToPos(renderAreaPos.x());
            int selLen = mousePos-lastPressPos;
            if(selLen>0) {
                selection.startPos=lastPressPos;
                selection.len=selLen;
            }
            else {
                selection.startPos=mousePos;
                selection.len=-selLen;
            }
            setSelection(selection);
            QWidget::mouseMoveEvent(me);
            return;
        }

        if(panView->isVisible()) {
            if(panSliderMovedRight) {
                OverviewRenderArea* ra = static_cast<OverviewRenderArea*>(renderArea);
                
                QRectF panSlider(ra->getPanSlider());
                qreal length = me->pos().x() - panSlider.right();
                int panVisLen = renderArea->coordToPos(qAbs(length));
                if(length < 0) {
                    panVisLen *= -1;
                }

                panVisLen = panView->getVisibleRange().len + panVisLen;
                pos = renderArea->coordToPos(panSlider.left()) ;
                if(panVisLen > 0 && panView->getSeqLen() >= (panVisLen + pos)) {
                    panView->setVisibleRange(LRegion(pos, panVisLen));
                }
            } else if(panSliderMovedLeft) {
                OverviewRenderArea* ra = static_cast<OverviewRenderArea*>(renderArea);

                QRectF panSlider(ra->getPanSlider());
                int length = panSlider.left() - me->pos().x();
                int panVisLen = renderArea->coordToPos(qAbs(length));
                if(length < 0) {
                    panVisLen *= -1;
                }

                panVisLen = panView->getVisibleRange().len + panVisLen + offset;
                pos = panView->getVisibleRange().endPos() - panVisLen;
                if(panVisLen > 0 && pos > 0 && (panVisLen + pos) <= ctx->getSequenceLen()) {
                    panView->setVisibleRange(LRegion(pos, panVisLen));
                }
            } else if(panSliderClicked) {
                int panVisLen = panView->getVisibleRange().len;
                pos = qBound(0, pos, ctx->getSequenceLen() - panVisLen);
                panView->setVisibleRange(LRegion(pos, panVisLen));
            }
        }
        if (detSliderClicked) {
            int detVisLen = detView->getVisibleRange().len;
            pos = qBound(0, pos, ctx->getSequenceLen() - detVisLen);
            detView->setVisibleRange(LRegion(pos, detVisLen));
        }
    }
    QWidget::mouseMoveEvent(me);
}

void Overview::mouseDoubleClickEvent(QMouseEvent* me) {
    if(me->buttons() & Qt::LeftButton) {
        OverviewRenderArea* ra = static_cast<OverviewRenderArea*>(renderArea);

        QRectF panSlider(ra->getPanSlider());
        int panVisLen = panView->getVisibleRange().len;
        QPoint renderAreaPos = toRenderAreaPoint(me->pos());
        int panPos = ra->coordToPos(renderAreaPos.x() - panSlider.width()/2);
        panPos = qBound(0, panPos, ctx->getSequenceLen() - panVisLen);
        panView->setVisibleRange(LRegion(panPos, panVisLen));
        
        //don't process detSlider when details view is collapsed
        ADVSingleSequenceWidget* parent = qobject_cast<ADVSingleSequenceWidget*>(parentWidget());
        assert(parent);
        if(!parent->isDetViewCollapsed()) {
            QRectF detSlider(ra->getDetSlider());
            int detVisLen = detView->getVisibleRange().len;
            int detPos = ra->coordToPos(renderAreaPos.x());
            detPos = qBound(0, detPos, ctx->getSequenceLen() - detVisLen);
            detView->setVisibleRange(LRegion(detPos, detVisLen));
        }

        panSliderClicked=false;
        detSliderClicked=false;
        panSliderMovedRight = false;
        panSliderMovedLeft = false;
    }
    QWidget::mouseDoubleClickEvent(me);
}

void Overview::wheelEvent(QWheelEvent *we) {
    bool renderAreaWheel = QRect(renderArea->x(), renderArea->y(),
        renderArea->width(), renderArea->height()).contains(we->pos());
    if (!renderAreaWheel) {
        QWidget::wheelEvent(we);
        return;
    }
    setFocus();
    bool toMin = we->delta() > 0;
    QAction* zoomAction = toMin ? panView->getZoomInAction() : panView->getZoomOutAction();
    if (zoomAction != NULL) {
        zoomAction->activate(QAction::Trigger);
    }
}

bool Overview::event(QEvent* e) {
    if (e->type() == QEvent::ToolTip) {
        QHelpEvent* he = static_cast<QHelpEvent*>(e);
        QString tip = createToolTip(he);
        if (!tip.isEmpty()) {
            QToolTip::showText(he->globalPos(), tip);
        }
    }
    return GSequenceLineView::event(e);
}

QString Overview::createToolTip(QHelpEvent *he) {
    OverviewRenderArea* ra = qobject_cast<OverviewRenderArea*>(renderArea);
    QPoint renderAreaPos = toRenderAreaPoint(he->pos());
    int halfChar = ra->getCurrentScale()/2;
    int pos = ra->coordToPos(renderAreaPos.x() + halfChar);
    int pos2 = ra->coordToPos(renderAreaPos.x() + halfChar +1);
    int delta=0;
    if(pos2-1>pos) {
        delta=pos2-pos-1;
    }
    QString tip = tr("Position ")+QString::number(pos);
    if(delta!=0) {
        tip+=".."+QString::number(pos+delta);
    }
    if(ra->showGraph) {
        int density = ra->getAnnotationDensity(pos);
        for (int i=pos;i<=pos+delta;++i) {
            int nextPosDensity = ra->getAnnotationDensity(i);
            density=qMax(density, nextPosDensity);
        }
        tip+="\n"+tr("Annotation density ")+QString::number(density);
    }
    return tip;
}

//////////////////////////////////////////////////////////////////////////
/// GlobalViewRenderArea
#define ANNOTATION_GRAPH_HEIGHT 4
#define RULER_NOTCH_SIZE 2
#define SLIDER_MIN_WIDTH 3
#define ARROW_WIDTH 12
#define ARROW_HEIGHT 10
#define ARROW_TOP_PAD ANNOTATION_GRAPH_HEIGHT + 2*RULER_NOTCH_SIZE
#define PEN_WIDTH 1

#define RENDER_AREA_HEIGHT lineHeight + ANNOTATION_GRAPH_HEIGHT
OverviewRenderArea::OverviewRenderArea(Overview *p)  : GSequenceLineViewRenderArea(p) {
    setFixedHeight(RENDER_AREA_HEIGHT);
    QLinearGradient gradient(0, 0, 0, 1); //vertical
    gradient.setCoordinateMode(QGradient::ObjectBoundingMode);
    gradient.setColorAt(0.00, QColor(255, 255, 255, 120));
    gradient.setColorAt(0.50, QColor(  0,   0,   0,   0));
    gradient.setColorAt(0.70, QColor(  0,   0,   0,   0));
    gradient.setColorAt(1.00, QColor(  0,   0,   0,  70));
    gradientMaskBrush = QBrush(gradient);
    showGraph=false;
}

//pos [1, seqLen]; annotationsOnPos [0, seqLen)
int OverviewRenderArea::getAnnotationDensity (int pos) const {
    if(pos<1 || pos>annotationsOnPos.size()) {
        return 0;
    }
    return annotationsOnPos.at(pos-1);
}

void OverviewRenderArea::setAnnotationsOnPos() {
    annotationsOnPos.clear();
    ADVSequenceObjectContext* ctx = view->getSequenceContext();
    int len = ctx->getSequenceLen();
    annotationsOnPos.resize(len);
    const LRegion& sequenceRange = ctx->getSequenceObject()->getSequenceRange();
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach(AnnotationTableObject* at, ctx->getAnnotationObjects()) {
        foreach(Annotation* a, at->getAnnotations()) {
            AnnotationSettings* as = asr->getAnnotationSettings(a->getAnnotationName());
            if(as->visible) {
                foreach(const LRegion& r, a->getLocation()) {
                    LRegion innerRegion = r.intersect(sequenceRange);
                    for(int i = innerRegion.startPos;i < innerRegion.endPos(); i++) {
                        annotationsOnPos[i]++;
                    }
                }
            }
        }
    }
}

void OverviewRenderArea::drawAll(QPaintDevice *pd) {
    QPen pen(Qt::SolidLine);
    pen.setWidth(PEN_WIDTH);
    GSLV_UpdateFlags uf = view->getUpdateFlags();
    bool completeRedraw = uf.testFlag(GSLV_UF_NeedCompleteRedraw) || uf.testFlag(GSLV_UF_AnnotationsChanged) || uf.testFlag(GSLV_UF_ViewResized);
    if(completeRedraw) {
        QPainter pCached(cachedView);
        pCached.fillRect(0, 0, pd->width(), pd->height(), Qt::white);
        if(showGraph) {
            setAnnotationsOnPos();
            drawGraph(pCached);
        }
        pCached.end();
    }

    QPainter p(pd);
    p.drawPixmap(0,0,*cachedView);
    Overview* gv = static_cast<Overview*>(view);
    int panX = posToCoord(gv->getPan()->getVisibleRange().startPos);
    int panW = qMax(posToCoord(gv->getPan()->getVisibleRange().len), SLIDER_MIN_WIDTH);
    int detX = posToCoord(gv->getDet()->getVisibleRange().startPos) - int(ARROW_WIDTH/2);

    int panSliderHeight = pd->height()-PEN_WIDTH;
    int panSliderTop=0;
    if(showGraph) {
        panSliderHeight-=ANNOTATION_GRAPH_HEIGHT;
        panSliderTop+=ANNOTATION_GRAPH_HEIGHT;
    }
    panSlider.setRect(panX, panSliderTop, panW-PEN_WIDTH, panSliderHeight);
    detSlider.setRect(detX,ARROW_TOP_PAD, ARROW_WIDTH, ARROW_HEIGHT);

    pen.setColor(Qt::darkGray);
    p.setPen(pen);
    
    //don't show arrow when det view collapsed
    Overview* overview = qobject_cast<Overview*>(view);
    ADVSingleSequenceWidget* ssw = qobject_cast<ADVSingleSequenceWidget*>(overview->parentWidget());
    assert(ssw);
    if(!ssw->isPanViewCollapsed()) {
        drawSlider(p, panSlider, QColor(230, 230, 230));
    }

    if (!ssw->isDetViewCollapsed()) {
        drawArrow(p, detSlider, QColor(255, 187, 0));
    }
    
    drawRuler(p);
    drawSelection(p);
}

void OverviewRenderArea::drawSlider(QPainter& p, QRectF rect, QColor col) {
    QPainterPath path;
    path.addRect(rect);
    path.setFillRule(Qt::WindingFill);
    p.fillPath(path, col);
    p.fillPath(path, gradientMaskBrush);
    p.drawPath(path);
}

void OverviewRenderArea::drawArrow(QPainter& p, QRectF rect, QColor col) {
    QPainterPath arrPath;
    arrPath.moveTo(rect.bottomLeft());
    arrPath.lineTo(rect.center().x(), rect.center().y()-rect.width()/2);
    arrPath.lineTo(rect.bottomRight());
    arrPath.closeSubpath();
    arrPath.setFillRule(Qt::WindingFill);
    p.fillPath(arrPath, col);
    p.fillPath(arrPath, gradientMaskBrush);
    p.drawPath(arrPath);
}

void OverviewRenderArea::drawRuler(QPainter &p) {
    p.save();
    QPen pen(Qt::black);
    pen.setWidth(PEN_WIDTH);
    p.setPen(pen);
    Overview* gv = static_cast<Overview*>(view);
    int seqLen = gv->ctx->getSequenceLen();
    LRegion visibleRange = gv->getVisibleRange();

    float halfChar = getCurrentScale() / 2;
    int firstCharCenter = qRound(posToCoordF(visibleRange.startPos) + halfChar);
    int lastCharCenter = qRound(posToCoordF(visibleRange.endPos() - 1) + halfChar);
    int firstLastWidth = lastCharCenter - firstCharCenter;
    if (qRound(halfChar) == 0) {
        firstLastWidth--; // make the end of the ruler visible
    }
    GraphUtils::RulerConfig c;
    if(showGraph) {
        c.singleSideNotches=true;
    }
    c.notchSize = RULER_NOTCH_SIZE;
    GraphUtils::drawRuler(p, QPoint(firstCharCenter, ANNOTATION_GRAPH_HEIGHT), firstLastWidth, 1, seqLen, rulerFont, c);	
    p.restore();
}

#define SELECTION_LINE_WIDTH 3
void OverviewRenderArea::drawSelection(QPainter& p) {
    QPen pen(QColor("#007DE3"));
    pen.setWidth(SELECTION_LINE_WIDTH);
    p.setPen(pen);
    Overview* gv = qobject_cast<Overview*>(view);
    DNASequenceSelection* sel = gv->ctx->getSequenceSelection();
    foreach(const LRegion& r, sel->getSelectedRegions()) {
        int x1 = posToCoord(r.startPos);
        int x2 = posToCoord(r.endPos());
        p.drawLine(x1, ANNOTATION_GRAPH_HEIGHT, x2, ANNOTATION_GRAPH_HEIGHT);
    }
}

void OverviewRenderArea::drawGraph(QPainter& p) {
    p.save();
    QPen graphPen;
    graphPen.setWidth(1);
    p.fillRect(0,0,width()-PEN_WIDTH, ANNOTATION_GRAPH_HEIGHT-PEN_WIDTH, Qt::white);

    int halfChar = getCurrentScale()/2;
    for(int i = 0; i < width(); i++) {
        int count;
        int pos1 = coordToPos(i+halfChar);
        if(pos1<1 || pos1>annotationsOnPos.size()+1) {
            continue;
        }
        if(coordToPos(i+1)>pos1) {
            count=annotationsOnPos.at(pos1-1);
        }
        else {
            int pos2 = coordToPos(i) - 1;
            count = annotationsOnPos.at(pos1-1);
            for (int pos = pos1; pos < pos2; pos++) {
                int nextCount = annotationsOnPos.at(pos-1);
                count = qMax(count, nextCount);
            }
        }
        QColor col = getUnitColor(count);
        graphPen.setColor(col);
        p.setPen(graphPen);
        p.drawLine(i,0,i,ANNOTATION_GRAPH_HEIGHT);
    }
    p.restore();
}

QColor OverviewRenderArea::getUnitColor(int count) {

    switch(count) {
        case 0: return QColor(0xFF, 0xFF, 0xFF);
        case 1: return QColor(0xCC, 0xCC, 0xCC);
        case 2: 
        case 3:  return QColor(0x66, 0x66, 0x66);
        default: return QColor(0x00, 0x00, 0x00);
    }
}
}//namespace
