/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowRunTask.h"
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/WorkflowManager.h>
#include <U2Lang/Schema.h>
#include <U2Lang/HRSchemaSerializer.h>
#include <U2Lang/WorkflowUtils.h>

#include <U2Core/Counter.h>

#include <QtCore/QFile>
#include <QtCore/QFileInfo>

namespace U2 {

/*******************************************
 * WorkflowRunTask
 *******************************************/
WorkflowRunTask::WorkflowRunTask(const Schema& sh, QList<Iteration> lst) :
Task(tr("Execute workflow"), TaskFlags(TaskFlag_NoRun) | TaskFlag_ReportingIsSupported) {
    GCOUNTER( cvar, tvar, "WorkflowRunTask" );
    assert(!lst.isEmpty());
    foreach(const Iteration& it, lst) {
        WorkflowIterationRunTask* t = new WorkflowIterationRunTask(sh, it);
        connect(t, SIGNAL(si_ticked()), SIGNAL(si_ticked()));
        addSubTask(t);
    }
    setMaxParallelSubtasks(MAX_PARALLEL_SUBTASKS_AUTO);
}

QString WorkflowRunTask::generateReport() const {
    QString res;
    res+="<table width='75%'>";
    res+=QString("<tr><th>%1</th><th>%2</th><th>%3</th></tr>").arg(tr("Iterations")).arg(tr("Status")).arg(tr("Details"));
    foreach(Task* sub, getSubtasks()) {
        QString name = Qt::escape(sub->getTaskName());
        QString status = sub->hasErrors() ? tr("Failed") : sub->isCanceled() ? tr("Canceled") : tr("Finished");
        QString error = Qt::escape(sub->getError()).replace("\n", "<br>");
            //AppContext::getTaskScheduler()->getStateName(sub);
        if (sub->hasErrors()) {
            name = "<font color='red'>"+name+"</font>";
            status = "<font color='red'>"+status+"</font>";
        } else if (sub->isCanceled()) {
            status = "<font color='blue'>"+status+"</font>";
        } else {
            status = "<font color='green'>"+status+"</font>";
        }
        res+=QString("<tr><td>%1</td><td>%2</td><td>%3</td></tr>").arg(name).arg(status).arg(error);

        QStringList links = (static_cast<WorkflowIterationRunTask*>(sub))->getFiles();
        if(!links.isEmpty() && !sub->hasErrors()) {
            res += QString("<tr><td><i>%1</i></td></tr>").arg(tr("Output files:"));
        
            foreach(const QString &link, links) {
                res += QString("<tr><td><a href=%1>%1</a></td></tr>").arg(link);
            }
        }
        res+="<tr><td></td></tr>";
    }
    res+="</table>";
    return res;
}

QList<WorkerState> WorkflowRunTask::getState( Actor* actor) {
    QList<WorkerState> ret;
    foreach(Task* t, getSubtasks()) {
        WorkflowIterationRunTask* rt = qobject_cast<WorkflowIterationRunTask*>(t);
        ret << rt->getState(actor);
    }
    return ret;
}

int WorkflowRunTask::getMsgNum( Link* l) {
    int ret = 0;
    foreach(Task* t, getSubtasks()) {
        WorkflowIterationRunTask* rt = qobject_cast<WorkflowIterationRunTask*>(t);
        ret += rt->getMsgNum(l);
    }
    return ret;
}

int WorkflowRunTask::getMsgPassed(Link* l) {
    int ret = 0;
    foreach(Task* t, getSubtasks()) {
        ret += qobject_cast<WorkflowIterationRunTask*>(t)->getMsgPassed(l);
    }
    return ret;
}

Task::ReportResult WorkflowRunTask::report() {
    propagateSubtaskError();
    return ReportResult_Finished;
}


/*******************************************
* WorkflowIterationRunTask
*******************************************/
WorkflowIterationRunTask::WorkflowIterationRunTask(const Schema& sh, const Iteration& it) : 
Task(QString("%1").arg(it.name), TaskFlags_NR_FOSCOE), schema(new Schema()), scheduler(NULL) {
    rmap = HRSchemaSerializer::deepCopy(sh, schema);
    schema->applyConfiguration(it, rmap);
    
    if(schema->getDomain().isEmpty()) {
        QList<DomainFactory*> factories = WorkflowEnv::getDomainRegistry()->getAllEntries();
        assert(!factories.isEmpty());
        schema->setDomain(factories.isEmpty() ? "" : factories.at(0)->getId());
    }
    DomainFactory* df = WorkflowEnv::getDomainRegistry()->getById(schema->getDomain());
    if (!df) {
        stateInfo.setError(  tr("Unknown domain %1").arg(schema->getDomain()) );
        return;
    }
}

WorkflowIterationRunTask::~WorkflowIterationRunTask() {
    lmap.clear();
    DomainFactory* df = WorkflowEnv::getDomainRegistry()->getById(schema->getDomain());
    if (df) {
        df->destroy(scheduler, schema);
    }
    scheduler = NULL;
    delete schema;
}

void WorkflowIterationRunTask::prepare() {
    if( hasErrors() || isCanceled() ) {
        return;
    }
    
    DomainFactory* df = WorkflowEnv::getDomainRegistry()->getById(schema->getDomain());
    assert( df != NULL ); // checked in constructor
    foreach(Actor* a, schema->getProcesses()) {
        Worker* w = df->createWorker(a);
        if (!w) {
            stateInfo.setError( tr("Failed to create worker %1 %2 in domain %3")\
                .arg(a->getProto()->getId()).arg(a->getId()).arg(schema->getDomain()) );
            return;
        }
    }
    foreach(Link* l, schema->getFlows()) {
        CommunicationChannel* cc = df->createConnection(l);
        if (!cc) {
            stateInfo.setError(  tr("Failed to create connection %1 %2 in domain %3") ); //fixme
            return;
        }
        QStringList lst;
        lst << rmap.key(l->source()->owner()->getId());
        lst << (l->source()->getId());
        lst << rmap.key(l->destination()->owner()->getId());
        lst << (l->destination()->getId());
        QString key = lst.join("|");
        lmap.insert(key, cc);
    }
    scheduler = df->createScheduler(schema);
    if (!scheduler) {
        stateInfo.setError(  tr("Failed to create scheduler in domain %1").arg(df->getDisplayName()) );
        return;
    }
    scheduler->init();
    while(scheduler->isReady() && !isCanceled()) {
        Task* t = scheduler->tick();
        if (t) {
            addSubTask(t);
            break;
        }
    }
}

QList<Task*> WorkflowIterationRunTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> tasks;
    if (subTask->hasErrors()) {
        emit si_ticked();
        propagateSubtaskError();
        return tasks;
    }
    while(scheduler->isReady() && !isCanceled()) {
        Task* t = scheduler->tick();
        if (t) {
            tasks << t;
            break;
        }
    }
    emit si_ticked();
    return tasks;
}

Task::ReportResult WorkflowIterationRunTask::report() {
    if (scheduler) {
        scheduler->cleanup();
        if (!scheduler->isDone()) {
            if(!hasErrors()) {
                setError(tr("No workers are ready, while not all workers are done. Schema is broken?"));
            }
        }
    }

    foreach(Actor *a, schema->getProcesses()) {
        foreach(Attribute *attr, a->getProto()->getAttributes()) {
            if(attr->getId() == CoreLibConstants::URL_OUT_ATTR().getId()) {
                QString str = a->getParameter(CoreLibConstants::URL_OUT_ATTR().getId())->getAttributeValue<QString>();
                QUrl url(str);
                if(url.isValid()) {
                    fileLinks << url.toString();
                }
            }
        }
    }

    return ReportResult_Finished;
}

WorkerState WorkflowIterationRunTask::getState(Actor* a)
{
    if (scheduler) {
        return scheduler->getWorkerState(rmap.value(a->getId()));
    }
    return WorkerWaiting;
}

static QString getKey(Link * l) {
    QStringList lst;
    lst << (l->source()->owner()->getId());
    lst << (l->source()->getId());
    lst << (l->destination()->owner()->getId());
    lst << (l->destination()->getId());
    return lst.join("|");
}

int WorkflowIterationRunTask::getMsgNum( Link* l)
{
    CommunicationChannel* cc = lmap.value(getKey(l));
    if (cc) {
        return cc->hasMessage();
    }
    return 0;
}

int WorkflowIterationRunTask::getMsgPassed(Link* l) {
    CommunicationChannel * cc = lmap.value(getKey(l));
    if(cc != NULL) {
        return cc->takenMessages();
    }
    return 0;
}

QStringList WorkflowIterationRunTask::getFiles() const {
    return fileLinks;
}

}//namespace
