/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CoreDataTypes.h"

#include <U2Lang/WorkflowEnv.h>

namespace U2 {

/****************************************
 * CoreDataTypes
 ****************************************/
const QString CoreDataTypes::STRING_TYPE_ID("string");
const QString CoreDataTypes::BOOL_TYPE_ID("bool");
const QString CoreDataTypes::NUM_TYPE_ID("number");
const QString CoreDataTypes::ANY_TYPE_ID("void");

using namespace Workflow;
DataTypePtr CoreDataTypes::STRING_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(STRING_TYPE_ID, tr("String"), tr("A string of characters"))));
        startup = false;
    }
    return dtr->getById(STRING_TYPE_ID);
}

DataTypePtr CoreDataTypes::BOOL_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(BOOL_TYPE_ID, tr("Boolean"), tr("A boolean value (true/false)"))));
        startup = false;
    }
    return dtr->getById(BOOL_TYPE_ID);
}

DataTypePtr CoreDataTypes::NUM_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(NUM_TYPE_ID, tr("Number"), tr("A number"))));
        startup = false;
    }
    return dtr->getById(NUM_TYPE_ID);
}

DataTypePtr CoreDataTypes::ANY_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(ANY_TYPE_ID, tr("Undefined"), tr("A void type"))));
        startup = false;
    }
    return dtr->getById(ANY_TYPE_ID);
}

static void setIfNotNull( bool * to, bool val ) {
    if( to != NULL ) {
        *to = val;
    }
}

/****************************************
* StringTypeValueFactory
****************************************/
QVariant StringTypeValueFactory::getValueFromString( const QString & str, bool * ok ) const {
    setIfNotNull( ok, true );
    return qVariantFromValue( str );
}

/****************************************
* BoolTypeValueFactory
****************************************/
const QString BoolTypeValueFactory::TRUE_STR    = "true";
const QString BoolTypeValueFactory::FALSE_STR   = "false";
const QString BoolTypeValueFactory::YES_STR     = "yes";
const QString BoolTypeValueFactory::NO_STR      = "no";
const QString BoolTypeValueFactory::ONE_STR     = "1";
const QString BoolTypeValueFactory::NIL_STR     = "0";

QVariant BoolTypeValueFactory::getValueFromString( const QString & s, bool * ok ) const {
    QString str = s.toLower();
    if( str == TRUE_STR || str == YES_STR || str == ONE_STR ) {
        setIfNotNull( ok, true );
        return qVariantFromValue( true );
    }
    if( str == FALSE_STR || str == NO_STR || str == NIL_STR ) {
        setIfNotNull( ok, true );
        return qVariantFromValue( false );
    }
    setIfNotNull( ok, false );
    return QVariant();
}

/****************************************
* NumTypeValueFactory
****************************************/
QVariant NumTypeValueFactory::getValueFromString( const QString & str, bool * okArg ) const {
    bool ok = false;
    int intCandidate = str.toInt(&ok);
    if(ok) {
        setIfNotNull(okArg, true);
        return qVariantFromValue(intCandidate);
    }

    double doubleCandidate = str.toDouble(&ok);
    if(ok) {
        setIfNotNull(okArg, true);
        return qVariantFromValue(doubleCandidate);
    }

    setIfNotNull(okArg, false);
    return QVariant();
}

} //namespace
