/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "LRegion.h"

namespace U2 {


QList<LRegion> LRegion::join(const QList<LRegion>& regions)  {
    QList<LRegion> result = regions;
    qStableSort(result.begin(), result.end()); //sort by region start pos first
    for (int i = 0; i < result.length()-1;) {
        const LRegion& ri0 = result[i];
        const LRegion& ri1 = result[i+1];
        if (!ri0.intersects(ri1)) {
            i++;
            continue;
        }
        LRegion newRi = LRegion::join(ri0, ri1);
        result[i] = newRi;
        result.removeAt(i+1);
    }
    return result;
}

/// checks if r intersects with any of regions. Returns index of region it intersected if yes
int LRegion::intersectsWithAny(const LRegion& r, const QList<LRegion>& regions) {
    for (int i = 0, n = regions.size(); i < n; i++) {
        const LRegion& ri = regions.at(i);
        if (r.intersects(ri)) {
            return i;
        }
    }
    return -1;
}

int LRegion::containsInAny(int pos, const QList<LRegion>& regions) {
    for (int i = 0, n = regions.size(); i < n; i++) {
        const LRegion& ri = regions.at(i);
        if (ri.contains(pos)) {
            return i;
        }
    }
    return -1;
}

bool LRegion::belongsToAny( const QList<LRegion>& rs ) const
{
    bool belongs = false;
    foreach (const LRegion& r, rs) {
        if (r.contains(*this)) {
            belongs = true;
            break;
        }
    }
    
    return belongs;
}

void LRegion::shift(int offset, QList<LRegion>& regions) {
    QList<LRegion> res;
    for (int i = 0, n = regions.size(); i < n; i++) {
        LRegion& r = regions[i];
        r.startPos += offset;
    }
}

void LRegion::bound(int minPos, int maxPos, QList<LRegion>& regions) {
    for (int i = 0, n = regions.size(); i < n; i++) {
        LRegion& r = regions[i];
        int start = qBound(minPos, r.startPos, maxPos);
        int end = qBound(minPos, r.endPos(), maxPos);
        r.startPos = start;
        r.len = end - start;
    }
}

void LRegion::mirror(int mirrorPos, QList<LRegion>& regions) {
    for (int i = 0, n = regions.size(); i < n; i++) {
        LRegion& r = regions[i];
        assert(r.endPos() < mirrorPos);
        r.startPos = mirrorPos - r.endPos();
    }
}

void LRegion::divide(int div, QList<LRegion>& regions) {
    for (int i = 0, n = regions.size(); i < n; i++) {
        LRegion& r = regions[i];
        r.startPos = r.startPos / div;
    }
}

void LRegion::multiply(int mult, QList<LRegion>& regions) {
    for (int i = 0, n = regions.size(); i < n; i++) {
        LRegion& r = regions[i];
        r.startPos = r.startPos * mult;
    }
}

void LRegion::reverse(QList<LRegion>& regions) {
    QList<LRegion> old = regions;
    regions.clear();
    foreach(const LRegion& r, old) {
        regions.prepend(r);
    }
}


static bool _registerMeta() {
    qRegisterMetaType<LRegion>("LRegion");
    qRegisterMetaTypeStreamOperators<LRegion>("U2::LRegion");

    qRegisterMetaType<QList<LRegion> >("QList<U2::LRegion>");
    qRegisterMetaTypeStreamOperators< QList<LRegion> >("QList<U2::LRegion>");
    return true;
}

bool LRegion::registerMeta = _registerMeta();

QDataStream &operator<<(QDataStream &out, const LRegion &myObj) {
    out << myObj.startPos << myObj.len;
    return out;
}

QDataStream &operator>>(QDataStream &in, LRegion &myObj) {
    in >> myObj.startPos;
    in >> myObj.len;
    return in;
}

}//namespace
