/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _U2_FINDALGORITHM_H_
#define _U2_FINDALGORITHM_H_

#include <U2Core/LRegion.h>
#include <U2Core/AnnotationData.h>

#include <QtCore/QList>

namespace U2 {


class U2ALGORITHM_EXPORT FindAlgorithmResult {
public:
    FindAlgorithmResult() : region(0, 0), translation(false), complement(false), err(0){}
    FindAlgorithmResult(const LRegion& _r, bool t, bool c, int _err) 
        : region(_r), translation(t), complement(c), err(_err){}
    
    void clear() {region.startPos = 0; region.len = 0; translation = false; complement = false; err = 0;}
    
    bool isEmpty() const {return region.startPos == 0 && region.len == 0;}

    bool operator ==(const FindAlgorithmResult& o) const {
        return region == o.region && err == o.err && complement == o.complement && translation == o.translation;
    }

    SharedAnnotationData toAnnotation(const QString& name) const {
        SharedAnnotationData data;
        data = new AnnotationData;
        data->name = name;
        data->location.append(region);
        data->complement = complement;
        data->aminoFrame = translation ? TriState_Yes : TriState_No;
        data->qualifiers.append(Qualifier("error", QString::number(err)));
        return data;
    }

    LRegion region;
    bool    translation;
    bool    complement;
    int     err;

    static QList<SharedAnnotationData> toTable(const QList<FindAlgorithmResult>& res, const QString& name)
    {
        QList<SharedAnnotationData> list;
        foreach (const FindAlgorithmResult& f, res) {
            list.append(f.toAnnotation(name));
        }
        return list;
    }

};

class DNATranslation;

class U2ALGORITHM_EXPORT FindAlgorithmResultsListener {
public:
    virtual ~FindAlgorithmResultsListener(){}
    virtual void onResult(const FindAlgorithmResult& r) = 0;
};

enum FindAlgorithmStrand {
    FindAlgorithmStrand_Both,
    FindAlgorithmStrand_Direct,
    FindAlgorithmStrand_Complement
};

class U2ALGORITHM_EXPORT FindAlgorithmSettings {
public:
    FindAlgorithmSettings(const QByteArray& pattern = QByteArray(),
        FindAlgorithmStrand strand = FindAlgorithmStrand_Direct,
        DNATranslation* complementTT = NULL,
        DNATranslation* proteinTT = NULL,
        const LRegion& searchRegion = LRegion(),
        bool singleShot = false,
        int maxErr = 0,
        bool insDelAlg = false
) : pattern(pattern), strand(strand), complementTT(complementTT), proteinTT(proteinTT),
searchRegion(searchRegion), singleShot(singleShot), maxErr(maxErr), insDelAlg(insDelAlg){}

    QByteArray          pattern;
    FindAlgorithmStrand strand;
    DNATranslation*     complementTT;
    DNATranslation*     proteinTT;
    LRegion             searchRegion;
    bool                singleShot;
    int                 maxErr;
    bool                insDelAlg;
};


class U2ALGORITHM_EXPORT FindAlgorithm {
public:
    // Note: pattern is never affected by either aminoTT or complTT
    
    static void find(
        FindAlgorithmResultsListener* rl, 
        DNATranslation* aminoTT, // if aminoTT!=NULL -> pattern must contain amino data and sequence must contain DNA data
        DNATranslation* complTT, // if complTT!=NULL -> sequence is complemented before comparison with pattern
        FindAlgorithmStrand strand, // if not direct there complTT must not be NULL
        bool insDel,
        const char* sequence, 
        int seqLen, 
        const LRegion& range,  
        const char* pattern, 
        int patternLen, 
        bool singleShot, 
        int maxErr, 
        int& stopFlag, 
        int& percentsCompleted, 
        int& currentPos); 

    static void find(
        FindAlgorithmResultsListener* rl,
        const FindAlgorithmSettings& config,
        const char* sequence, 
        int seqLen, 
        int& stopFlag, 
        int& percentsCompleted, 
        int& currentPos) {
            find(rl,
                config.proteinTT,
                config.complementTT,
                config.strand,
                config.insDelAlg,
                sequence,
                seqLen,
                config.searchRegion,
                config.pattern.constData(),
                config.pattern.length(),
                config.singleShot,
                config.maxErr,
                stopFlag, 
                percentsCompleted, 
                currentPos);
    }

};

}//namespace

#endif
